/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <sys/time.h>

#include <xine/sorted_array.h>

#include "_xitk.h"

#include "image.h"
#include "utils.h"
#include "font.h"
#include "_backend.h"

static int _xitk_pix_font_find_char (xitk_pix_font_t *pf, xitk_point_t *found, int this_char) {
  int range, n = 0;

  for (range = 0; pf->unicode_ranges[range].first > 0; range++) {
    if ((this_char >= pf->unicode_ranges[range].first) && (this_char <= pf->unicode_ranges[range].last))
      break;
    n += pf->unicode_ranges[range].last - pf->unicode_ranges[range].first + 1;
  }

  if (pf->unicode_ranges[range].first <= 0) {
    *found = pf->unknown;
    return 0;
  }

  n += this_char - pf->unicode_ranges[range].first;
  found->x = (n % pf->chars_per_row) * pf->char_width;
  found->y = (n / pf->chars_per_row) * pf->char_height;
  return 1;
}

void xitk_image_set_pix_font (xitk_image_t *image, const char *format) {
  xitk_pix_font_t *pf;
  int range, total;

  if (!image || !format)
    return;
  if (image->pix_font)
    return;
  image->pix_font = pf = malloc (sizeof (*pf));
  if (!pf)
    return;

  pf->width = image->width;
  pf->height = image->height;

  range = 0;
  total = 0;
  do {
    const uint8_t *p = (const uint8_t *)format;
    int v;
    uint8_t z;
    if (!p)
      break;
    while (1) {
      while (*p && (*p != '('))
        p++;
      if (!(*p))
        break;
      p++;
      if ((*p ^ '0') < 10)
        break;
    }
    if (!(*p))
      break;

    v = 0;
    while ((z = (*p ^ '0')) < 10)
      v = v * 10u + z, p++;
    if (v <= 0)
      break;
    pf->chars_per_row = v;
    if (*p != '!')
      break;
    p++;

    while (1) {
      pf->unicode_ranges[range].last = 0;
      v = 0;
      while ((z = (*p ^ '0')) < 10)
        v = v * 10u + z, p++;
      pf->unicode_ranges[range].first = v;
      if (*p != '-')
        break;
      p++;
      v = 0;
      while ((z = (*p ^ '0')) < 10)
        v = v * 10u + z, p++;
      pf->unicode_ranges[range].last = v;
      if (pf->unicode_ranges[range].last < pf->unicode_ranges[range].first)
        pf->unicode_ranges[range].last = pf->unicode_ranges[range].first;
      total += pf->unicode_ranges[range].last - pf->unicode_ranges[range].first + 1;
      range++;
      if (range >= XITK_MAX_UNICODE_RANGES)
        break;
      if (*p != '!')
        break;
      p++;
    }
  } while (0);

  if (range == 0) {
    pf->chars_per_row = 32;
    pf->unicode_ranges[0].first = 32;
    pf->unicode_ranges[0].last = 127;
    total = 127 - 32 + 1;
    range = 1;
  }

  pf->char_width = pf->width / pf->chars_per_row;
  pf->chars_total = total;
  total = (total + pf->chars_per_row - 1) / pf->chars_per_row;
  pf->char_height = pf->height / total;
  pf->unicode_ranges[range].first = 0;
  pf->unicode_ranges[range].last = 0;

  pf->unknown.x = 0;
  pf->unknown.y = 0;
  _xitk_pix_font_find_char (pf, &pf->unknown, 127);
  _xitk_pix_font_find_char (pf, &pf->space, ' ');
  _xitk_pix_font_find_char (pf, &pf->asterisk, '*');
}

static void _xitk_image_destroy_pix_font (xitk_pix_font_t **pix_font) {
  free (*pix_font);
  *pix_font = NULL;
}

/*
 *
 */
void xitk_image_ref (xitk_image_t *img) {
  if (!img)
    return;
  img->refs += 1;
  if (img->refs > img->max_refs)
    img->max_refs = img->refs;
}

int xitk_image_free_image (xitk_image_t **src) {
  xitk_image_t *image;

  if (!src)
    return 0;
  image = *src;
  if (!image)
    return 0;

  image->refs -= 1;
  if (image->refs > 0)
    return image->refs;

#ifdef XINE_SARRAY_MODE_UNIQUE
  if (image->wl && image->wl->shared_images) {
    xine_sarray_remove_ptr (image->wl->shared_images, image);
    if (image->wl->xitk->verbosity >= 2)
      printf ("xitk.image.free (%s, %d x %d, %d refs).\n",
        image->key, image->width, image->height, image->max_refs);
  }
#endif
  if (image->beimg) {
    image->beimg->_delete (&image->beimg);
  }
  _xitk_image_destroy_pix_font (&image->pix_font);

  XITK_FREE (image);
  *src = NULL;
  return 0;
}

void xitk_image_copy (xitk_image_t *from, xitk_image_t *to) {
  if (!from || !to)
    return;
  if (!from->beimg || !to->beimg)
    return;
  to->beimg->copy_rect (to->beimg, from->beimg, 0, 0, from->width, from->height, 0, 0);
}

void xitk_image_copy_rect (xitk_image_t *from, xitk_image_t *to, int x1, int y1, int w, int h, int x2, int y2) {
  if (!from || !to)
    return;
  if (!from->beimg || !to->beimg)
    return;
  to->beimg->copy_rect (to->beimg, from->beimg, x1, y1, w, h, x2, y2);
}

static void _xitk_image_add_beimg (xitk_image_t *img, const char *data, int dsize) {
  xitk_tagitem_t tags[7] = {
    {XITK_TAG_FILENAME, (uintptr_t)NULL},
    {XITK_TAG_FILEBUF, (uintptr_t)NULL},
    {XITK_TAG_FILESIZE, 0},
    {XITK_TAG_RAW, (uintptr_t)NULL},
    {XITK_TAG_WIDTH, img->width},
    {XITK_TAG_HEIGHT, img->height},
    {XITK_TAG_END, 0}
  };
  if (data) {
    if (dsize > 0) {
      tags[1].value = (uintptr_t)data;
      tags[2].value = dsize;
    } else if (dsize == 0) {
      tags[0].value = (uintptr_t)data;
    } else if (dsize == -1) {
      tags[3].value = (uintptr_t)data;
    }
  }
  img->beimg = img->xitk->d->image_new (img->xitk->d, tags);
  if (!img->beimg)
    return;
  if (data) {
    img->beimg->get_props (img->beimg, tags + 4);
    img->width = tags[4].value;
    img->height = tags[5].value;
  }
}

/* if (data != NULL):
 * dsize > 0: decode (down)loaded image file contents,
 *            and aspect preserving scale to fit w and/or h if set.
 * dsize == 0: read from named file, then same as above.
 * dsize == -1: use raw data as (w, h). */
xitk_image_t *xitk_image_new (xitk_t *xitk, const char *data, int dsize, int w, int h) {
  xitk_image_t *img;
  struct timeval t1 = {0, 0}, t2 = {0, 0};

  if (!xitk)
    return NULL;
  img = (xitk_image_t *)xitk_xmalloc (sizeof (*img));
  if (!img)
    return NULL;
  img->xitk = xitk;
  img->width = w;
  img->height = h;
  img->last_state = XITK_IMG_STATE_NORMAL;
  if ((dsize == 0) && data && (xitk->verbosity >= 2))
    xitk_gettime_tv (&t1);
  _xitk_image_add_beimg (img, data, dsize);
  if (!img->beimg) {
    if ((dsize == 0) && data && (xitk->verbosity >= 1))
      printf ("xitk.image.load (%s) [failed].\n", (const char *)data);
    XITK_FREE (img);
    return NULL;
  }
  if ((dsize == 0) && data && (xitk->verbosity >= 2)) {
    int d;
    xitk_gettime_tv (&t2);
    d = ((int)t2.tv_usec - (int)t1.tv_usec) / 100;
    d += (t2.tv_sec - t1.tv_sec) * 10000;
    printf ("xitk.image.load (%s) [%0d.%04ds].\n", (const char *)data, d / 10000, d % 10000);
  }
  img->wl = NULL;
  img->max_refs =
  img->refs     = 1;
  img->key[0] = 0;
  return img;
}

int xitk_image_inside (xitk_image_t *img, int x, int y) {
  if (!img)
    return 0;
  if (!_ZERO_TO_MAX_MINUS_1 (x, img->width) || !_ZERO_TO_MAX_MINUS_1 (y, img->height))
    return 0;
  if (!img->beimg)
    return 1;
  return img->beimg->pixel_is_visible (img->beimg, x, y);
}

#ifdef XINE_SARRAY_MODE_UNIQUE
static int _xitk_shared_image_cmp (void *a, void *b) {
  xitk_image_t *d = (xitk_image_t *)a;
  xitk_image_t *e = (xitk_image_t *)b;
  int f;

  if ((f = d->width - e->width))
    return f;
  if ((f = d->height - e->height))
    return f;
  return strcmp (d->key, e->key);
}
#endif

int xitk_shared_image (xitk_widget_list_t *wl, const char *key, int width, int height, xitk_image_t **image) {
  xitk_image_t *i;
  size_t keylen;

  ABORT_IF_NOT_COND (width > 0);
  ABORT_IF_NOT_COND (height > 0);

  if (!image)
    return 0;
  if (!wl || !key) {
    *image = NULL;
    return 0;
  }

  keylen = xitk_find_byte (key, 0);
  if (keylen > sizeof (i->key) - 1)
    keylen = sizeof (i->key) - 1;
  i = (xitk_image_t *)xitk_xmalloc (sizeof (*i) + keylen);
  if (!i) {
    *image = NULL;
    return 0;
  }
  i->xitk  = wl->xitk;
  i->width = width;
  i->height = height;
  i->last_state = XITK_IMG_STATE_NORMAL;
  memcpy (i->key, key, keylen);
  i->key[keylen] = 0;

#ifdef XINE_SARRAY_MODE_UNIQUE
  if (!wl->shared_images) {
    wl->shared_images = xine_sarray_new (16, _xitk_shared_image_cmp);
    xine_sarray_set_mode (wl->shared_images, XINE_SARRAY_MODE_UNIQUE);
  }
  if (wl->shared_images) {
    int ai;

    ai = xine_sarray_add (wl->shared_images, i);
    if (ai < 0) {
      XITK_FREE (i);
      i = xine_sarray_get (wl->shared_images, ~ai);
      i->refs += 1;
      if (i->refs > i->max_refs)
        i->max_refs = i->refs;
      *image = i;
      return i->refs;
    }
  }
#endif
  _xitk_image_add_beimg (i, NULL, 0);
  if (!i->beimg) {
    XITK_FREE (i);
    return 0;
  }
  i->max_refs = i->refs = 1;
  i->wl = wl;
  *image = i;
  return i->refs;
}

void xitk_shared_image_list_delete (xitk_widget_list_t *wl) {
  if (wl && wl->shared_images) {
    int i, max = xine_sarray_size (wl->shared_images);

    for (i = 0; i < max; i++) {
      xitk_image_t *image = xine_sarray_get (wl->shared_images, i);

      image->wl = NULL;
    }

    xine_sarray_delete(wl->shared_images);
    wl->shared_images = NULL;
  }
}

static void _xitk_image_fill_rectangle (xitk_image_t *img, int x, int y, int w, int h, unsigned int color) {
  xitk_be_rect_t xr = {.x = x, .y = y, .w = w, .h = h};

  img->beimg->display->lock (img->beimg->display);
  img->beimg->fill_rects (img->beimg, &xr, 1, color, 0);
  img->beimg->display->unlock (img->beimg->display);
}

void xitk_image_fill_rectangle (xitk_image_t *img, int x, int y, int w, int h, unsigned int color) {
  if (img && img->beimg) {
    xitk_be_rect_t xr = {.x = x, .y = y, .w = w, .h = h};

    img->beimg->display->lock (img->beimg->display);
    /* set mask if it exitsts (mask color == 1) */
    img->beimg->fill_rects (img->beimg, &xr, 1, 1, 1);
    img->beimg->fill_rects (img->beimg, &xr, 1, color, 0);
    img->beimg->display->unlock (img->beimg->display);
  }
}

static void _xitk_image_draw_string (xitk_image_t *img, xitk_font_t *xtfs,
  int x, int y, const char *text, size_t nbytes, int color) {
  img->beimg->display->lock (img->beimg->display);
  img->beimg->draw_text (img->beimg, xtfs, text, nbytes, x, y, color);
  img->beimg->display->unlock (img->beimg->display);
}

void xitk_image_draw_string (xitk_image_t *img, xitk_font_t *xtfs,
  int x, int y, const char *text, size_t nbytes, int color) {
  ABORT_IF_NULL(xtfs);
  ABORT_IF_NULL(text);

  if (!img)
    return;
  if (!img->beimg || !img->xitk)
    return;

#ifdef DEBUG
  if (nbytes > strlen(text) + 1) {
    XITK_WARNING("draw_string: %zu > %zu\n", nbytes, strlen(text));
  }
#endif
  _xitk_image_draw_string (img, xtfs, x, y, text, nbytes, color);
}

/*
 *
 */
xitk_image_t *xitk_image_create_image_with_colors_from_string (xitk_t *xitk,
  const char *fontname, int width, int pad_x, int pad_y, int align, const char *str,
  unsigned int foreground, unsigned int background) {
#define _p    1 /* printable, keep this == 1 */
#define _r    2 /* remove */
#define _t    4 /* tab replace */
#define _n    8 /* new line, split paragraph */
#define _s   16 /* space, wrap with remove */
#define _e   32 /* end */
#define _m   64 /* multibyte sequence */
#define _w (128 | _p) /* / ? &, wrap without remove */
  static const uint8_t tab_type[256] = {
    _e,_r,_r,_r,_r,_r,_r,_r,_r,_t,_n,_r,_r,_r,_r,_r,
    _r,_r,_r,_r,_r,_r,_r,_r,_r,_r,_r,_r,_r,_r,_r,_r,
    _s,_p,_p,_p,_p,_p,_w,_p,_p,_p,_p,_p,_p,_p,_p,_w,
    _p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_w,
    _p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,
    _p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,
    _p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,
    _p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_r,
    _m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,
    _m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,
    _m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,
    _m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,_m,
    _p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,
    _p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,
    _p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,
    _p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p,_p
  };
  struct {
    int lb;    /** << left bearing. */
    int rb;    /** << right bearing. */
    int start; /** << offset into str (stop >= 0) or buf (stop < 0). */
    int stop;  /** << offset into str or buf. if 0, insert lb / 2 empty lines. */
  } lines[80];
  xitk_image_t  *image;
  xitk_font_t   *fs;
  int            wmax = 0, amax = 0, dmax = 0, line_height, total_height;
  uint32_t       nline = 0;
  const uint8_t *p;
  uint8_t        buf[BUFSIZ], *bq = buf, *bend = buf + sizeof (buf) - 8;

  ABORT_IF_NULL (xitk);
  ABORT_IF_NULL (fontname);
  ABORT_IF_NULL (str);

  /* due to sub pixel blending aka anti aliasing inside font drawing,
   * we may need 1 more pixel around. */
  if (pad_x < 1)
    pad_x = 1;
  if (pad_y < 1)
    pad_y = 1;
  /* never return more width than user requested. */
  width -= 2 * pad_x;
  ABORT_IF_NOT_COND (width > 0);

  fs = xitk_font_load_font (xitk, fontname);
  if (!str[0])
    str = "   ";
#if 0
  str = "\tThis is an \tintentionally \bhuge and at the \asame time totally useless text for testing "
        "the wrap and filter features, although we most likely will not find all hidden bugs.";
#endif

  if (align == ALIGN_DEFAULT)
    align = ALIGN_LEFT;

  p = (const uint8_t *)str;
  while (nline < sizeof (lines) / sizeof (lines[0])) {
    uint32_t len, nchars = 0, flags = 0, bmode = 0;
    int lbearing, rbearing, lw, ascent, descent;
    const uint8_t *start_par = p, *end_par;
    /* scan paragraph #1: keep origin. */
    while (1) {
      uint32_t fl2 = tab_type[p[0]];
      flags |= fl2;
      if (flags & (_r | _t | _n | _e))
        break;
      nchars += (~fl2) & _p;
      p++;
    }
    len = p - start_par;
    lines[nline].start = start_par - (const uint8_t *)str;
    end_par = p;
    if (flags & (_r | _t)) {
      /* need filtering, switch to temp buffer. */
      if (len > (uint32_t)(bend - bq))
        len = bend - bq;
      if (len)
        memcpy (bq, start_par, len);
      start_par = bq;
      bmode = 1;
      /* filter rest of paragraph. */
      while (bq < bend) {
        uint32_t fl2 = tab_type[p[0]];
        flags |= fl2;
        if (fl2 & (_p | _s)) {
          /* space, printable. */
          nchars++;
          *bq++ = *p++;
        } else if (fl2 & _m) {
          /* multibyte sequence. */
          *bq++ = *p++;
        } else if (fl2 & _r) {
          /* remove. */
          p++;
        } else if (fl2 & _t) {
          /* tab. */
          p++;
          memcpy (bq, "        ", 8);
          len = 8 - (nchars & 7);
          if (len > (uint32_t)(bend - bq))
            len = bend - bq;
          nchars += len;
          bq += len;
        } else {
          break;
        }
      }
      len = bq - start_par;
      lines[nline].start = buf - start_par;
      end_par = bq;
    }
    if (!(flags & (_p | _m))) {
      /* empty line. */
      if (nline && !lines[nline - 1].stop) {
        lines[--nline].lb += 1;
      } else {
        lines[nline].lb = 1;
        lines[nline].stop = 0;
        nline++;
      }
    } else {
      /* try whole paragraph. */
      xitk_font_text_extent (fs, (const char *)start_par, len, &lbearing, &rbearing, NULL, &ascent, &descent);
      if (ascent > amax)
        amax = ascent;
      if (descent > dmax)
        dmax = descent;
      lw = rbearing - lbearing;
      if (lw <= width) {
        /* fits 1 target line, use as is. */
        if (lw > wmax)
          wmax = lw;
        lines[nline].lb = lbearing;
        lines[nline].rb = rbearing;
        lines[nline].stop = bmode ? buf - bq : p - (const uint8_t *)str;
        nline++;
      } else {
        int indent1 = 0, indent2 = 0;
        const uint8_t *tp;

        *bq = 0;
        if (align == ALIGN_LEFT) {
          /* wrap #1: use estimated initial indent plus text height for lines 2ff. */
          for (tp = start_par; tp[0] == ' '; tp++) ;
          indent1 = nchars ? (lw * (tp - start_par) + (nchars >> 1)) / nchars : 0;
          indent1 += amax + dmax;
        }
        /* wrap #2: split lines. */
        while ((tp = start_par) < end_par) {
          int last_diff = 0x7fffffff;
          uint32_t try, last_len = len;
          const uint8_t *next, *stop;

          if (indent2 + lw <= width) {
            /* last line in this paragraph. */
            stop = next = end_par;
          } else {
            /* phonebook search. */
            for (try = 4; try && lw; try--) {
              int diff;
              /* estimate byte pos. */
              len = (width - indent2) * len / lw;
              if (!len)
                break;
              /* get width. */
              xitk_font_text_extent (fs, (const char *)start_par, len, &lbearing, &rbearing, NULL, NULL, NULL);
              lw = rbearing - lbearing;
              diff = lw - (width - indent2);
              if (diff < 0)
                diff = -diff;
              if (diff < last_diff) {
                last_diff = diff;
                last_len = len;
              }
            }
            /* find end of previous word, usually just 1 pass. */
            do {
              /* start next line at previous space or wrap char. */
              for (next = start_par + last_len; (next > start_par) && !(tab_type[next[-1]] & (_s | _w) & (~_p)); next--) ;
              /* cut spaces from the end of this line. */
              for (stop = next; (stop > start_par) && (stop[-1] == ' '); stop--) ;
              if (stop <= start_par) {
                /* first word too long, wrap insidr that. */
                next = start_par + last_len;
                if ((next > start_par) && !(tab_type[next[-1]] & _m))
                  next--;
                for (; (next > start_par) && (tab_type[next[-1]] & _m); next--);
                stop = next;
              }
              xitk_font_text_extent (fs, (const char *)start_par, stop - start_par, &lbearing, &rbearing, NULL, NULL, NULL);
              last_len = stop - start_par;
              lw = rbearing - lbearing;
            } while (indent2 + lw > width);
          }
          /* add line. */
          if (indent2 + lw > wmax)
            wmax = indent2 + lw;
          lines[nline].lb = lbearing - indent2;
          lines[nline].rb = rbearing;
          lines[nline].start = bmode ? start_par - buf : start_par - (const uint8_t *)str;
          lines[nline].stop = bmode ? buf - stop : stop - (const uint8_t *)str;
          nline++;
          if (next >= end_par)
            break;
          if (nline >= sizeof (lines) / sizeof (lines[0]))
            break;
          /* prepare next line. */
          start_par = next;
          len = end_par - start_par;
          xitk_font_text_extent (fs, (const char *)start_par, len, &lbearing, &rbearing, NULL, NULL, NULL);
          lw = rbearing - lbearing;
          /* turn on indent now. */
          indent2 = indent1;
        }
      }
    }
    if (!p[0])
      break;
    p++;
  }

  /* get total text height. */
  if (nline && !lines[nline - 1].stop)
    nline--;
  line_height = amax + dmax + 2;
  {
    uint32_t u;
    for (total_height = 0, u = 0; u < nline; u++)
      total_height += lines[u].stop ? 2 : lines[u].lb;
    total_height *= line_height;
    total_height >>= 1;
    total_height -= 2;
  }

  /* If default resp. left aligned, we may shrink the image */
  if (align == ALIGN_LEFT)
    width = xitk_min (width, wmax);

  /* get the image. */
  image = xitk_image_new (xitk, NULL, 0, width + 2 * pad_x, total_height + 2 * pad_y);
  if (!image) {
    xitk_font_unload_font (fs);
    return NULL;
  }
  _xitk_image_fill_rectangle (image, 0, 0, image->width, image->height, background);

  /* draw the strings. */
  {
    uint32_t u;
    int y; /* half pixels */

    for (y = 2 * (amax + pad_y), u = 0; u < nline; u++) {
      if (lines[u].stop) {
        const char *text;
        int tlen, x, w;
        /* an actual line. */
        if (lines[u].stop > 0) {
          text = str + lines[u].start;
          tlen = lines[u].stop - lines[u].start;
        } else {
          text = (const char *)buf + lines[u].start;
          tlen = -lines[u].stop - lines[u].start;
        }
        w = lines[u].rb - lines[u].lb;
        x = (align == ALIGN_CENTER) ? (width - w) >> 1
          : (align == ALIGN_RIGHT) ? width - w
          : 0;
        x += pad_x;
        _xitk_image_draw_string (image, fs, x - lines[u].lb, y >> 1, text, tlen, foreground);
        y += 2 * line_height;
      } else {
        /* an empty line. */
        y += lines[u].lb * line_height;
      }
    }
  }

  xitk_font_unload_font (fs);
  return image;
}

xitk_image_t *xitk_image_from_string (xitk_t *xitk, const char *fontname,
    int width, int align, const char *str) {

  return xitk_image_create_image_with_colors_from_string (xitk, fontname, width, 0, 0, align, str,
    xitk_get_cfg_num (xitk, XITK_BLACK_COLOR), xitk_get_cfg_num (xitk, XITK_BG_COLOR));
}

static void _xitk_image_fill_polygon (xitk_image_t *img, const xitk_point_t *points, int npoints, unsigned color) {
  img->beimg->display->lock (img->beimg->display);
  img->beimg->fill_polygon (img->beimg, points, npoints, color, 0);
  img->beimg->display->unlock (img->beimg->display);
}

void xitk_image_fill_polygon (xitk_image_t *img, const xitk_point_t *points, int npoints, unsigned color) {
  if (img && img->beimg && img->xitk)
    _xitk_image_fill_polygon (img, points, npoints, color);
}

void xitk_part_image_draw_menu_arrow_branch (xitk_part_image_t *pi) {
  int w, h, i, x1, x2, x3, y1, y2, y3;
  xitk_image_t *img;
  xitk_point_t points[4];

  if (!pi)
    return;
  img = pi->image;
  if (!img)
    return;
  if (!img->beimg || !img->xitk)
    return;

  w = pi->width / pi->num_states;
  h = pi->height;

  x1 = pi->x + (w - 5);
  y1 = pi->y + (h / 2);

  x2 = pi->x + (w - 10);
  y2 = pi->y + ((h / 2) + 5);

  x3 = pi->x + (w - 10);
  y3 = pi->y + ((h / 2) - 5);

  for(i = 0; i < 3; i++) {

    if(i == 2) {
      x1++; x2++; x3++;
      y1++; y2++; y3++;
    }

    points[0].x = x1;
    points[0].y = y1;
    points[1].x = x2;
    points[1].y = y2;
    points[2].x = x3;
    points[2].y = y3;
    points[3].x = x1;
    points[3].y = y1;

    _xitk_image_fill_polygon (img, &points[0], 4, xitk_get_cfg_num (img->xitk, XITK_BLACK_COLOR));

    x1 += w;
    x2 += w;
    x3 += w;
  }

}

/*
 *
 */
void xitk_image_draw_symbol (xitk_image_t *img, xitk_symbol_t type) {
  int n, w, h, nsegments, linewidth, linelen, s, x1, x2, dx, y1, y2, dy;
  uint32_t color;
  xitk_be_line_t segments[256];

  if (!img || (type == XITK_SYMBOL_USER) || (type == XITK_SYMBOL_NONE))
    return;
  if (!img->beimg)
    return;

  n = (int)img->last_state + 1;
  w = img->width / n;
  h = img->height;

  /*
   * XFillPolygon doesn't yield equally shaped arbitrary sized small triangles
   * because of its filling algorithm (see also fill-rule in XCreateGC(3X11)
   * as for which pixels are drawn on the boundary).
   * So we handcraft them using XDrawSegments applying Bresenham's algorithm.
   */

  /* Coords of the enclosing rectangle for the triangle:   */
  /* Pay attention to integer precision loss and calculate */
  /* carefully to obtain symmetrical and centered shapes.  */
  x1 = ((w - 1) >> 1) - (w >> 2);
  x2 = ((w - 1) >> 1) + (w >> 2);
  y1 = ((h - 1) >> 1) - (h >> 2);
  y2 = ((h - 1) >> 1) + (h >> 2);

  dx = x2 - x1 + 1;
  dy = y2 - y1 + 1;

  nsegments = 0;
  linelen = xitk_min (w, h) - 3;
  linewidth = linelen / 9;
  if (linewidth < 1)
    linewidth = 1;
  linelen -= 2 * linewidth;

  switch (type) {
    case XITK_SYMBOL_UP:
    case XITK_SYMBOL_DOWN:
      {
        int y, iy, dd;

        nsegments = dy;
        if (nsegments > (int)(sizeof (segments) / sizeof (segments[0])))
          nsegments = sizeof (segments) / sizeof (segments[0]);

        if (type == XITK_SYMBOL_DOWN)
          y = y1, iy = 1;
        else
          y = y2, iy = -1;
        dx = (dx + 1) >> 1;
        dd = 0;
        if (dy >= dx) {
          for (s = 0; s < nsegments; s++) {
            segments[s].y1 = y; segments[s].x1 = x1;
            segments[s].y2 = y; segments[s].x2 = x2;
            y += iy;
            if ((dd += dx) >= dy) {
              x1++; x2--;
              dd -= dy;
            }
          }
        } else {
          for (s = 0; s < nsegments; s++) {
            segments[s].y1 = y; segments[s].x1 = x1;
            segments[s].y2 = y; segments[s].x2 = x2;
            y += iy;
            do {
              x1++; x2--;
            } while ((dd += dy) < dx);
            dd -= dx;
          }
        }
      }
      break;
    case XITK_SYMBOL_LEFT:
    case XITK_SYMBOL_RIGHT:
      {
        int x, ix, dd;

        nsegments = dx;
        if (nsegments > (int)(sizeof (segments) / sizeof (segments[0])))
          nsegments = sizeof (segments) / sizeof (segments[0]);

        if (type == XITK_SYMBOL_RIGHT)
          x = x1, ix = 1;
        else
          x = x2, ix = -1;
        dy = (dy + 1) >> 1;
        dd = 0;
        if (dx >= dy) {
          for (s = 0; s < nsegments; s++) {
            segments[s].x1 = x; segments[s].y1 = y1;
            segments[s].x2 = x; segments[s].y2 = y2;
            x += ix;
            if ((dd += dy) >= dx) {
              y1++; y2--;
              dd -= dx;
            }
          }
        } else {
          for (s = 0; s < nsegments; s++) {
            segments[s].x1 = x; segments[s].y1 = y1;
            segments[s].x2 = x; segments[s].y2 = y2;
            x += ix;
            do {
              y1++; y2--;
            } while ((dd += dx) < dy);
            dd -= dy;
          }
        }
      }
      break;
    case XITK_SYMBOL_PLUS:
      {
        int x, m1 = ((w - 1) >> 1) - (linewidth >> 1), m2 = m1 + linewidth;
        /* | */
        for (x = m1; x < m2; x++) {
          segments[nsegments].x1 = x; segments[nsegments].y1 = ((h - 1) >> 1) - (linelen >> 1);
          segments[nsegments].x2 = x; segments[nsegments].y2 = segments[nsegments].y1 + linelen - 1;
          nsegments++;
        }
      }
      /* fall through */
    case XITK_SYMBOL_MINUS:
      {
        int y, m1 = ((h - 1) >> 1) - (linewidth >> 1), m2 = m1 + linewidth;
        /* - */
        for (y = m1; y < m2; y++) {
          segments[nsegments].x1 = ((w - 1) >> 1) - (linelen >> 1); segments[nsegments].y1 = y;
          segments[nsegments].x2 = segments[nsegments].x1 + linelen - 1; segments[nsegments].y2 = y;
          nsegments++;
        }
      }
    default: ;
  }

  img->beimg->display->lock (img->beimg->display);
  do {
    color = xitk_get_cfg_num (img->xitk, XITK_BLACK_COLOR);
    img->beimg->draw_lines (img->beimg, segments, nsegments, color, 0);
    if (n < 2)
        break;
    for (s = 0; s < nsegments; s++) {
      segments[s].x1 += w;
      segments[s].x2 += w;
    }
    img->beimg->draw_lines (img->beimg, segments, nsegments, color, 0);
    if (n < 3)
        break;
    dx = w + 1;
    for (s = 0; s < nsegments; s++) {
      segments[s].x1 += dx;
      segments[s].x2 += dx;
      segments[s].y1 += 1;
      segments[s].y2 += 1;
    }
    color = xitk_get_cfg_num (img->xitk, XITK_FOCUS_COLOR);
    img->beimg->draw_lines (img->beimg, segments, nsegments, color, 0);
    if (n < 4)
        break;
    for (s = 0; s < nsegments; s++) {
      segments[s].x1 += w;
      segments[s].x2 += w;
    }
    img->beimg->draw_lines (img->beimg, segments, nsegments, color, 0);
    if (n < 5)
        break;
    dx = w - 1;
    for (s = 0; s < nsegments; s++) {
      segments[s].x1 += dx;
      segments[s].x2 += dx;
      segments[s].y1 -= 1;
      segments[s].y2 -= 1;
    }
    color = xitk_get_cfg_num (img->xitk, XITK_SELECT_COLOR);
    img->beimg->draw_lines (img->beimg, segments, nsegments, color, 0);
    if (n < 6)
        break;
    dx = w + 1;
    for (s = 0; s < nsegments; s++) {
      segments[s].x1 += dx;
      segments[s].x2 += dx;
      segments[s].y1 += 1;
      segments[s].y2 += 1;
    }
    color = xitk_get_cfg_num (img->xitk, XITK_BG_COLOR);
    img->beimg->draw_lines (img->beimg, segments, nsegments, color, 0);
  } while (0);
  img->beimg->display->unlock (img->beimg->display);
}

/*
 *
 */
void xitk_image_draw_line (xitk_image_t *img, int x0, int y0, int x1, int y1, unsigned color) {
  if (img && img->beimg) {
    xitk_be_line_t lines[1];

    lines[0].x1 = x0; lines[0].y1 = y0; lines[0].x2 = x1; lines[0].y2 = y1;
    img->beimg->display->lock (img->beimg->display);
    img->beimg->draw_lines (img->beimg, lines, 1, color, 0);
    img->beimg->display->unlock (img->beimg->display);
  }
}

void xitk_image_draw_rectangle (xitk_image_t *img, int x, int y, int w, int h, unsigned int color) {
  if (img && img->beimg) {
    xitk_be_line_t lines[4];

    lines[0].x1 = x;         lines[0].y1 = y;         lines[0].x2 = x + w - 1; lines[0].y2 = y;
    lines[1].x1 = x;         lines[1].y1 = y + h - 1; lines[1].x2 = x + w - 1; lines[1].y2 = y + h - 1;
    lines[2].x1 = x;         lines[2].y1 = y;         lines[2].x2 = x;         lines[2].y2 = y + h - 1;
    lines[3].x1 = x + w - 1; lines[3].y1 = y;         lines[3].x2 = x + w - 1; lines[3].y2 = y + h - 1;
    img->beimg->display->lock (img->beimg->display);
    img->beimg->draw_lines (img->beimg, lines, 4, color, 0);
    img->beimg->display->unlock (img->beimg->display);
  }
}

/*
 *
 */
static void _xitk_image_draw_rectangular_box (xitk_image_t *img,
  int x, int y, int excstart, int excstop, int width, int height, uint32_t type) {
  unsigned int color[2];
  xitk_be_line_t xs[5], *q;

  {
    static const uint8_t tab1[4] = { [XITK_DRAW_OUTTER / XITK_DRAW_INNER] = 1 };
    static const uint8_t tab2[2] = { XITK_BLACK_COLOR, XITK_SELECT_COLOR };
    uint32_t idx = tab1[(type / XITK_DRAW_INNER) & 3];
    color[idx ^ 1] = xitk_get_cfg_num (img->xitk, XITK_WHITE_COLOR);
    color[idx] = xitk_get_cfg_num (img->xitk, tab2[(type / XITK_DRAW_LIGHT) & 1]);
  }

  /* +---     ----- *              | *
   * |              *              | *
   * |              * -------------+ */
  q = xs;
  if (excstart < excstop) {
    q->x1 = x + 1; q->x2 = x + excstart;        q->y1 = q->y2 = y; q++;
    q->x1 = x + excstop; q->x2 = x + width - 2; q->y1 = q->y2 = y; q++;
  } else {
    q->x1 = x + 1; q->x2 = x + width - 2;       q->y1 = q->y2 = y; q++;
  }
  q->x1 = q->x2 = x; q->y1 = y + 1; q->y2 = y + height - 2; q++;
  if (type & XITK_DRAW_DOUBLE) {
    q->x1 = q->x2 = x + width - 2;        q->y1 = y + 2; q->y2 = y + height - 3; q++;
    q->x1 = x + 2; q->x2 = x + width - 3; q->y1 = q->y2 = y + height - 2; q++;
  }
  img->beimg->display->lock (img->beimg->display);
  img->beimg->draw_lines (img->beimg, xs, q - xs, color[0], 0);
  img->beimg->display->unlock (img->beimg->display);

  /*              | * +---     ----- *
   *              | * |              *
   * -------------+ * |              */
  q = xs;
  q->x1 = q->x2 = x + width - 1;        q->y1 = y + 1; q->y2 = y + height - 2; q++;
  q->x1 = x + 1; q->x2 = x + width - 2; q->y1 = q->y2 = y + height - 1;        q++;
  if (type & XITK_DRAW_DOUBLE) {
    if (excstart < excstop) {
      q->x1 = x + 2; q->x2 = x + excstart;        q->y1 = q->y2 = y + 1; q++;
      q->x1 = x + excstop; q->x2 = x + width - 3; q->y1 = q->y2 = y + 1; q++;
    } else {
      q->x1 = x + 2; q->x2 = x + width - 3;       q->y1 = q->y2 = y + 1; q++;
    }
    q->x1 = q->x2 = x + 1; q->y1 = y + 2; q->y2 = y + height - 3; q++;
  }
  img->beimg->display->lock (img->beimg->display);
  img->beimg->draw_lines (img->beimg, xs, q - xs, color[1], 0);
  img->beimg->display->unlock (img->beimg->display);
}

/*
 *
 */
void xitk_image_draw_rectangular_box (xitk_image_t *img, int x, int y, int width, int height, uint32_t type) {
  if (!img)
    return;
  if (!img->beimg || !img->xitk)
    return;
  _xitk_image_draw_rectangular_box (img, x, y, 0, 0, width, height, type);
}

static void _xitk_image_draw_check_round (xitk_image_t *img, int x, int y, int d, int checked) {
  img->beimg->display->lock (img->beimg->display);
  img->beimg->fill_arc (img->beimg, x, y, d, d, (30 * 64), (180 * 64),
    xitk_get_cfg_num (img->xitk, XITK_BLACK_COLOR), 0);
  img->beimg->fill_arc (img->beimg, x, y, d, d, (210 * 64), (180 * 64),
    xitk_get_cfg_num (img->xitk, XITK_SELECT_COLOR), 0);
  img->beimg->fill_arc (img->beimg, x + 2, y + 2, d - 4, d - 4, (0 * 64), (360 * 64),
    xitk_get_cfg_num (img->xitk, XITK_WHITE_COLOR), 0);
  if (checked)
    img->beimg->fill_arc (img->beimg, x + 4, y + 4, d - 8, d - 8, (0 * 64), (360 * 64),
      xitk_get_cfg_num (img->xitk, XITK_BLACK_COLOR), 0);
  img->beimg->display->unlock (img->beimg->display);
}

typedef enum {
  _XITK_IMG_STATE_NORMAL = 0,
  _XITK_IMG_STATE_FOCUS,
  _XITK_IMG_STATE_CLICK,
  _XITK_IMG_STATE_SELECTED,
  _XITK_IMG_STATE_SEL_FOCUS,
  _XITK_IMG_STATE_SEL_CLICK,
  _XITK_IMG_STATE_DISABLED_NORMAL,
  _XITK_IMG_STATE_DISABLED_SELECTED,
  _XITK_IMG_STATE_LAST
} _xitk_img_state_t;

xitk_img_state_t xitk_image_find_state (xitk_img_state_t max, uint32_t state) {
  static const uint8_t want[16] = {
    _XITK_IMG_STATE_DISABLED_NORMAL,
    _XITK_IMG_STATE_DISABLED_SELECTED,
    _XITK_IMG_STATE_DISABLED_NORMAL,
    _XITK_IMG_STATE_DISABLED_SELECTED,
    _XITK_IMG_STATE_DISABLED_NORMAL,
    _XITK_IMG_STATE_DISABLED_SELECTED,
    _XITK_IMG_STATE_DISABLED_NORMAL,
    _XITK_IMG_STATE_DISABLED_SELECTED,

    _XITK_IMG_STATE_NORMAL,
    _XITK_IMG_STATE_SELECTED,
    _XITK_IMG_STATE_NORMAL,
    _XITK_IMG_STATE_SELECTED,
    _XITK_IMG_STATE_FOCUS,
    _XITK_IMG_STATE_SEL_FOCUS,
    _XITK_IMG_STATE_CLICK,
    _XITK_IMG_STATE_SEL_CLICK
  };
  static const uint8_t have[XITK_IMG_STATE_LAST][_XITK_IMG_STATE_LAST] = {
    [XITK_IMG_STATE_NORMAL] = {
        [_XITK_IMG_STATE_NORMAL]            = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_FOCUS]             = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_CLICK]             = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_SELECTED]          = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_SEL_FOCUS]         = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_SEL_CLICK]         = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_DISABLED_NORMAL]   = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_DISABLED_SELECTED] = XITK_IMG_STATE_NORMAL
    },
    [XITK_IMG_STATE_FOCUS] = {
        [_XITK_IMG_STATE_NORMAL]            = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_FOCUS]             = XITK_IMG_STATE_FOCUS,
        [_XITK_IMG_STATE_CLICK]             = XITK_IMG_STATE_FOCUS,
        [_XITK_IMG_STATE_SELECTED]          = XITK_IMG_STATE_FOCUS,
        [_XITK_IMG_STATE_SEL_FOCUS]         = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_SEL_CLICK]         = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_DISABLED_NORMAL]   = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_DISABLED_SELECTED] = XITK_IMG_STATE_FOCUS
    },
    [XITK_IMG_STATE_SELECTED] = {
        [_XITK_IMG_STATE_NORMAL]            = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_FOCUS]             = XITK_IMG_STATE_FOCUS,
        [_XITK_IMG_STATE_CLICK]             = XITK_IMG_STATE_SELECTED,
        [_XITK_IMG_STATE_SELECTED]          = XITK_IMG_STATE_SELECTED,
        [_XITK_IMG_STATE_SEL_FOCUS]         = XITK_IMG_STATE_FOCUS,
        [_XITK_IMG_STATE_SEL_CLICK]         = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_DISABLED_NORMAL]   = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_DISABLED_SELECTED] = XITK_IMG_STATE_SELECTED
    },
    [XITK_IMG_STATE_SEL_FOCUS] = {
        [_XITK_IMG_STATE_NORMAL]            = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_FOCUS]             = XITK_IMG_STATE_FOCUS,
        [_XITK_IMG_STATE_CLICK]             = XITK_IMG_STATE_SEL_FOCUS,
        [_XITK_IMG_STATE_SELECTED]          = XITK_IMG_STATE_SELECTED,
        [_XITK_IMG_STATE_SEL_FOCUS]         = XITK_IMG_STATE_SEL_FOCUS,
        [_XITK_IMG_STATE_SEL_CLICK]         = XITK_IMG_STATE_FOCUS,
        [_XITK_IMG_STATE_DISABLED_NORMAL]   = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_DISABLED_SELECTED] = XITK_IMG_STATE_SELECTED
    },
    [XITK_IMG_STATE_DISABLED_NORMAL] = {
        [_XITK_IMG_STATE_NORMAL]            = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_FOCUS]             = XITK_IMG_STATE_FOCUS,
        [_XITK_IMG_STATE_CLICK]             = XITK_IMG_STATE_SEL_FOCUS,
        [_XITK_IMG_STATE_SELECTED]          = XITK_IMG_STATE_SELECTED,
        [_XITK_IMG_STATE_SEL_FOCUS]         = XITK_IMG_STATE_SEL_FOCUS,
        [_XITK_IMG_STATE_SEL_CLICK]         = XITK_IMG_STATE_FOCUS,
        [_XITK_IMG_STATE_DISABLED_NORMAL]   = XITK_IMG_STATE_DISABLED_NORMAL,
        [_XITK_IMG_STATE_DISABLED_SELECTED] = XITK_IMG_STATE_DISABLED_NORMAL
    },
    [XITK_IMG_STATE_DISABLED_SELECTED] = {
        [_XITK_IMG_STATE_NORMAL]            = XITK_IMG_STATE_NORMAL,
        [_XITK_IMG_STATE_FOCUS]             = XITK_IMG_STATE_FOCUS,
        [_XITK_IMG_STATE_CLICK]             = XITK_IMG_STATE_SEL_FOCUS,
        [_XITK_IMG_STATE_SELECTED]          = XITK_IMG_STATE_SELECTED,
        [_XITK_IMG_STATE_SEL_FOCUS]         = XITK_IMG_STATE_SEL_FOCUS,
        [_XITK_IMG_STATE_SEL_CLICK]         = XITK_IMG_STATE_FOCUS,
        [_XITK_IMG_STATE_DISABLED_NORMAL]   = XITK_IMG_STATE_DISABLED_NORMAL,
        [_XITK_IMG_STATE_DISABLED_SELECTED] = XITK_IMG_STATE_DISABLED_SELECTED
    }
  };
  uint32_t u = xitk_bitmove (state, XITK_WIDGET_STATE_ENABLE, 8)
             | xitk_bitmove (state, XITK_WIDGET_STATE_MOUSE,  4)
             | xitk_bitmove (state, XITK_WIDGET_STATE_FOCUS,  4)
             | xitk_bitmove (state, XITK_WIDGET_STATE_CLICK,  2)
             | xitk_bitmove (state, XITK_WIDGET_STATE_ON,     1);
  /* cant be on, dont be on. */
  u &= xitk_bitmove (state, XITK_WIDGET_STATE_TOGGLE, 1) | ~1u;
  /* revert early (de)selection. */
  u ^= (0x80000000 - ((~state) & (XITK_WIDGET_STATE_TOGGLE | XITK_WIDGET_STATE_CLICK | XITK_WIDGET_STATE_IMMEDIATE))) >> 31;
  if (max > XITK_IMG_STATE_LAST - 1)
    max = XITK_IMG_STATE_LAST - 1;
  return have[max][want[u]];
}

void xitk_part_image_states (xitk_part_image_t *part_image, xitk_hv_t *list, int default_num_states) {
  XITK_HV_INIT;
  xitk_hv_t pos, bump;
  int n;

  if (!part_image || !list)
    return;
  if ((default_num_states / 2) == 0)
    default_num_states = 1;
  if ((part_image->num_states / 2) == 0) /* -1, 0, 1 */
    part_image->num_states = default_num_states;
  bump.w = 0;
  if (part_image->num_states >= 0) {
    /* horizontal stack */
    XITK_HV_H (list[0]) = XITK_HV_H (bump) = part_image->width / part_image->num_states;
    XITK_HV_V (list[0]) = part_image->height;
  } else {
    /* vertical stack */
    part_image->num_states = -part_image->num_states;
    XITK_HV_H (list[0]) = part_image->width;
    XITK_HV_V (list[0]) = XITK_HV_V (bump) = part_image->height / part_image->num_states;
  }
  if (part_image->num_states > (int)XITK_IMG_STATE_LAST)
    part_image->num_states = XITK_IMG_STATE_LAST;
  pos.w = 0;
  for (n = 0; n <= part_image->num_states; n++) {
    list[1 + n].w = pos.w;
    pos.w += bump.w;
  }
}

static void _xitk_image_draw_check_check (xitk_image_t *img, int x, int y, int d, xitk_img_state_t state) {
  static const uint8_t bg[XITK_IMG_STATE_LAST] = {
    [XITK_IMG_STATE_NORMAL]            = XITK_FOCUS_COLOR,
    [XITK_IMG_STATE_FOCUS]             = XITK_WHITE_COLOR,
    [XITK_IMG_STATE_SELECTED]          = XITK_FOCUS_COLOR,
    [XITK_IMG_STATE_SEL_FOCUS]         = XITK_WHITE_COLOR,
    [XITK_IMG_STATE_DISABLED_NORMAL]   = XITK_BG_COLOR,
    [XITK_IMG_STATE_DISABLED_SELECTED] = XITK_BG_COLOR
  };
  static const uint8_t sel[XITK_IMG_STATE_LAST] = {
    [XITK_IMG_STATE_NORMAL]            = 0,
    [XITK_IMG_STATE_FOCUS]             = 0,
    [XITK_IMG_STATE_SELECTED]          = 1,
    [XITK_IMG_STATE_SEL_FOCUS]         = 1,
    [XITK_IMG_STATE_DISABLED_NORMAL]   = 0,
    [XITK_IMG_STATE_DISABLED_SELECTED] = 1
  };
  xitk_be_rect_t xr[1];
  xitk_be_line_t xs[4];

  /* background */
  xr[0].x = x, xr[0].y = y, xr[0].w = xr[0].h = d;
  img->beimg->display->lock (img->beimg->display);
  img->beimg->fill_rects (img->beimg, xr, 1, xitk_get_cfg_num (img->xitk, bg[state]), 0);
  img->beimg->display->unlock (img->beimg->display);
  /* */
  xs[0].x1 = x, xs[0].y1 = y, xs[0].x2 = x + d - 1, xs[0].y2 = y;
  xs[1].x1 = x, xs[1].y1 = y, xs[1].x2 = x,         xs[1].y2 = y + d - 1;
  img->beimg->display->lock (img->beimg->display);
  img->beimg->draw_lines (img->beimg, xs, 2, xitk_get_cfg_num (img->xitk, XITK_BLACK_COLOR), 0);
  img->beimg->display->unlock (img->beimg->display);

  xs[0].x1 = x,         xs[0].y1 = y + d - 1, xs[0].x2 = x + d - 1, xs[0].y2 = y + d - 1;
  xs[1].x1 = x + d - 1, xs[1].y1 = y,         xs[1].x2 = x + d - 1, xs[1].y2 = y + d - 1;
  img->beimg->display->lock (img->beimg->display);
  img->beimg->draw_lines (img->beimg, xs, 2, xitk_get_cfg_num (img->xitk, XITK_SELECT_COLOR), 0);
  img->beimg->display->unlock (img->beimg->display);

  if (sel[state]) {
    xs[0].x1 = x + (d / 5),     xs[0].y1 = (y + ((d / 3) * 2)) - 2, xs[0].x2 = x + (d / 2),     xs[0].y2 = y + d - 3;
    xs[1].x1 = x + (d / 5) + 1, xs[1].y1 = (y + ((d / 3) * 2)) - 2, xs[1].x2 = x + (d / 2) + 1, xs[1].y2 = y + d - 3;
    xs[2].x1 = x + (d / 2),     xs[2].y1 =  y +   d            - 3, xs[2].x2 = x +  d      - 3, xs[2].y2 = y     + 1;
    xs[3].x1 = x + (d / 2) + 1, xs[3].y1 =  y +   d            - 3, xs[3].x2 = x +  d      - 2, xs[3].y2 = y     + 1;
    img->beimg->display->lock (img->beimg->display);
    img->beimg->draw_lines (img->beimg, xs, 4, xitk_get_cfg_num (img->xitk, XITK_BLACK_COLOR), 0);
    img->beimg->display->unlock (img->beimg->display);
  }
}

void xitk_part_image_draw_menu_check (xitk_part_image_t *pi, int checked) {
  xitk_image_t *img;
  int style, w, h;

  if (!pi)
    return;
  img = pi->image;
  if (!img)
    return;
  if (!img->beimg || !img->xitk)
    return;

  w = pi->width / pi->num_states;
  h = pi->height;
  style = xitk_get_cfg_num (img->xitk, XITK_CHECK_STYLE);

  switch (style) {
    case CHECK_STYLE_CHECK:
      h -= 8;
      _xitk_image_draw_check_check (img, pi->x + 4, pi->y + 4, h, checked ? XITK_IMG_STATE_SELECTED : XITK_IMG_STATE_NORMAL);
      _xitk_image_draw_check_check (img, pi->x + 4 + w, pi->y + 4, h, checked ? XITK_IMG_STATE_SEL_FOCUS : XITK_IMG_STATE_FOCUS);
      _xitk_image_draw_check_check (img, pi->x + 4 + 2 * w, pi->y + 4, h, checked ? XITK_IMG_STATE_FOCUS : XITK_IMG_STATE_SEL_FOCUS);
      break;

    case CHECK_STYLE_ROUND:
        h -= 8;
      _xitk_image_draw_check_round (img, pi->x + 4, pi->y + 4, h, checked);
      _xitk_image_draw_check_round (img, pi->x + 4 + w, pi->y + 4, h, checked);
      _xitk_image_draw_check_round (img, pi->x + 4 + 2 * w, pi->y + 4, h, !checked);
      break;

    case CHECK_STYLE_OLD:
    default:
      {
        int relief = (checked) ? XITK_DRAW_INNER : XITK_DRAW_OUTTER;
        int nrelief = (checked) ? XITK_DRAW_OUTTER : XITK_DRAW_INNER;

        h -= 12;
        _xitk_image_draw_rectangular_box (img, pi->x + 4,               pi->y + 6,     0, 0, 12, h, relief);
        _xitk_image_draw_rectangular_box (img, pi->x + w + 4,           pi->y + 6,     0, 0, 12, h, relief);
        _xitk_image_draw_rectangular_box (img, pi->x + (w * 2) + 4 + 1, pi->y + 6 + 1, 0, 0, 12, h, nrelief);
      }
      break;
  }
}

static uint32_t _xitk_bg_color_from_style (uint32_t gray, uint32_t style) {
  uint32_t sat = style >> 24;
  const uint32_t div = ((1u << 23) + 50 * 255) / (100 * 255);
  static const uint32_t tab_h[8] = {  0 * div, 11 * div, 59 * div, 70 * div, 30 * div, 41 * div, 89 * div, 100 * div };
  static const uint32_t tab_m[8] = { 0x000000, 0x000001, 0x000100, 0x000101, 0x010000, 0x010001, 0x010100, 0x010101 };
  uint32_t hi, lo, mode = (style / XITK_DRAW_B) & 7;

  /* tint to rgb base color of same SD video luminance with given saturation, and limit rgb value to 255. */
  hi = (gray * ((1u << 23) + sat * tab_h[mode ^ 7]) + (1u << 22)) >> 23;
  lo = (gray * ((1u << 23) - sat * tab_h[mode])     + (1u << 22)) >> 23;
  hi |= (int32_t)(hi << 23) >> 31;
  hi &= 255;
  return hi * tab_m[mode] + lo * tab_m[mode ^ 7];
}

/*
 *
 */
void xitk_image_draw_bevel_style (xitk_image_t *img, uint32_t style) {
  xitk_part_image_t pi;

  if (!img)
    return;
  pi.image = img;
  pi.x = 0;
  pi.y = 0;
  pi.width = img->width;
  pi.height = img->height;
  if (img->last_state < XITK_IMG_STATE_SELECTED)
    img->last_state = XITK_IMG_STATE_SELECTED;
  pi.num_states = img->last_state + 1;
  xitk_part_image_draw_bevel_style (&pi, style);
}

void xitk_part_image_draw_bevel_style (xitk_part_image_t *pi, uint32_t style) {
  xitk_image_t *img;
  xitk_hv_t tl[XITK_IMG_STATE_LIST_SIZE], br[XITK_IMG_STATE_LIST_SIZE];
  uint32_t _style = style & 0xffff, focuscolor;
  int w, h;
  xitk_be_rect_t xr[1];
  xitk_be_line_t xs[16], *q;
  XITK_HV_INIT;

  if (!pi)
    return;
  img = pi->image;
  if (!img)
    return;
  if (!img->beimg)
    return;

  xitk_part_image_states (pi, tl, 3);
  {
    xitk_hv_t offs;
    uint32_t u, size = tl[0].w - 0x00010001;
    XITK_HV_H (offs) = pi->x;
    XITK_HV_V (offs) = pi->y;
    for (u = 1; u < XITK_IMG_STATE_LIST_SIZE; u++) {
      tl[u].w += offs.w;
      br[u].w = tl[u].w + size;
    }
  }

  focuscolor = xitk_color_db_get (img->xitk, _xitk_bg_color_from_style (xitk_get_cfg_num (img->xitk, XITK_FOCUS_GRAY), style));

  w = XITK_HV_H (tl[0]);
  h = XITK_HV_V (tl[0]);

  img->beimg->display->lock (img->beimg->display);
  xr[0].x = XITK_HV_H (tl[1]), xr[0].y = XITK_HV_V (tl[1]);
  xr[0].w = w, xr[0].h = h;
  img->beimg->fill_rects (img->beimg, xr, 1, xitk_get_cfg_num (img->xitk, XITK_BG_COLOR), 0);
  xr[0].x = XITK_HV_H (tl[2]), xr[0].y = XITK_HV_V (tl[2]);
  xr[0].w = XITK_HV_H (br[3]) - XITK_HV_H (tl[2]);
  xr[0].h = XITK_HV_V (br[3]) - XITK_HV_V (tl[2]);
  img->beimg->fill_rects (img->beimg, xr, 1, focuscolor, 0);
  img->beimg->display->unlock (img->beimg->display);

  /* +----+----            *      +----            *
   * |    |                *      |                *
   * |    |                *      |                */
  q = xs;
  if (_style == XITK_DRAW_BEVEL) {
    q->x1 = XITK_HV_H (tl[1]); q->x2 = XITK_HV_H (br[1]);
    q->y1 = q->y2 = XITK_HV_V (tl[1]); q++;
    q->x1 = q->x2 = XITK_HV_H (tl[1]);
    q->y1 = XITK_HV_V (tl[1]); q->y2 = XITK_HV_V (br[1]); q++;
  }
  q->x1 = XITK_HV_H (tl[2]); q->x2 = XITK_HV_H (br[2]);
  q->y1 = q->y2 = XITK_HV_V (tl[2]); q++;
  q->x1 = q->x2 = XITK_HV_H (tl[2]);
  q->y1 = XITK_HV_V (tl[2]); q->y2 = XITK_HV_V (br[2]); q++;
  img->beimg->display->lock (img->beimg->display);
  img->beimg->draw_lines (img->beimg, xs, q - xs, xitk_get_cfg_num (img->xitk, XITK_WHITE_COLOR), 0);
  img->beimg->display->unlock (img->beimg->display);

  /*     |    |+----+----  *          |+----+----  *
   *     |    ||    |      *          ||    |      *
   * ----+----+|    |      *      ----+|    |      */
  q = xs;
  if (_style == XITK_DRAW_BEVEL) {
    q->x1 = q->x2 = XITK_HV_H (br[1]) - 1;
    q->y1 = XITK_HV_V (tl[1]); q->y2 = XITK_HV_V (br[1]) - 2; q++;
    q->x1 = XITK_HV_H (tl[1]) + 2; q->x2 = XITK_HV_H (br[1]) - 1;
    q->y1 = q->y2 = XITK_HV_V (br[1]) - 1; q++;
  }
  q->x1 = q->x2 = XITK_HV_H (br[2]) - 1;
  q->y1 = XITK_HV_V (tl[2]) + 2; q->y2 = XITK_HV_V (br[2]) - 2; q++;
  q->x1 = XITK_HV_H (tl[2]) + 2; q->x2 = XITK_HV_H (br[2]) - 1;
  q->y1 = q->y2 = XITK_HV_V (br[2]) - 1; q++;
  q->x1 = XITK_HV_H (tl[3]); q->x2 = XITK_HV_H (br[3]) + 1;
  q->y1 = q->y2 = XITK_HV_V (tl[3]); q++;
  q->x1 = q->x2 = XITK_HV_H (tl[3]);
  q->y1 = XITK_HV_V (tl[3]); q->y2 = XITK_HV_V (br[3]); q++;
  if (pi->num_states > XITK_IMG_STATE_SEL_FOCUS) {
    q->x1 = XITK_HV_H (tl[4]); q->x2 = XITK_HV_H (br[4]) + 1;
    q->y1 = q->y2 = XITK_HV_V (tl[4]); q++;
    q->x1 = q->x2 = XITK_HV_H (tl[4]);
    q->y1 = XITK_HV_V (tl[4]); q->y2 = XITK_HV_V (br[4]); q++;
  }
  img->beimg->display->lock (img->beimg->display);
  img->beimg->draw_lines (img->beimg, xs, q - xs, xitk_get_cfg_num (img->xitk, XITK_SELECT_COLOR), 0);
  xr[0].x = XITK_HV_H (tl[3]), xr[0].y = XITK_HV_V (tl[3]);
  xr[0].w = w - 1, xr[0].h = h - 1;
  img->beimg->fill_rects (img->beimg, xr, 1, xitk_get_cfg_num (img->xitk, XITK_SELECT_COLOR), 0);
  img->beimg->display->unlock (img->beimg->display);

  q = xs;
  if (_style == XITK_DRAW_BEVEL) {
    q->x1 = q->x2 = XITK_HV_H (br[1]);
    q->y1 = XITK_HV_V (tl[1]); q->y2 = XITK_HV_V (br[1]); q++;
    q->x1 = XITK_HV_H (tl[1]); q->x2 = XITK_HV_H (br[1]);
    q->y1 = q->y2 = XITK_HV_V (br[1]); q++;
  }
  q->x1 = XITK_HV_H (tl[3]) + 1; q->x2 = XITK_HV_H (br[3]);
  q->y1 = q->y2 = XITK_HV_V (tl[3]) + 1; q++;
  q->x1 = q->x2 = XITK_HV_H (tl[3]) + 1;
  q->y1 = XITK_HV_V (tl[3]) + 1; q->y2 = XITK_HV_V (br[3]) - 1; q++;
  q->x1 = q->x2 = XITK_HV_H (br[2]);
  q->y1 = XITK_HV_V (tl[2]); q->y2 = XITK_HV_V (br[2]) + 1; q++;
  q->x1 = XITK_HV_H (tl[2]); q->x2 = XITK_HV_H (br[2]);
  q->y1 = q->y2 = XITK_HV_V (br[2]); q++;
  img->beimg->display->lock (img->beimg->display);
  img->beimg->draw_lines (img->beimg, xs, q - xs, xitk_get_cfg_num (img->xitk, XITK_BLACK_COLOR), 0);
  if (pi->num_states > XITK_IMG_STATE_SEL_FOCUS) {
    xr[0].x = XITK_HV_H (tl[4]) + 1, xr[0].y = XITK_HV_V (tl[4]) + 1;
    xr[0].w = w - 2, xr[0].h = h - 2;
    img->beimg->fill_rects (img->beimg, xr, 1, xitk_get_cfg_num (img->xitk, XITK_SEL_FOCUS_COLOR), 0);
  }
  img->beimg->display->unlock (img->beimg->display);

  /*                |    | *
   *                |    | *
   *            ----+----+ */
  q = xs;
  q->x1 = q->x2 = XITK_HV_H (br[3]);
  q->y1 = XITK_HV_V (tl[3]) + 1; q->y2 = XITK_HV_V (br[3]); q++;
  q->x1 = XITK_HV_H (tl[3]) + 1; q->x2 = XITK_HV_H (br[3]) - 1;
  q->y1 = q->y2 = XITK_HV_V (br[3]); q++;
  if (pi->num_states > XITK_IMG_STATE_SEL_FOCUS) {
    q->x1 = q->x2 = XITK_HV_H (br[4]);
    q->y1 = XITK_HV_V (tl[4]) + 1; q->y2 = XITK_HV_V (br[4]); q++;
    q->x1 = XITK_HV_H (tl[4]) + 1; q->x2 = XITK_HV_H (br[4]) - 1;
    q->y1 = q->y2 = XITK_HV_V (br[4]); q++;
  }
  img->beimg->display->lock (img->beimg->display);
  img->beimg->draw_lines (img->beimg, xs, q - xs, xitk_get_cfg_num (img->xitk, XITK_WHITE_COLOR), 0);
  img->beimg->display->unlock (img->beimg->display);

  /* +   ++   ++   ++   + *
   *                      *
   * +   ++   ++   ++   + */
  {
    uint32_t mask = ((tl[2].w - tl[1].w) & 0xffff) ? 0xffff : 0xffff0000;
    uint32_t steps[4], u;
    q = xs;
    steps[0] = tl[1].w;
    steps[1] = (tl[0].w - 0x00010001) & mask;
    steps[2] = 0x00010001 & mask;
    steps[3] = (tl[0].w - 0x00010001) & ~mask;
    q->x1 = q->x2 = steps[0] & 0xffff;
    q->y1 = q->y2 = steps[0] >> 16;
    q++;
    for (u = pi->num_states - 1; u; u--) {
      steps[0] += steps[1];
      q->x1 = steps[0] & 0xffff; q->y1 = steps[0] >> 16;
      steps[0] += steps[2];
      q->x2 = steps[0] & 0xffff; q->y2 = steps[0] >> 16;
      q++;
    }
    steps[0] += steps[1];
    q->x1 = q->x2 = steps[0] & 0xffff;
    q->y1 = q->y2 = steps[0] >> 16;
    q++;
    steps[0] = tl[1].w + steps[3];
    q->x1 = q->x2 = steps[0] & 0xffff;
    q->y1 = q->y2 = steps[0] >> 16;
    q++;
    for (u = pi->num_states - 1; u; u--) {
      steps[0] += steps[1];
      q->x1 = steps[0] & 0xffff; q->y1 = steps[0] >> 16;
      steps[0] += steps[2];
      q->x2 = steps[0] & 0xffff; q->y2 = steps[0] >> 16;
      q++;
    }
    steps[0] += steps[1];
    q->x1 = q->x2 = steps[0] & 0xffff;
    q->y1 = q->y2 = steps[0] >> 16;
    q++;
  }
  img->beimg->display->lock (img->beimg->display);
  img->beimg->draw_lines (img->beimg, xs, q - xs, xitk_get_cfg_num (img->xitk, XITK_BG_COLOR), 0);
  img->beimg->display->unlock (img->beimg->display);

  /* 0 1 2 3 -> 0 1 2 3 0 2 */
  if (pi->num_states > XITK_IMG_STATE_DISABLED_NORMAL) {
    img->beimg->display->lock (img->beimg->display);
    img->beimg->copy_rect (img->beimg, img->beimg,
      XITK_HV_H (tl[1]), XITK_HV_V (tl[1]), w, h, XITK_HV_H (tl[5]), XITK_HV_V (tl[5]));
    if (pi->num_states > XITK_IMG_STATE_DISABLED_SELECTED)
      img->beimg->copy_rect (img->beimg, img->beimg,
        XITK_HV_H (tl[3]), XITK_HV_V (tl[3]), w, h, XITK_HV_H (tl[6]), XITK_HV_V (tl[6]));
    img->beimg->display->unlock (img->beimg->display);
  }
}

/*
 *
 */
static void _xitk_image_draw_two_state (xitk_image_t *img, int style) {
  uint32_t _style = style & 0xffff, focuscolor;
  int w, h;
  xitk_be_rect_t xr[1];
  xitk_be_line_t xs[3], *q;

  if (!img)
    return;
  if (!img->beimg)
    return;

  focuscolor = xitk_color_db_get (img->xitk, _xitk_bg_color_from_style (xitk_get_cfg_num (img->xitk, XITK_FOCUS_GRAY), style));

  w = img->width / 2;
  h = img->height;

  img->beimg->display->lock (img->beimg->display);
  xr[0].x = 0, xr[0].y = 0, xr[0].w = w - 1, xr[0].h = h - 1;
  img->beimg->fill_rects (img->beimg, xr, 1, xitk_get_cfg_num (img->xitk, XITK_BG_COLOR), 0);
  xr[0].x = w, xr[0].y = 0, xr[0].w = (w * 2) - 1, xr[0].h = h - 1;
  img->beimg->fill_rects (img->beimg, xr, 1, focuscolor, 0);
  img->beimg->display->unlock (img->beimg->display);

  /* +-----+----- *       +----- *
   * |     |      *       |      *
   * |     |      *       |      */
  q = xs;
  if (_style == XITK_DRAW_BEVEL) {
    q->x1 = 0; q->x2 = 2 * w - 1; q->y1 = q->y2 = 0;        q++;
    q->x1 = q->x2 = 0;            q->y1 = 0; q->y2 = h - 1; q++;
  } else {
    q->x1 = w; q->x2 = 2 * w - 1; q->y1 = q->y2 = 0; q++;
  }
  q->x1 = q->x2 = w; q->y1 = 0; q->y2 = h - 1; q++;
  img->beimg->display->lock (img->beimg->display);
  img->beimg->draw_lines (img->beimg, xs, q - xs, xitk_get_cfg_num (img->xitk, XITK_BLACK_COLOR), 0);
  img->beimg->display->unlock (img->beimg->display);

  /*      |     | *            | *
   *      |     | *            | *
   * -----+-----+ *       -----+ */
  q = xs;
  q->x1 = q->x2 = 2 * w - 1;      q->y1 = 0; q->y2 = h - 1; q++;
  if (_style == XITK_DRAW_BEVEL) {
    q->x1 = q->x2 = 1 * w - 1;    q->y1 = 0; q->y2 = h - 1; q++;
    q->x1 = 0; q->x2 = 2 * w - 1; q->y1 = q->y2 = h - 1;    q++;
  } else {
    q->x1 = 1 * w; q->x2 = 2 * w - 1; q->y1 = q->y2 = h - 1; q++;
  }
  img->beimg->display->lock (img->beimg->display);
  img->beimg->draw_lines (img->beimg, xs, q - xs, xitk_get_cfg_num (img->xitk, XITK_WHITE_COLOR), 0);
  img->beimg->display->unlock (img->beimg->display);
}

/*
 *
 */
void xitk_image_draw_relief (xitk_image_t *img, int w, int h, uint32_t style) {
  if (!img)
    return;
  if (!img->beimg || !img->xitk)
    return;

  _xitk_image_fill_rectangle (img, 0, 0, w, h, xitk_get_cfg_num (img->xitk, XITK_BG_COLOR));

  if (((style & XITK_DRAW_FLATTER) == XITK_DRAW_OUTTER) || ((style & XITK_DRAW_FLATTER) == XITK_DRAW_INNER))
    _xitk_image_draw_rectangular_box (img, 0, 0, 0, 0, w, h, style);
}

void xitk_image_draw_checkbox_check (xitk_image_t *img) {
  int style, w;
  xitk_img_state_t state;

  if (!img)
    return;
  if (!img->xitk || !img->beimg)
    return;

  style = xitk_get_cfg_num (img->xitk, XITK_CHECK_STYLE);

  _xitk_image_fill_rectangle (img, 0, 0, img->width, img->height, xitk_get_cfg_num (img->xitk, XITK_BG_COLOR));

  w = img->width / ((int)img->last_state + 1);
  switch (style) {
    case CHECK_STYLE_CHECK:
      for (state = XITK_IMG_STATE_NORMAL; state <= img->last_state; state += 1)
        _xitk_image_draw_check_check (img, w * (int)state, 0, img->height, state);
      break;

    case CHECK_STYLE_ROUND:
      {
        int w = img->width / 3;
        _xitk_image_draw_check_round (img, 0, 0, img->height, 0);
        _xitk_image_draw_check_round (img, w, 0, img->height, 0);
        _xitk_image_draw_check_round (img, w * 2, 0, img->height, 1);
      }
      break;

    case CHECK_STYLE_OLD:
    default:
      xitk_image_draw_bevel_style (img, XITK_DRAW_BEVEL);
      break;
  }
}

/*
 *
 */
void xitk_image_draw_bevel_two_state (xitk_image_t *img) {
  _xitk_image_draw_two_state (img, XITK_DRAW_BEVEL);
}

void xitk_image_draw_paddle_three_state (xitk_image_t *img, int width, int height) {
  int w, h, gap, m, dir;
  xitk_be_line_t xs[9];
  xitk_be_rect_t xr[3];

  if (!img)
    return;
  if (!img->beimg)
    return;

  dir = width > height;
  w = img->width / 3;
  if ((width > 0) && (width <= w))
    w = width;
  h = img->height;
  if ((height > 0) && (height <= h))
    h = height;

  gap = (w < 11) || (h < 11) ? 1 : 2;

  /* ------------
   * |  ||  ||  |
   * ------------ */
  /* Draw mask */
  xr[0].x = 0, xr[0].y = 0, xr[0].w = img->width, xr[0].h = img->height;
  img->beimg->display->lock (img->beimg->display);
  img->beimg->fill_rects (img->beimg, xr, 1, 0, 1);
  img->beimg->display->unlock (img->beimg->display);
  xr[0].x = 0 * w + 1; xr[0].y = 1; xr[0].w = w - 2; xr[0].h = h - 2;
  xr[1].x = 1 * w + 1; xr[1].y = 1; xr[1].w = w - 2; xr[1].h = h - 2;
  xr[2].x = 2 * w + 1; xr[2].y = 1; xr[2].w = w - 2; xr[2].h = h - 2;
  img->beimg->display->lock (img->beimg->display);
  img->beimg->fill_rects (img->beimg, xr, 3, 1, 1);
  img->beimg->display->unlock (img->beimg->display);

  img->beimg->display->lock (img->beimg->display);
  xr[0].x = 0 * w, xr[0].y = 0, xr[0].w = w, xr[0].h = h;
  img->beimg->fill_rects (img->beimg, xr, 1, xitk_get_cfg_num (img->xitk, XITK_BG_COLOR), 0);
  xr[0].x = 1 * w, xr[0].y = 0, xr[0].w = w, xr[0].h = h;
  img->beimg->fill_rects (img->beimg, xr, 1, xitk_get_cfg_num (img->xitk, XITK_FOCUS_COLOR), 0);
  xr[0].x = 2 * w, xr[0].y = 0, xr[0].w = w, xr[0].h = h;
  img->beimg->fill_rects (img->beimg, xr, 1, xitk_get_cfg_num (img->xitk, XITK_SELECT_COLOR), 0);
  img->beimg->display->unlock (img->beimg->display);
  /* +---+---
   * |   |       |
   *          ---+ */
  xs[0].x1 = 0 * w + gap + 1; xs[0].x2 = 1 * w - gap - 2; xs[0].y1 = xs[0].y2 = gap;
  xs[1].x1 = 1 * w + gap + 1; xs[1].x2 = 2 * w - gap - 2; xs[1].y1 = xs[1].y2 = gap;
  xs[2].x1 = xs[2].x2 = 0 * w + gap;     xs[2].y1 = gap + 1; xs[2].y2 = h - gap - 2;
  xs[3].x1 = xs[3].x2 = 1 * w + gap;     xs[3].y1 = gap + 1; xs[3].y2 = h - gap - 2;
  xs[4].x1 = xs[4].x2 = 3 * w - gap - 1; xs[4].y1 = gap + 1; xs[4].y2 = h - gap - 2;
  xs[5].x1 = 2 * w + gap + 1; xs[5].x2 = 3 * w - gap - 2; xs[5].y1 = xs[5].y2 = h - gap - 1;
  if (!dir) {
    /*   -    -    -   */
    m = (h - 1) >> 1;
    xs[6].x1 = 0 * w + gap + 3; xs[6].x2 = 1 * w - gap - 4; xs[6].y1 = xs[6].y2 = m;
    xs[7].x1 = 1 * w + gap + 3; xs[7].x2 = 2 * w - gap - 4; xs[7].y1 = xs[7].y2 = m;
    xs[8].x1 = 2 * w + gap + 3; xs[8].x2 = 3 * w - gap - 4; xs[8].y1 = xs[8].y2 = m;
  } else {
    /*   |    |    |   */
    m = (w - 1) >> 1;
    xs[6].x1 = xs[6].x2 = 0 * w + m; xs[6].y1 = gap + 3; xs[6].y2 = h - gap - 4;
    xs[7].x1 = xs[7].x2 = 1 * w + m; xs[7].y1 = gap + 3; xs[7].y2 = h - gap - 4;
    xs[8].x1 = xs[8].x2 = 2 * w + m; xs[8].y1 = gap + 3; xs[8].y2 = h - gap - 4;
  }
  img->beimg->display->lock (img->beimg->display);
  img->beimg->draw_lines (img->beimg, xs, 9, xitk_get_cfg_num (img->xitk, XITK_WHITE_COLOR), 0);
  img->beimg->display->unlock (img->beimg->display);
  /*         +---
   *    |   ||
   * ---+---+    */
  xs[0].x1 = 2 * w + gap + 1; xs[0].x2 = 3 * w - gap - 2; xs[0].y1 = xs[0].y2 = gap;
  xs[1].x1 = xs[1].x2 = 1 * w - gap - 1; xs[1].y1 = gap + 1; xs[1].y2 = h - gap - 2;
  xs[2].x1 = xs[2].x2 = 2 * w - gap - 1; xs[2].y1 = gap + 1; xs[2].y2 = h - gap - 2;
  xs[3].x1 = xs[3].x2 = 2 * w + gap;     xs[3].y1 = gap + 1; xs[3].y2 = h - gap - 2;
  xs[4].x1 = 0 * w + gap + 1; xs[4].x2 = 1 * w - gap - 2; xs[4].y1 = xs[4].y2 = h - gap - 1;
  xs[5].x1 = 1 * w + gap + 1; xs[5].x2 = 2 * w - gap - 2; xs[5].y1 = xs[5].y2 = h - gap - 1;
  if (!dir) {
    /*   -    -    -   */
    m = ((h - 1) >> 1) + 1;
    xs[6].x1 = 0 * w + gap + 3; xs[6].x2 = 1 * w - gap - 4; xs[6].y1 = xs[6].y2 = m;
    xs[7].x1 = 1 * w + gap + 3; xs[7].x2 = 2 * w - gap - 4; xs[7].y1 = xs[7].y2 = m;
    xs[8].x1 = 2 * w + gap + 3; xs[8].x2 = 3 * w - gap - 4; xs[8].y1 = xs[8].y2 = m;
  } else {
    /*   |    |    |   */
    m = ((w - 1) >> 1) + 1;
    xs[6].x1 = xs[6].x2 = 0 * w + m; xs[6].y1 = gap + 3; xs[6].y2 = h - gap - 4;
    xs[7].x1 = xs[7].x2 = 1 * w + m; xs[7].y1 = gap + 3; xs[7].y2 = h - gap - 4;
    xs[8].x1 = xs[8].x2 = 2 * w + m; xs[8].y1 = gap + 3; xs[8].y2 = h - gap - 4;
  }
  img->beimg->display->lock (img->beimg->display);
  img->beimg->draw_lines (img->beimg, xs, 9, xitk_get_cfg_num (img->xitk, XITK_BLACK_COLOR), 0);
  img->beimg->display->unlock (img->beimg->display);
}

/*
 *
 */
void xitk_image_draw_paddle_three_state_vertical (xitk_image_t *img) {
  xitk_image_draw_paddle_three_state (img, 0, img->height);
}
void xitk_image_draw_paddle_three_state_horizontal (xitk_image_t *img) {
  xitk_image_draw_paddle_three_state (img, img->width / 3, 0);
}

/*
 * Draw a frame outline with embedded title.
 */
uint32_t xitk_image_draw_frame (xitk_image_t *img, const char *title, const char *fontname,
  int x, int y, int w, int h, uint32_t style) {
  xitk_t *xitk;
  xitk_font_t *fs = NULL;
  uint32_t tlen = 0;
  int yoff = 0, xstart = 0, xstop = 0, ascent = 0, descent = 0, lbearing = 0, rbearing = 0;
  uint32_t bgcolor = 0;
  char buf[BUFSIZ];

  if (!img)
    return 0;
  if (!img->beimg)
    return 0;

  xitk = img->xitk;
#if 0
  title = "This is an intentionally huge and at the same time totally useless text for testing the truncation feature.";
#endif
  do {
    int maxwidth;

    if (!title)
      break;
    fs = xitk_font_load_font (xitk, (fontname ? fontname : DEFAULT_FONT_12));
    if (!fs)
      break;

    maxwidth = w - 12;
    tlen = xitk_find_byte (title, 0);
    xitk_font_text_extent (fs, title, tlen, &lbearing, &rbearing, NULL, &ascent, &descent);
    if (rbearing - lbearing <= maxwidth)
      break;

    if (tlen > sizeof (buf) - 8)
      tlen = sizeof (buf) - 8;
    memcpy (buf, "    ", 4);
    memcpy (buf + 4, title, tlen);
    memcpy (buf + 4 + tlen, "....", 4);
    title = buf + 4;

    do {
      /* remove 1 possibly multibyte char. */
      do {
        buf[4 + (--tlen)] = '.';
      } while ((buf[4 + tlen - 1] & 0xc0) == 0x80);
      xitk_font_text_extent (fs, title, tlen + 3, &lbearing, &rbearing, NULL, &ascent, &descent);
    } while (rbearing - lbearing > maxwidth);
    tlen += 3;
  } while (0);

  if (title) {
    /* Dont draw frame box under frame title */
    yoff = (ascent >> 1) + 1; /* Roughly v-center outline to ascent part of glyphs */
    xstart = 4 - 1;
    xstop = (rbearing - lbearing) + 8;
  }

  if (style & (XITK_DRAW_R | XITK_DRAW_G | XITK_DRAW_B)) {
    xitk_be_rect_t xr[2];

    bgcolor = _xitk_bg_color_from_style (xitk_get_cfg_num (xitk, XITK_BG_GRAY), style);

    xr[0].x = x + 2; xr[0].y = y + 2 + yoff; xr[0].w = w - 4; xr[0].h = h - 4 - yoff;
    xr[1].x = x + xstart; xr[1].y = y; xr[1].w = xstop - xstart; xr[1].h = yoff + 2;
    img->beimg->display->lock (img->beimg->display);
    img->beimg->fill_rects (img->beimg, xr, 1 + (xstart < xstop), xitk_color_db_get (xitk, bgcolor), 0);
    img->beimg->display->unlock (img->beimg->display);
  }

  _xitk_image_draw_rectangular_box (img, x, y + yoff, xstart, xstop, w, h - yoff, style | XITK_DRAW_DOUBLE | XITK_DRAW_LIGHT);

  if (title) {
    _xitk_image_draw_string (img, fs, x - lbearing + 6, y + ascent, title, tlen,
        xitk_get_cfg_num (img->xitk, XITK_BLACK_COLOR));
    xitk_font_unload_font (fs);
  }

  return bgcolor;
}

/*
 *
 */
void xitk_image_draw_tab (xitk_image_t *img) {
  int           w, h;
  xitk_be_line_t xs[13];
  xitk_be_rect_t xr[2];

  if (!img)
    return;
  if (!img->beimg)
    return;

  if (img->last_state < XITK_IMG_STATE_SELECTED)
    img->last_state = XITK_IMG_STATE_SELECTED;
  w = img->width / ((int)img->last_state + 1);
  h = img->height;

  xr[0].x = 0 * w; xr[0].w = 2 * w; xr[0].y = 0; xr[0].h = 5;
  xr[1].x = 2 * w; xr[1].w = 1 * w; xr[1].y = 0; xr[1].h = h;
  img->beimg->display->lock (img->beimg->display);
  img->beimg->fill_rects (img->beimg, xr, 2, xitk_get_cfg_num (img->xitk, XITK_BG_COLOR), 0);
  img->beimg->display->unlock (img->beimg->display);

  xr[0].x = 0 * w + 1; xr[0].w = w - 2; xr[0].y = 4; xr[0].h = h - 4;
  xr[1].x = 1 * w + 1; xr[1].w = w - 2; xr[1].y = 0; xr[1].h = h;
  img->beimg->display->lock (img->beimg->display);
  img->beimg->fill_rects (img->beimg, xr, 2, xitk_get_cfg_num (img->xitk, XITK_FOCUS_COLOR), 0);
  img->beimg->display->unlock (img->beimg->display);

  if (img->last_state >= XITK_IMG_STATE_SEL_FOCUS) {
    xr[0].x = 3 * w + 1; xr[0].w = w - 2; xr[0].y = 0; xr[0].h = h;
    img->beimg->display->lock (img->beimg->display);
    img->beimg->fill_rects (img->beimg, xr, 1, xitk_get_cfg_num (img->xitk, XITK_SELECT_COLOR), 0);
    img->beimg->display->unlock (img->beimg->display);
  }

  /*          *  /-----  *  /-----  *  /----  *
   *  /-----  * |        * |        * |       *
   * |        * |        * |        * |       *
   * +------- * +------- * |        * |       */
  xs[0].x1 = 1 * w + 2; xs[0].x2 = 2 * w - 3; xs[0].y1 = xs[0].y2 = 0;
  xs[1].x1 = xs[1].x2 = 1 * w + 1;            xs[1].y1 = xs[1].y2 = 1;
  xs[2].x1 = 2 * w + 2; xs[2].x2 = 3 * w - 3; xs[2].y1 = xs[2].y2 = 0;
  xs[3].x1 = xs[3].x2 = 2 * w + 1;            xs[3].y1 = xs[3].y2 = 1;
  xs[4].x1 = 0 * w + 2; xs[4].x2 = 1 * w - 3; xs[4].y1 = xs[4].y2 = 3;
  xs[5].x1 = xs[5].x2 = 0 * w + 1;            xs[5].y1 = xs[5].y2 = 4;
  xs[6].x1 = xs[6].x2 = 0 * w;                xs[6].y1 = 5; xs[6].y2 = h - 1;
  xs[7].x1 = xs[7].x2 = 1 * w;                xs[7].y1 = 2; xs[7].y2 = h - 1;
  xs[8].x1 = xs[8].x2 = 2 * w;                xs[8].y1 = 2; xs[8].y2 = h - 1;
  xs[9].x1 = 0 * w; xs[9].x2 = 2 * w - 1;     xs[9].y1 = xs[9].y2 = h - 1;
  xs[10].x1 = 3 * w + 2; xs[10].x2 = 4 * w - 3; xs[10].y1 = xs[10].y2 = 0;
  xs[11].x1 = xs[11].x2 = 3 * w + 1;            xs[11].y1 = xs[11].y2 = 1;
  xs[12].x1 = xs[12].x2 = 3 * w;                xs[12].y1 = 2; xs[12].y2 = h - 1;
  img->beimg->display->lock (img->beimg->display);
  img->beimg->draw_lines (img->beimg, xs, (img->last_state >= XITK_IMG_STATE_SEL_FOCUS ? 13 : 10),
    xitk_get_cfg_num (img->xitk, XITK_WHITE_COLOR), 0);
  img->beimg->display->unlock (img->beimg->display);

  /*          *          *          *          *
   *          *        | *        | *        | *
   *        | *        | *        | *        | *
   *        | *        | *        | *        | */
  xs[0].x1 = xs[0].x2 = 1 * w - 1;            xs[0].y1 = 5; xs[0].y2 = h - 1;
  xs[1].x1 = xs[1].x2 = 2 * w - 1;            xs[1].y1 = 2; xs[1].y2 = h - 1;
  xs[2].x1 = xs[2].x2 = 3 * w - 1;            xs[2].y1 = 2; xs[2].y2 = h - 1;
  xs[3].x1 = xs[3].x2 = 4 * w - 1;            xs[3].y1 = 2; xs[3].y2 = h - 1;
  img->beimg->display->lock (img->beimg->display);
  img->beimg->draw_lines (img->beimg, xs, (img->last_state >= XITK_IMG_STATE_SEL_FOCUS ? 4 : 3),
    xitk_get_cfg_num (img->xitk, XITK_BLACK_COLOR), 0);
  img->beimg->display->unlock (img->beimg->display);
}

/*
 *
 */
void xitk_image_draw_paddle_rotate (xitk_image_t *img) {
  int w, h;
  unsigned int ccolor, fcolor, ncolor;

  if (!img)
    return;
  if (!img->beimg || !img->xitk)
    return;

  w = img->width / 3;
  h = img->height;
  ncolor = xitk_get_cfg_num (img->xitk, XITK_SELECT_COLOR);
  fcolor = xitk_get_cfg_num (img->xitk, XITK_WARN_BG_COLOR);
  ccolor = xitk_get_cfg_num (img->xitk, XITK_FOCUS_COLOR);

  {
    int x, i;
    unsigned int bg_colors[3] = { ncolor, fcolor, ccolor };
    xitk_be_rect_t rect[1] = {{0, 0, w * 3, h}};

    img->beimg->display->lock (img->beimg->display);
    img->beimg->fill_rects (img->beimg, rect, 1, 0, 1);

    for (x = 0, i = 0; i < 3; i++) {
      img->beimg->fill_arc (img->beimg, x, 0, w - 1, h - 1, (0 * 64), (360 * 64), 1, 1);
      img->beimg->draw_arc (img->beimg, x, 0, w - 1, h - 1, (0 * 64), (360 * 64), 1, 1);

      img->beimg->fill_arc (img->beimg, x, 0, w - 1, h - 1, (0 * 64), (360 * 64), bg_colors[i], 0);
      img->beimg->draw_arc (img->beimg, x, 0, w - 1, h - 1, (0 * 64), (360 * 64), xitk_get_cfg_num (img->xitk, XITK_BLACK_COLOR), 0);

      x += w;
    }
    img->beimg->display->unlock (img->beimg->display);
  }
}

/*
 *
 */
void xitk_image_draw_rotate_button (xitk_image_t *img) {
  int w, h;

  if (!img)
    return;
  if (!img->beimg || !img->xitk)
    return;

  w = img->width;
  h = img->height;

  img->beimg->display->lock (img->beimg->display);

  /* Draw mask */
  {
    xitk_be_rect_t rect[1] = {{0, 0, w, h}};
    img->beimg->fill_rects (img->beimg, rect, 1, 0, 1);
  }
  img->beimg->fill_arc (img->beimg, 0, 0, w - 1, h - 1, (0 * 64), (360 * 64), 1, 1);

  /* */
  img->beimg->fill_arc (img->beimg, 0, 0, w - 1, h - 1, (0 * 64), (360 * 64), xitk_get_cfg_num (img->xitk, XITK_BG_COLOR), 0);
/*img->beimg->draw_arc (img->beimg, 0, 0, w - 1, h - 1, (30 * 64), (180 * 64), xitk_get_cfg_num (img->xitk, XITK_WHITE_COLOR), 0);*/
  img->beimg->draw_arc (img->beimg, 1, 1, w - 2, h - 2, (30 * 64), (180 * 64), xitk_get_cfg_num (img->xitk, XITK_WHITE_COLOR), 0);

/*img->beimg->draw_arc (img->beimg, 0, 0, w - 1, h - 1, (210 * 64), (180 * 64), xitk_get_cfg_num (img->xitk, XITK_SELECT_COLOR), 0);*/
  img->beimg->draw_arc (img->beimg, 1, 1, w - 3, h - 3, (210 * 64), (180 * 64), xitk_get_cfg_num (img->xitk, XITK_SELECT_COLOR), 0);

  img->beimg->display->unlock (img->beimg->display);
}

/*
 *
 */
void xitk_part_image_copy (xitk_widget_list_t *wl, xitk_part_image_t *from, xitk_part_image_t *to,
  int src_x, int src_y, int width, int height, int dst_x, int dst_y) {

  if (!wl || !from || !to)
    return;
  if (!from->image || !to->image)
    return;
  if (!from->image->xitk || !from->image->beimg || !to->image->beimg)
    return;

  to->image->beimg->copy_rect (to->image->beimg, from->image->beimg,
    from->x + src_x, from->y + src_y, width, height, to->x + dst_x, to->y + dst_y);
}

void xitk_part_image_draw (xitk_widget_list_t *wl, xitk_part_image_t *origin, xitk_part_image_t *copy,
  int src_x, int src_y, int width, int height, int dst_x, int dst_y) {
  if (!wl || !origin)
    return;
  if (!wl->xwin)
    return;
  if (!wl->xwin->bewin)
    return;
  if (!origin->image)
    return;

  if (copy) {
    if (!copy->image)
      return;
  } else {
    copy = origin;
  }
  if (!copy->image->beimg)
    return;

  wl->xwin->bewin->copy_rect (wl->xwin->bewin, copy->image->beimg,
    copy->x + src_x, copy->y + src_y, width, height, dst_x, dst_y, 0);
}

void xitk_image_draw_image (xitk_widget_list_t *wl, xitk_image_t *img,
  int src_x, int src_y, int width, int height, int dst_x, int dst_y, int sync) {
  if (!wl || !img)
    return;
  if (!wl->xwin)
    return;
  if (!wl->xwin->bewin)
    return;
  if (!img->beimg)
    return;

  wl->xwin->bewin->copy_rect (wl->xwin->bewin, img->beimg,
    src_x, src_y, width, height, dst_x, dst_y, sync);
}

int xitk_image_width(xitk_image_t *i) {
  return i->width;
}

int xitk_image_height(xitk_image_t *i) {
  return i->height;
}

/*
 * ********************************************************************************
 *                              Widget specific part
 * ********************************************************************************
 */

typedef struct {
  xitk_widget_t w;
  xitk_hv_t img_state_list[XITK_IMG_STATE_LIST_SIZE];
  xitk_part_image_t skin;
} _image_private_t;

static void _xitk_image_set_skin (_image_private_t *wp, xitk_skin_config_t *skonfig) {
  const xitk_skin_element_info_t *info = xitk_skin_get_info (skonfig, wp->w.skin_element_name);
  XITK_HV_INIT;

  if (info) {
    wp->skin = info->pixmap_img;
    xitk_part_image_states (&wp->skin, wp->img_state_list, 1);
    XITK_HV_H (wp->w.pos) = info->x;
    XITK_HV_V (wp->w.pos) = info->y;
  } else {
    wp->skin.image = NULL;
    wp->skin.x = 0;
    wp->skin.y = 0;
    wp->skin.width = 0;
    wp->skin.height = 0;
    wp->w.pos.w = 0;
  }
}

static void _xitk_image_change_skin (_image_private_t *wp, xitk_skin_config_t *skonfig) {
  XITK_HV_INIT;

  if (wp->w.skin_element_name[0]) {
    _xitk_image_set_skin (wp, skonfig);
    wp->w.size.w = wp->img_state_list[0].w;
    xitk_set_widget_pos (&wp->w, XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos));
  }
}

static int _xitk_image_event (xitk_widget_t *w, const widget_event_t *event) {
  _image_private_t *wp;
  XITK_HV_INIT;

  xitk_container (wp, w, w);
  if (!wp || !event)
    return 0;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_IMAGE)
    return 0;
  switch (event->type) {
    case WIDGET_EVENT_PAINT:
      if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
        xitk_img_state_t state = xitk_image_find_state (wp->skin.num_states - 1, wp->w.state);
        xitk_part_image_draw (wp->w.wl, &wp->skin, NULL,
          (int)XITK_HV_H (wp->img_state_list[1 + state]) + event->x - XITK_HV_H (wp->w.pos),
          (int)XITK_HV_V (wp->img_state_list[1 + state]) + event->y - XITK_HV_V (wp->w.pos),
          event->width, event->height,
          event->x, event->y);
      }
      return 0;
    case WIDGET_EVENT_INSIDE:
      return xitk_image_inside (wp->skin.image,
        wp->skin.x + event->x - XITK_HV_H (wp->w.pos), wp->skin.y + event->y - XITK_HV_V (wp->w.pos)) ? 1 : 2;
    case WIDGET_EVENT_CHANGE_SKIN:
      _xitk_image_change_skin (wp, event->skonfig);
      return 0;
    case WIDGET_EVENT_DESTROY:
      if (!wp->w.skin_element_name[0])
        xitk_image_free_image (&(wp->skin.image));
      return 0;
    case WIDGET_EVENT_GET_SKIN:
      if (event->image) {
        *event->image = (event->skin_layer == BACKGROUND_SKIN) ? wp->skin.image : NULL;
        return 1;
      }
      return 0;
    default:
      return 0;
  }
}

xitk_widget_t *xitk_image_create (const xitk_image_widget_t *im, xitk_skin_config_t *skonfig) {
  _image_private_t *wp;

  /* Always be there for the application, even if the start skin is missing us here.
   * User may switch to a "better" skin later. */
  wp = (_image_private_t *)xitk_widget_new (&im->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  _xitk_image_set_skin (wp, skonfig);

  wp->w.state   &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
  wp->w.size.w = wp->img_state_list[0].w;
  wp->w.type     = WIDGET_TYPE_IMAGE | WIDGET_PARTIAL_PAINTABLE;
  wp->w.event    = _xitk_image_event;

  _xitk_new_widget_apply (&im->nw, &wp->w);

  return &wp->w;
}

/*
 *
 */
xitk_widget_t *xitk_noskin_image_create (const xitk_image_widget_t *im, xitk_image_t *image, int x, int y) {
  _image_private_t *wp;
  XITK_HV_INIT;

  wp = (_image_private_t *)xitk_widget_new (&im->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  wp->w.skin_element_name[0] = 0;

  wp->skin.image  = image;
  wp->skin.x      = 0;
  wp->skin.y      = 0;
  wp->skin.width  = image ? image->width : 0;
  wp->skin.height = image ? image->height : 0;
  xitk_image_ref (wp->skin.image);
  xitk_part_image_states (&wp->skin, wp->img_state_list, 1);
  wp->w.state   &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
  XITK_HV_H (wp->w.pos) = x;
  XITK_HV_V (wp->w.pos) = y;
  wp->w.size.w = wp->img_state_list[0].w;
  wp->w.type     = WIDGET_TYPE_IMAGE | WIDGET_PARTIAL_PAINTABLE;
  wp->w.event    = _xitk_image_event;

  _xitk_new_widget_apply (&im->nw, &wp->w);

  return &wp->w;
}

xitk_widget_t *xitk_noskin_part_image_create (const xitk_image_widget_t *im, const xitk_part_image_t *image, int x, int y) {
  _image_private_t *wp;
  XITK_HV_INIT;

  if (!image)
    return NULL;
  if (!image->image)
    return NULL;

  wp = (_image_private_t *)xitk_widget_new (&im->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  wp->w.skin_element_name[0] = 0;

  wp->skin       = *image;
  xitk_image_ref (wp->skin.image);
  xitk_part_image_states (&wp->skin, wp->img_state_list, 1);
  wp->w.state   &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
  XITK_HV_H (wp->w.pos) = x;
  XITK_HV_V (wp->w.pos) = y;
  wp->w.size.w = wp->img_state_list[0].w;
  wp->w.type     = WIDGET_TYPE_IMAGE | WIDGET_PARTIAL_PAINTABLE;
  wp->w.event    = _xitk_image_event;

  _xitk_new_widget_apply (&im->nw, &wp->w);

  return &wp->w;
}

typedef struct {
  xitk_widget_t w;
  xitk_image_t *bg;
  unsigned int color[6], shown;
} _color_rect_private_t;

static int _xitk_color_rect_event (xitk_widget_t *w, const widget_event_t *event) {
  _color_rect_private_t *wp;
  XITK_HV_INIT;

  xitk_container (wp, w, w);
  if (!wp || !event)
    return 0;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_IMAGE)
    return 0;
  switch (event->type) {
    case WIDGET_EVENT_PAINT:
      /* NOTE: this is used for noskin window borders. state changes are rare (window (un)focus).
       * also, we have no backend interface for rendering into a window directly. */
      if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
        xitk_img_state_t state = xitk_image_find_state (sizeof (wp->color) / sizeof (wp->color[0]), wp->w.state);
        unsigned int color = wp->color[state];
        if (color != wp->shown) {
          /* color change: paint into the window background and update whole widget. */
          wp->shown = color;
          _xitk_image_fill_rectangle (wp->bg, XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos),
            XITK_HV_H (wp->w.size), XITK_HV_V (wp->w.size), color);
          xitk_image_draw_image (wp->w.wl, wp->bg, XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos),
            XITK_HV_H (wp->w.size), XITK_HV_V (wp->w.size), XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos), 0);
        } else {
          /* just in case the server uses a private copy of the background image that still has
           * the old color: repaint selected part. */
          xitk_image_draw_image (wp->w.wl, wp->bg,
            event->x, event->y, event->width, event->height, event->x, event->y, 0);
        }
      }
      return 0;
    case WIDGET_EVENT_INSIDE:
      return 1;
    case WIDGET_EVENT_DESTROY:
      return 0;
    case WIDGET_EVENT_GET_SKIN:
      return 0;
    default:
      return 0;
  }
}

xitk_widget_t *xitk_noskin_color_rect_create (const xitk_color_rect_widget_t *rw, int x, int y) {
  _color_rect_private_t *wp;
  XITK_HV_INIT;
  unsigned int u;

  if (!rw->nw.wl->xwin)
    return NULL;
  if (!rw->nw.wl->xwin->bg_image)
    return NULL;

  wp = (_color_rect_private_t *)xitk_widget_new (&rw->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  wp->bg = rw->nw.wl->xwin->bg_image;
  for (u = 0; u < sizeof (wp->color) / sizeof (wp->color[0]); u++)
    wp->color[u] = rw->color[u];
  wp->shown = ~0u;

  wp->w.state   &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
  XITK_HV_H (wp->w.pos) = x;
  XITK_HV_V (wp->w.pos) = y;
  XITK_HV_H (wp->w.size) = rw->width;
  XITK_HV_V (wp->w.size) = rw->height;
  wp->w.type     = WIDGET_TYPE_IMAGE | WIDGET_PARTIAL_PAINTABLE;
  wp->w.event    = _xitk_color_rect_event;

  _xitk_new_widget_apply (&rw->nw, &wp->w);

  return &wp->w;
}
