% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MCMCnegbin.R
\name{MCMCnegbin}
\alias{MCMCnegbin}
\title{Markov Chain Monte Carlo for Negative Binomial Regression}
\usage{
MCMCnegbin(
  formula,
  data = parent.frame(),
  b0 = 0,
  B0 = 1,
  e = 2,
  f = 2,
  g = 10,
  burnin = 1000,
  mcmc = 1000,
  thin = 1,
  verbose = 0,
  seed = NA,
  beta.start = NA,
  rho.start = NA,
  rho.step = 0.1,
  nu.start = NA,
  marginal.likelihood = c("none", "Chib95"),
  ...
)
}
\arguments{
\item{formula}{Model formula.}

\item{data}{Data frame.}

\item{b0}{The prior mean of \eqn{\beta}.  This can either be a scalar
or a column vector with dimension equal to the number of betas. If this
takes a scalar value, then that value will serve as the prior mean for all
of the betas.}

\item{B0}{The prior precision of \eqn{\beta}.  This can either be a
scalar or a square matrix with dimensions equal to the number of betas.  If
this takes a scalar value, then that value times an identity matrix serves
as the prior precision of \eqn{\beta}. Default value of 0 is
equivalent to an improper uniform prior for beta.}

\item{e}{The hyperprior for the distribution \eqn{\rho}. See details.}

\item{f}{The hyperprior for the distribution \eqn{\rho}. See details.}

\item{g}{The hyperprior for the distribution \eqn{\rho}. See details.}

\item{burnin}{The number of burn-in iterations for the sampler.}

\item{mcmc}{The number of Metropolis iterations for the sampler.}

\item{thin}{The thinning interval used in the simulation.  The number of
mcmc iterations must be divisible by this value.}

\item{verbose}{A switch which determines whether or not the progress of the
sampler is printed to the screen.  If \code{verbose} is greater than 0 the
iteration number, the current beta vector, and the Metropolis acceptance
rate are printed to the screen every \code{verbose}th iteration.}

\item{seed}{The seed for the random number generator.  If NA, the Mersenne
Twister generator is used with default seed 12345; if an integer is passed
it is used to seed the Mersenne twister.  The user can also pass a list of
length two to use the L'Ecuyer random number generator, which is suitable
for parallel computation.  The first element of the list is the L'Ecuyer
seed, which is a vector of length six or NA (if NA a default seed of
\code{rep(12345,6)} is used).  The second element of list is a positive
substream number. See the MCMCpack specification for more details.}

\item{beta.start}{The starting value for the \eqn{\beta} vector.  This
can either be a scalar or a column vector with dimension equal to the number
of betas. If this takes a scalar value, then that value will serve as the
starting value for all of the betas.  The default value of NA will use the
maximum likelihood estimate of \eqn{\beta} as the starting value.}

\item{rho.start}{The starting value for the \eqn{\rho} variable.
The default value is 1.}

\item{rho.step}{Tuning parameter for the slice sampling approach to
sampling \eqn{rho}. Determines the size of the step-out used to
find the correct slice to draw from. Lower values are more
accurate, but will take longer (up to a fixed searching limit).
Default is 0.1.}

\item{nu.start}{The starting values for the random effect,
\eqn{\nu}. The default value is a vector of ones.}

\item{marginal.likelihood}{How should the marginal likelihood be calculated?
Options are: \code{none} in which case the marginal likelihood will not be
calculated or \code{Laplace} in which case the Laplace approximation (see
Kass and Raftery, 1995) is used.}

\item{...}{further arguments to be passed.}
}
\value{
An mcmc object that contains the posterior sample.  This object can
be summarized by functions provided by the coda package.
}
\description{
This function generates a sample from the posterior distribution of a
Negative Binomial regression model via auxiliary mixture sampling. The user
supplies data and priors, and a sample from the posterior distribution is
returned as an mcmc object, which can be subsequently analyzed with
functions provided in the coda package.
}
\details{
\code{MCMCnegbin} simulates from the posterior distribution of a
Negative Binomial regression model using a combination of auxiliary
mixture sampling and slice sampling. The simulation proper is done
in compiled C++ code to maximize efficiency. Please consult the
coda documentation for a comprehensive list of functions that can
be used to analyze the posterior sample.

The model takes the following form:

\deqn{y_i \sim \mathcal{P}oisson(\nu_i\mu_i)}

Where the inverse link function:

\deqn{\mu_i = \exp(x_i'\beta)}

We assume a multivariate Normal prior on \eqn{\beta}:

\deqn{\beta \sim \mathcal{N}(b_0,B_0^{-1})}

The unit-level random effect that handles overdispersion is assumed
to be distributed Gamma:

\deqn{\nu_i \sim \mathcal{G}amma(\rho, \rho)}

The overdispersion parameter has a prior with the following form:

\deqn{f(\rho|e,f,g) \propto \rho^{e-1}(\rho + g)^{-(e+f)}}

The model is simulated via blocked Gibbs, with the the \eqn{\beta}
being simulated via the auxiliary mixture sampling method of
Fuerhwirth-Schanetter et al. (2009). The \eqn{\rho} is updated via
slice sampling. The \eqn{\nu_i} are updated their (conjugate) full
conditional, which is also Gamma.
}
\examples{

 \dontrun{
   n <- 150
   mcmcs <- 5000
   burnin <- 5000
   thin <- 5
   x1 <- runif(n, 0, 2)
   rho.true <- 1.5
   nu.true <- rgamma(n, rho.true, rho.true)
   mu <- nu.true * exp(1 + x1 * 1)
   y <- rpois(n, mu)
   posterior <- MCMCnegbin(y ~ x1)
   plot(posterior)
   summary(posterior)
   }

}
\references{
Andrew D. Martin, Kevin M. Quinn, and Jong Hee Park. 2011.
``MCMCpack: Markov Chain Monte Carlo in R.'', \emph{Journal of Statistical
Software}. 42(9): 1-21.  \doi{10.18637/jss.v042.i09}.

Daniel Pemstein, Kevin M. Quinn, and Andrew D. Martin.  2007.  \emph{Scythe
Statistical Library 1.0.} \url{http://scythe.wustl.edu.s3-website-us-east-1.amazonaws.com/}.

Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2006.  ``Output
Analysis and Diagnostics for MCMC (CODA)'', \emph{R News}. 6(1): 7-11.
\url{https://CRAN.R-project.org/doc/Rnews/Rnews_2006-1.pdf}.

Sylvia Fruehwirth-Schnatter, Rudolf Fruehwirth, Leonhard Held, and
    Havard Rue. 2009. ``Improved auxiliary mixture sampling for
    hierarchical models of non-Gaussian data'', \emph{Statistics
    and Computing} 19(4): 479-492.
    <doi:10.1007/s11222-008-9109-4>
}
\seealso{
\code{\link[coda]{plot.mcmc}},\code{\link[coda]{summary.mcmc}},
\code{\link[MASS]{glm.nb}}
}
\keyword{models}
