\name{Plotting TITAN results}
\alias{plotCNlogRByChr}
\alias{plotAllelicRatio}
\alias{plotClonalFrequency}
\alias{plotSubcloneProfiles}
\alias{plotSegmentMedians}
\alias{plotHaplotypeFraction}

\title{
  Plotting functions for \pkg{TitanCNA} results.
}
\description{
  Three plotting functions for \pkg{TitanCNA} results. \code{plotCNlogRByChr} plots the copy number results from log ratio data. \code{plotAllelicRatio} plots the allelic imbalance and loss of heterozygosity (LOH) from allelic ratio data. \code{plotClonalFrequency} plots the clonal cluster and cellular prevalence results for each data point.
}

\usage{
  plotAllelicRatio(dataIn, chr = c(1:22), geneAnnot = NULL, spacing = 4, 
      xlim = NULL, ...)
  plotClonalFrequency(dataIn, chr = c(1:22), normal = NULL, geneAnnot = NULL, 
      spacing = 4, xlim = NULL, ...)
  plotCNlogRByChr(dataIn, chr = c(1:22), segs = NULL, plotCorrectedCN = TRUE, 
  	  geneAnnot = NULL, ploidy = NULL, normal = NULL, spacing = 4, alphaVal = 1, xlim = NULL, ...)
  plotSubcloneProfiles(dataIn, chr = c(1:22), geneAnnot = NULL, 
  	  spacing = 4, xlim = NULL, ...)
  plotSegmentMedians(dataIn, resultType = "LogRatio", plotType = "CopyNumber", plotCorrectedCN = TRUE,
      chr = c(1:22), geneAnnot = NULL, ploidy = NULL, spacing = 4, alphaVal = 1, xlim = NULL, 
		  plot.new = FALSE, lwd = 8, ...)
  plotHaplotypeFraction(dataIn, chr = c(1:22), resultType = "HaplotypeRatio", colType = "Haplotypes",
      phaseBlockCol = c("#9ad0f3", "#CC79A7"), geneAnnot = NULL, spacing = 4,  xlim = NULL,  ...)
}

\arguments{
  \item{dataIn}{
    Formatted \pkg{TitanCNA} results output from \code{\link{outputTitanResults}}. See Example.
  }
  \item{chr}{
    Plot results for specified \code{chr}. The default is to plot chromosomes 1 to 22. The chromosome naming style will be automatically set to the input \code{dataIn}.
  }
  \item{segs}{
  	\code{\link[base:data.frame]{data.frame}} containing named columns: \code{Chromosome}, \code{Median_logR}, \code{Start_Position.bp.}, \code{End_Position.bp.}.  This data can be read in from the segments generated by the TITANRunner pipeline.  These segments will be overlaid in the plots as lines at the median log ratio for each segment.
  }
  \item{resultType}{
    For \code{plotSegmentMedians}: specify the data type (\sQuote{LogRatio} or \sQuote{AllelicRatio}) to plot.
    For \code{plotHaplotypeFraction}: specify the data type (\sQuote{HaplotypeRatio} or \sQuote{AllelicRatio}) to plot.
  }
  \item{plotType}{
    Specify whether to plot the \sQuote{CopyNumber} or \sQuote{Ratio} values for the \code{resultType}.
  }
  \item{colType}{
    Specify the color scheme to use: \sQuote{Haplotypes} or \sQuote{CopyNumber}. For \sQuote{Haplotypes}, alternating blue and red used to illustrated the data within phased haplotype blocks.  For \sQuote{CopyNumber}, the same colors as \code{plotAllelicRatio} are used for allelic copy number events.
  }
  \item{plotCorrectedCN}{
  	\code{TRUE} if the plot will use \sQuote{Corrected_Copy_Number} for color of data points or lines.
  }
  \item{geneAnnot}{
    \code{\link[base:data.frame]{data.frame}} specifying the genes to annotate in the plot. Gene boundaries are indicated using vertical dotted grey lines and gene symbols are shown at the top of the plot. \code{geneAnnot} must have four columns: gene symbol, chr, start coordinate, stop coordinate.
  }
  \item{normal}{
    \code{numeric} scalar indicating the normal contamination. This can be obtained from converge parameters output using \code{\link{runEMclonalCN}}. See Example.
  }
  \item{ploidy}{
    \code{numeric} scalar indicating the tumour ploidy used to adjust the copy number plot \code{\link{plotCNlogRByChr}}. This can be obtained from converge parameters output using \code{\link{runEMclonalCN}}. See Example. If \code{NULL} is used, then ploidy adjustment is not used in the plot.
  }
  \item{phaseBlockCol}{
  	Two-element vector specifying the color to plot for alternating haplotype phase blocks.
  }
  \item{spacing}{
    Number of lines of spacing for the margin spacing at the bottom of the plot. Useful if an idiogram/karogram is plot underneath.
  }
  \item{alphaVal}{
    Set an alpha value between 0 and 1 to allow transparency in the points being plot.
  }
  \item{xlim}{
  	Two element vector to specify the xlim for the plot. If \code{NULL}, then entire chromosome is plot.
  }
  \item{lwd}{
    Explicitly specify the line width for segments being plot.
  }
  \item{plot.new}{
    \code{TRUE} if a new plot is used.  Set to \code{FALSE} to overlay an existing plot. 
  }
  \item{...}{
    Additional arguments used in the \code{\link[graphics:plot]{plot}} function.  
  }
  
}

\details{
  \code{\link{plotCNlogRByChr}} plots the copy number alterations from log ratio data. The Y-axis is based on log ratios. Log ratios are computed ratios between normalized tumour and normal read depths. Data points close to 0 represent diploid, above 0 are copy gains, below 0 are deletions. \code{ploidy} argument adjusts the baseline of the data points. Colours represent the copy number state.
  Bright Green - Homozygous deletion (HOMD)
  Green - Hemizygous deletion (DLOH)
  Blue - Diploid heterozygous (HET), Copy-neutral LOH (NLOH)
  Dark Red - GAIN
  Red - Allele-specific CNA (ASCNA), Unbalanced CNA (UBCNA), Balanced CNA (BCNA)
  
  \code{\link{plotAllelicRatio}} plots the allelic imbalance and loss of heterozygosity from allelic ratio data. The Y-axis is based on allelic ratios. Allelic ratios are computed as RefCount/Depth. Data points close to 1 represent homozygous reference base, close to 0 represent homozygous non-reference base, and close to 0.5 represent heterozygous. Normal contamination influences the divergence away from 0.5 for LOH events. No adjustments are made to the plot as the original data from \code{dataIn} are shown. Colours represent the allelic imbalance and LOH state.
  Grey - HET, BCNA
  Bright Green - HOMD
  Green - DLOH, ALOH
  Blue - NLOH
  Dark Red - GAIN
  Red - ASCNA, UBCNA
  
  \code{\link{plotClonalFrequency}} plots the cellular prevalence and clonal clusters from the results. The Y-axis is the cellular prevalence that includes the normal proportion. Therefore, the cellular prevalence here refers to the proportion in the sample (including normal). Lines are drawn for each data point indicating the cellular prevalence. Heterozygous diploid are not shown because it is a normal genotype and is not categorized as being subclonal (this means 100\% of cells are normal). The black horizontal line represents the tumour content labeled as \sQuote{T}. Each horizontal grey line represents the cellular prevalence of the clonal clusters labeled as Z1, Z2, etc. Colours are the sames for allelic ratio plots.
  
  \code{\link{plotSubcloneProfiles}} plots the predicted copy number profile for individual subclones inferred by TITAN. Currently, this only works for solutions having 1 or 2 clonal clusters. The colours are the same as for \code{\link{plotAllelicRatio}}.
  
  \code{\link{plotSegmentMedians}} plots the segment means for \sQuote{LogRatio} or \sQuote{AllelicRatio} data.  There are also two types of plots for each of the datatypes: \sQuote{Ratio} or \sQuote{CopyNumber}.  For \sQuote{Ratio}, the logRatio or allelic ratios in the output results files are plot.  For \sQuote{CopyNumber}, the y-axis is converted to the exponentiated absolute copy number levels for the easier interpretability of the results.
  
  \code{\link{plotHaplotypeFraction}} plots the phased SNP read count of the higher coverage haplotype, normalized by the  total coverage of the haplotype. For \sQuote{Haplotypes}, alternating colors of blue and red are used to illustrate the phased haplotype blocks provided from the input data (see \code{\link{loadHaplotypeAlleleCounts}}).
}

\references{
 Ha, G., Roth, A., Khattra, J., Ho, J., Yap, D., Prentice, L. M., Melnyk, N., McPherson, A., Bashashati, A., Laks, E., Biele, J., Ding, J., Le, A., Rosner, J., Shumansky, K., Marra, M. A., Huntsman, D. G., McAlpine, J. N., Aparicio, S. A. J. R., and Shah, S. P. (2014). TITAN: Inference of copy number architectures in clonal cell populations from tumour whole genome sequence data. Genome Research, 24: 1881-1893. (PMID: 25060187)
}

\author{
  Gavin Ha <gavinha@gmail.com>
}

\seealso{
  \code{\link{outputTitanResults}}, \code{\link{runEMclonalCN}}, \code{\link{computeSDbwIndex}}
}

\examples{
data(EMresults)

#### COMPUTE OPTIMAL STATE PATH USING VITERBI ####
optimalPath <- viterbiClonalCN(data, convergeParams)

#### FORMAT RESULTS ####
results <- outputTitanResults(data, convergeParams, optimalPath, 
  filename = NULL, posteriorProbs = FALSE, 
  correctResults = TRUE, proportionThreshold = 0.05, 
  proportionThresholdClonal = 0.05)
convergeParams <- results$convergeParams
results <- results$corrResults # use corrected results
#### PLOT RESULTS ####
norm <- tail(convergeParams$n, 1)
ploidy <- tail(convergeParams$phi, 1)

par(mfrow=c(4, 1))    
plotCNlogRByChr(results, chr = 2, segs = NULL, ploidy = ploidy, normal = norm, 
  geneAnnot = NULL, ylim = c(-2, 2), cex = 0.5, xlab = "", 
  main = "Chr 2")
plotAllelicRatio(results, chr = 2, geneAnnot = NULL, ylim = c(0, 1), cex = 0.5, 
  xlab = "", main = "Chr 2")
plotClonalFrequency(results, chr = 2, normal = norm, geneAnnot = NULL, 
  ylim = c(0, 1), cex = 0.5, xlab = "", main = "Chr 2")
plotSubcloneProfiles(results, chr = 2, cex = 2, main = "Chr 2")

segs <- outputTitanSegments(results, id = "test", convergeParams)

plotSegmentMedians(segs, chr=2, resultType = "LogRatio", 
  plotType = "CopyNumber", plot.new = TRUE)

}

\keyword{aplot}
\keyword{color}
