/*========================== begin_copyright_notice ============================

Copyright (C) 2024 Intel Corporation

SPDX-License-Identifier: MIT

============================= end_copyright_notice ===========================*/
/*
// ALGORITHM DESCRIPTION:
//
//      HIGH LEVEL OVERVIEW
//      Here we use polynomial and reciprocal calculation for 32 subintervals
//      at reduction interval.
//
//      For large arguments ( |a[i]| >= LARGE_ARG_HIBITS,
//      ( where LARGE_ARG_HIBITS = 16 high bits of 12800.0 value )
//      the Payne/Hanek "pre-reduction" performed. Result of this routine
//      becomes argument for regular reduction.
//
//      The regular range reduction scheme is:
//
//          a[i] = N * pi/2^R + z
//          where R = 5 for this implementation - reduction value (2^R = 32),
//          and |z| <= pi/(2*2^R)) - this is reduced argument.
//
//      Also alternative reduction performed in parallel:
//
//          a[i] = NN * pi/2 + zz,
//          (NN = N mod 16)
//
//      The reason is getting remainder modulo Pi/2. The value zz is used
//      for reciprocal calculation.
//
//      Futher tan calculation performed by this way:
//
//         r[i] := TAU * 1/zz + C0 + C1*r + C1*c + C2*r^2 + ... + C15*r^15
//
//         (TAU - multiplier for the reciprocal 1/zz,
//         and always -1 or 0 depending on subinterval)
//
//      For tiny arguments ( |a[i]| < 2^TINY_ARG_EXP )
//      the simple separate branch used:
//
//          r[i] = a[i]
//
//      IEEE SPECIAL CONDITIONS:
//      a[i] = +/-Inf, r[i] = QNaN
//      a[i] = +/-0,   r[i] = +/-0
//      a[i] = QNaN,   r[i] = QNaN
//      a[i] = SNaN,   r[i] = QNaN
//
//
//      ALGORITHM DETAILS
//      Executable parts:
//
//      1) a[i]  = +/-Inf
//         Return r[i] := a[i]*0.0
//         and error handler called with IML_STATUS_ERRDOM error code
//
//      2) a[i]  = NaN
//         Return r[i] := a[i] * a[i]
//
//      3) Tiny arguments path
//         |a[i]| < 2^TINY_ARG_EXP,
//         where TINY_ARG_EXP = -252
//
//         3.1) a[i]  = 0.0
//              Return r[i] := a[i]
//
//         3.2) 0 < |a[i]| < 2^TINY_ARG_EXP
//              Return r[i] := TWOp55 * ( TWOpM55*a[i] - a[i] ),
//              where TWOp55 = 2^55, TWOpM55 = 2^-55
//
//              Here is path where underflow or denormal exceptions can happen
//              during intermediate computations.
//              For correct work in all rounding modes we need to
//              return a[i] - TWOpM55 * a[i]
//              To avoid disappearing of second term we using scaling
//              like this TWOp55 * ( TWOpM55*a[i] - a[i] )
//
//
//      4) Main path (the most frequently used and the most wide)
//         2^TINY_ARG_EXP <= |a[i]| < LARGE_ARG_HIBITS
//
//         a) Pre-reduction.
//
//            For large arguments |a[i]| >= LARGE_ARG_HIBITS
//            special argument pre-range-reduction routine is called:
//            NR := _vml_reduce_pio2d( a[i], rr ),
//            where NR - number of octants of pre-reduction,
//            rr[0], rr[1] - high and low parts of pre-reduced argument.
//            The Payne/Hanek algorithm is used there (not described).
//            Assign   x := rr[0]
//            In case of no pre-reduction   x := a[i]
//
//         b) Main reduction.
//
//            The standard range reduction scheme is
//            zc := x - N * (PIo32_HI + PIo32_LO + PIo32_TAIL)
//            zc - reduced argument
//
//            Integer N obtained by famous "right-shifter" technique -
//            add and subtract RS = 2^52+2^51 value.
//
//            After that we add N := N + NR*(2^(R-1))
//            if large arguments pre-reduction
//            routine called.  NR = result octant number of Pi/2
//            pre-reduction.
//            For a[i] < LARGE_ARG_HIBITS the NR = 0 and N is unchanged.
//
//            PIo32_HI and PIo32_LO are 32-bit numbers (so multiplication
//            by N is exact) and PIo32_TAIL is a 53-bit number. Together, these
//            approximate pi well enough for all cases in this restricted
//            range. Reduction performed in accurate way with low part (c) of
//            result correct processing.
//            For large arguments added c = c + rr[1].
//            Finally we have zc = z + c multiprecision value.
//
//            In parallel we are doing another reduction for
//            getting remainder modulo Pi/2.  Here we perform
//            a sort of "more rounding".
//            It means have the same computation sequences but using N = (N mod
16)
//            that is also obtained by "right shifter" technique,
//            where right shifter value is (2^55+2^56) instead of usual
(2^51+2^52)
//            Pi values presented by 38+38+53 form for accurate multiplication
by
//            14-bit of (N mod 16).
//            The result is zzc = zz+cc multiprecision value.
//
//            For existing large arguments reduction we need to add
//            extra low part rr[1] to c and cc correction terms.
//            c := c + rr[1],  cc := cc + rr[1]
//            but it is necessary to resplit z+c and zz + cc values
//            to preserve proportions betwee high and low parts.
//            Doing it this way:
//
//               v1 := z + c;   v2 := z - v1;   c := v2 + c;   z := v1;
//               v1 := zz + cc; v2 := zz - v1;  cc := v2 + cc; zz := v1;
//
//         c) General computations.
//
//            The whole computation range (Pi/2) is splitted to
//            32 even ranges and for each breakpoint we have
//            unique set of coefficients stored as table.
//            The table lookup performed by index that is 5 least
//            significant bits of integer N (octant number) value.
//
//            The constants are:
//            1) C2 ... C15 polynomial coefficients for r^2 ... r^15
//            2) C0_HI + C0_LO - accurate constant C0 term in power series
//            3) C1_HI + C1_LO - accurate coefficient C1 of r in power series
//            4) TAU - multiplier for the reciprocal, always -1 or 0
//            5) MSK - 35 significant bit mask for the reciprocal
//
//
//            The basic reconstruction formula using these constants is:
//
//               High := TAU*recip_hi + C0_HI
//               Med + Low := C1_HI*r + C1_LO*r (accurate sum)
//               Low  := Low + TAU*recip_lo + C0_LO + (C1_LO+C1_HI)*c + pol,
//                 where pol := C2*r^2 + ... + C15*r^15
//
//            The recip_hi + recip_lo is an accurate reciprocal of the remainder
//            modulo pi/2 = 1/zz
//            Finally we doing a compensated sum High + Med + Low:
//
//            Return r[i] := (High + (Med + Low))
// --
//
*/
#include "../imf.h"
#pragma OPENCL FP_CONTRACT OFF
typedef struct {
  unsigned long _dPtable[2048][3];
} __dtan_ha_ReductionTab_t;
static __dtan_ha_ReductionTab_t __ocl_svml_internal_dtan_ha_reduction_data = {
    {
        /*            P_hi                          P_med P_lo         */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 0 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 1 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 2 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 3 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 4 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 5 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 6 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 7 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 8 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 9 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 10 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 11 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 12 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 13 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 14 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 15 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 16 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 17 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 18 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 19 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 20 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 21 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 22 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 23 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 24 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 25 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 26 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 27 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 28 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 29 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 30 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 31 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 32 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 33 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 34 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 35 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 36 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 37 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 38 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 39 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 40 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 41 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 42 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 43 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 44 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 45 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 46 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 47 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 48 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 49 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 50 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 51 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 52 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 53 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 54 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 55 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 56 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 57 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 58 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 59 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 60 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 61 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 62 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 63 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 64 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 65 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 66 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 67 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 68 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 69 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 70 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 71 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 72 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 73 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 74 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 75 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 76 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 77 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 78 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 79 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 80 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 81 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 82 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 83 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 84 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 85 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 86 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 87 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 88 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 89 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 90 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 91 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 92 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 93 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 94 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 95 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 96 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 97 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 98 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 99 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 100 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 101 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 102 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 103 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 104 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 105 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 106 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 107 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 108 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 109 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 110 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 111 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 112 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 113 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 114 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 115 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 116 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 117 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 118 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 119 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 120 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 121 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 122 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 123 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 124 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 125 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 126 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 127 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 128 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 129 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 130 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 131 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 132 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 133 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 134 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 135 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 136 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 137 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 138 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 139 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 140 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 141 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 142 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 143 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 144 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 145 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 146 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 147 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 148 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 149 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 150 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 151 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 152 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 153 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 154 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 155 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 156 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 157 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 158 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 159 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 160 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 161 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 162 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 163 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 164 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 165 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 166 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 167 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 168 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 169 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 170 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 171 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 172 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 173 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 174 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 175 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 176 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 177 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 178 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 179 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 180 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 181 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 182 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 183 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 184 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 185 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 186 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 187 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 188 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 189 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 190 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 191 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 192 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 193 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 194 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 195 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 196 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 197 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 198 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 199 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 200 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 201 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 202 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 203 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 204 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 205 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 206 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 207 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 208 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 209 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 210 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 211 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 212 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 213 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 214 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 215 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 216 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 217 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 218 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 219 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 220 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 221 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 222 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 223 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 224 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 225 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 226 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 227 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 228 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 229 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 230 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 231 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 232 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 233 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 234 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 235 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 236 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 237 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 238 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 239 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 240 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 241 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 242 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 243 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 244 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 245 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 246 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 247 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 248 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 249 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 250 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 251 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 252 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 253 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 254 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 255 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 256 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 257 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 258 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 259 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 260 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 261 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 262 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 263 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 264 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 265 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 266 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 267 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 268 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 269 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 270 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 271 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 272 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 273 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 274 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 275 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 276 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 277 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 278 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 279 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 280 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 281 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 282 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 283 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 284 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 285 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 286 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 287 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 288 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 289 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 290 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 291 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 292 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 293 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 294 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 295 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 296 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 297 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 298 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 299 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 300 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 301 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 302 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 303 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 304 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 305 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 306 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 307 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 308 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 309 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 310 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 311 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 312 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 313 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 314 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 315 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 316 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 317 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 318 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 319 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 320 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 321 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 322 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 323 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 324 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 325 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 326 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 327 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 328 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 329 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 330 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 331 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 332 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 333 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 334 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 335 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 336 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 337 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 338 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 339 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 340 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 341 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 342 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 343 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 344 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 345 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 346 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 347 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 348 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 349 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 350 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 351 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 352 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 353 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 354 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 355 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 356 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 357 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 358 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 359 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 360 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 361 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 362 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 363 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 364 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 365 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 366 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 367 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 368 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 369 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 370 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 371 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 372 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 373 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 374 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 375 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 376 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 377 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 378 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 379 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 380 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 381 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 382 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 383 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 384 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 385 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 386 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 387 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 388 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 389 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 390 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 391 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 392 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 393 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 394 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 395 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 396 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 397 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 398 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 399 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 400 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 401 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 402 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 403 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 404 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 405 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 406 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 407 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 408 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 409 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 410 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 411 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 412 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 413 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 414 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 415 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 416 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 417 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 418 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 419 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 420 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 421 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 422 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 423 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 424 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 425 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 426 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 427 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 428 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 429 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 430 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 431 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 432 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 433 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 434 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 435 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 436 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 437 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 438 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 439 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 440 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 441 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 442 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 443 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 444 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 445 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 446 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 447 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 448 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 449 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 450 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 451 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 452 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 453 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 454 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 455 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 456 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 457 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 458 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 459 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 460 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 461 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 462 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 463 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 464 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 465 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 466 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 467 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 468 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 469 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 470 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 471 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 472 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 473 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 474 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 475 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 476 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 477 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 478 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 479 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 480 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 481 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 482 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 483 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 484 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 485 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 486 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 487 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 488 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 489 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 490 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 491 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 492 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 493 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 494 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 495 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 496 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 497 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 498 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 499 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 500 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 501 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 502 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 503 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 504 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 505 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 506 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 507 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 508 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 509 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 510 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 511 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 512 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 513 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 514 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 515 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 516 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 517 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 518 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 519 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 520 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 521 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 522 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 523 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 524 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 525 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 526 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 527 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 528 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 529 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 530 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 531 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 532 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 533 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 534 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 535 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 536 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 537 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 538 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 539 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 540 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 541 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 542 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 543 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 544 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 545 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 546 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 547 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 548 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 549 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 550 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 551 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 552 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 553 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 554 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 555 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 556 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 557 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 558 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 559 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 560 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 561 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 562 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 563 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 564 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 565 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 566 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 567 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 568 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 569 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 570 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 571 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 572 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 573 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 574 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 575 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 576 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 577 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 578 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 579 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 580 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 581 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 582 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 583 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 584 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 585 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 586 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 587 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 588 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 589 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 590 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 591 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 592 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 593 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 594 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 595 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 596 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 597 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 598 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 599 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 600 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 601 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 602 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 603 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 604 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 605 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 606 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 607 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 608 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 609 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 610 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 611 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 612 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 613 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 614 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 615 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 616 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 617 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 618 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 619 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 620 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 621 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 622 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 623 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 624 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 625 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 626 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 627 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 628 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 629 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 630 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 631 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 632 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 633 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 634 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 635 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 636 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 637 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 638 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 639 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 640 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 641 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 642 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 643 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 644 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 645 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 646 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 647 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 648 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 649 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 650 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 651 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 652 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 653 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 654 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 655 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 656 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 657 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 658 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 659 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 660 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 661 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 662 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 663 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 664 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 665 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 666 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 667 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 668 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 669 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 670 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 671 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 672 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 673 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 674 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 675 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 676 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 677 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 678 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 679 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 680 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 681 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 682 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 683 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 684 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 685 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 686 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 687 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 688 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 689 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 690 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 691 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 692 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 693 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 694 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 695 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 696 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 697 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 698 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 699 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 700 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 701 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 702 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 703 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 704 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 705 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 706 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 707 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 708 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 709 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 710 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 711 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 712 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 713 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 714 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 715 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 716 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 717 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 718 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 719 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 720 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 721 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 722 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 723 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 724 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 725 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 726 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 727 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 728 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 729 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 730 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 731 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 732 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 733 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 734 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 735 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 736 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 737 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 738 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 739 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 740 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 741 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 742 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 743 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 744 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 745 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 746 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 747 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 748 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 749 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 750 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 751 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 752 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 753 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 754 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 755 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 756 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 757 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 758 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 759 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 760 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 761 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 762 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 763 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 764 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 765 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 766 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 767 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 768 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 769 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 770 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 771 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 772 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 773 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 774 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 775 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 776 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 777 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 778 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 779 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 780 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 781 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 782 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 783 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 784 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 785 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 786 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 787 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 788 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 789 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 790 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 791 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 792 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 793 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 794 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 795 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 796 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 797 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 798 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 799 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 800 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 801 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 802 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 803 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 804 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 805 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 806 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 807 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 808 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 809 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 810 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 811 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 812 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 813 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 814 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 815 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 816 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 817 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 818 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 819 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 820 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 821 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 822 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 823 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 824 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 825 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 826 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 827 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 828 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 829 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 830 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 831 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 832 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 833 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 834 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 835 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 836 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 837 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 838 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 839 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 840 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 841 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 842 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 843 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 844 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 845 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 846 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 847 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 848 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 849 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 850 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 851 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 852 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 853 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 854 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 855 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 856 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 857 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 858 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 859 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 860 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 861 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 862 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 863 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 864 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 865 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 866 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 867 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 868 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 869 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 870 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 871 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 872 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 873 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 874 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 875 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 876 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 877 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 878 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 879 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 880 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 881 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 882 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 883 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 884 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000000uL}, /* 885 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000001uL}, /* 886 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000002uL}, /* 887 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000005uL}, /* 888 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000000000000000AuL}, /* 889 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000014uL}, /* 890 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000028uL}, /* 891 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000051uL}, /* 892 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00000000000000A2uL}, /* 893 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000145uL}, /* 894 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000000000000028BuL}, /* 895 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000517uL}, /* 896 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000000A2FuL}, /* 897 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000000000000145FuL}, /* 898 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00000000000028BEuL}, /* 899 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000000000000517CuL}, /* 900 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000000000000A2F9uL}, /* 901 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00000000000145F3uL}, /* 902 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000028BE6uL}, /* 903 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00000000000517CCuL}, /* 904 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00000000000A2F98uL}, /* 905 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000145F30uL}, /* 906 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000000000028BE60uL}, /* 907 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000517CC1uL}, /* 908 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000000A2F983uL}, /* 909 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000000000145F306uL}, /* 910 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00000000028BE60DuL}, /* 911 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000000000517CC1BuL}, /* 912 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000000000A2F9836uL}, /* 913 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00000000145F306DuL}, /* 914 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000028BE60DBuL}, /* 915 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00000000517CC1B7uL}, /* 916 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00000000A2F9836EuL}, /* 917 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000145F306DCuL}, /* 918 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000000028BE60DB9uL}, /* 919 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000517CC1B72uL}, /* 920 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000000A2F9836E4uL}, /* 921 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000000145F306DC9uL}, /* 922 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00000028BE60DB93uL}, /* 923 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000000517CC1B727uL}, /* 924 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000000A2F9836E4EuL}, /* 925 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00000145F306DC9CuL}, /* 926 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000028BE60DB939uL}, /* 927 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00000517CC1B7272uL}, /* 928 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00000A2F9836E4E4uL}, /* 929 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000145F306DC9C8uL}, /* 930 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000028BE60DB9391uL}, /* 931 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000517CC1B72722uL}, /* 932 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0000A2F9836E4E44uL}, /* 933 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000145F306DC9C88uL}, /* 934 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00028BE60DB93910uL}, /* 935 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000517CC1B727220uL}, /* 936 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x000A2F9836E4E441uL}, /* 937 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00145F306DC9C882uL}, /* 938 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0028BE60DB939105uL}, /* 939 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00517CC1B727220AuL}, /* 940 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x00A2F9836E4E4415uL}, /* 941 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0145F306DC9C882AuL}, /* 942 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x028BE60DB9391054uL}, /* 943 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0517CC1B727220A9uL}, /* 944 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x0A2F9836E4E44152uL}, /* 945 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x145F306DC9C882A5uL}, /* 946 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x28BE60DB9391054AuL}, /* 947 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0x517CC1B727220A94uL}, /* 948 */
        {0x0000000000000000uL, 0x0000000000000000uL,
         0xA2F9836E4E441529uL}, /* 949 */
        {0x0000000000000000uL, 0x0000000000000001uL,
         0x45F306DC9C882A53uL}, /* 950 */
        {0x0000000000000000uL, 0x0000000000000002uL,
         0x8BE60DB9391054A7uL}, /* 951 */
        {0x0000000000000000uL, 0x0000000000000005uL,
         0x17CC1B727220A94FuL}, /* 952 */
        {0x0000000000000000uL, 0x000000000000000AuL,
         0x2F9836E4E441529FuL}, /* 953 */
        {0x0000000000000000uL, 0x0000000000000014uL,
         0x5F306DC9C882A53FuL}, /* 954 */
        {0x0000000000000000uL, 0x0000000000000028uL,
         0xBE60DB9391054A7FuL}, /* 955 */
        {0x0000000000000000uL, 0x0000000000000051uL,
         0x7CC1B727220A94FEuL}, /* 956 */
        {0x0000000000000000uL, 0x00000000000000A2uL,
         0xF9836E4E441529FCuL}, /* 957 */
        {0x0000000000000000uL, 0x0000000000000145uL,
         0xF306DC9C882A53F8uL}, /* 958 */
        {0x0000000000000000uL, 0x000000000000028BuL,
         0xE60DB9391054A7F0uL}, /* 959 */
        {0x0000000000000000uL, 0x0000000000000517uL,
         0xCC1B727220A94FE1uL}, /* 960 */
        {0x0000000000000000uL, 0x0000000000000A2FuL,
         0x9836E4E441529FC2uL}, /* 961 */
        {0x0000000000000000uL, 0x000000000000145FuL,
         0x306DC9C882A53F84uL}, /* 962 */
        {0x0000000000000000uL, 0x00000000000028BEuL,
         0x60DB9391054A7F09uL}, /* 963 */
        {0x0000000000000000uL, 0x000000000000517CuL,
         0xC1B727220A94FE13uL}, /* 964 */
        {0x0000000000000000uL, 0x000000000000A2F9uL,
         0x836E4E441529FC27uL}, /* 965 */
        {0x0000000000000000uL, 0x00000000000145F3uL,
         0x06DC9C882A53F84EuL}, /* 966 */
        {0x0000000000000000uL, 0x0000000000028BE6uL,
         0x0DB9391054A7F09DuL}, /* 967 */
        {0x0000000000000000uL, 0x00000000000517CCuL,
         0x1B727220A94FE13AuL}, /* 968 */
        {0x0000000000000000uL, 0x00000000000A2F98uL,
         0x36E4E441529FC275uL}, /* 969 */
        {0x0000000000000000uL, 0x0000000000145F30uL,
         0x6DC9C882A53F84EAuL}, /* 970 */
        {0x0000000000000000uL, 0x000000000028BE60uL,
         0xDB9391054A7F09D5uL}, /* 971 */
        {0x0000000000000000uL, 0x0000000000517CC1uL,
         0xB727220A94FE13ABuL}, /* 972 */
        {0x0000000000000000uL, 0x0000000000A2F983uL,
         0x6E4E441529FC2757uL}, /* 973 */
        {0x0000000000000000uL, 0x000000000145F306uL,
         0xDC9C882A53F84EAFuL}, /* 974 */
        {0x0000000000000000uL, 0x00000000028BE60DuL,
         0xB9391054A7F09D5FuL}, /* 975 */
        {0x0000000000000000uL, 0x000000000517CC1BuL,
         0x727220A94FE13ABEuL}, /* 976 */
        {0x0000000000000000uL, 0x000000000A2F9836uL,
         0xE4E441529FC2757DuL}, /* 977 */
        {0x0000000000000000uL, 0x00000000145F306DuL,
         0xC9C882A53F84EAFAuL}, /* 978 */
        {0x0000000000000000uL, 0x0000000028BE60DBuL,
         0x9391054A7F09D5F4uL}, /* 979 */
        {0x0000000000000000uL, 0x00000000517CC1B7uL,
         0x27220A94FE13ABE8uL}, /* 980 */
        {0x0000000000000000uL, 0x00000000A2F9836EuL,
         0x4E441529FC2757D1uL}, /* 981 */
        {0x0000000000000000uL, 0x0000000145F306DCuL,
         0x9C882A53F84EAFA3uL}, /* 982 */
        {0x0000000000000000uL, 0x000000028BE60DB9uL,
         0x391054A7F09D5F47uL}, /* 983 */
        {0x0000000000000000uL, 0x0000000517CC1B72uL,
         0x7220A94FE13ABE8FuL}, /* 984 */
        {0x0000000000000000uL, 0x0000000A2F9836E4uL,
         0xE441529FC2757D1FuL}, /* 985 */
        {0x0000000000000000uL, 0x000000145F306DC9uL,
         0xC882A53F84EAFA3EuL}, /* 986 */
        {0x0000000000000000uL, 0x00000028BE60DB93uL,
         0x91054A7F09D5F47DuL}, /* 987 */
        {0x0000000000000000uL, 0x000000517CC1B727uL,
         0x220A94FE13ABE8FAuL}, /* 988 */
        {0x0000000000000000uL, 0x000000A2F9836E4EuL,
         0x441529FC2757D1F5uL}, /* 989 */
        {0x0000000000000000uL, 0x00000145F306DC9CuL,
         0x882A53F84EAFA3EAuL}, /* 990 */
        {0x0000000000000000uL, 0x0000028BE60DB939uL,
         0x1054A7F09D5F47D4uL}, /* 991 */
        {0x0000000000000000uL, 0x00000517CC1B7272uL,
         0x20A94FE13ABE8FA9uL}, /* 992 */
        {0x0000000000000000uL, 0x00000A2F9836E4E4uL,
         0x41529FC2757D1F53uL}, /* 993 */
        {0x0000000000000000uL, 0x0000145F306DC9C8uL,
         0x82A53F84EAFA3EA6uL}, /* 994 */
        {0x0000000000000000uL, 0x000028BE60DB9391uL,
         0x054A7F09D5F47D4DuL}, /* 995 */
        {0x0000000000000000uL, 0x0000517CC1B72722uL,
         0x0A94FE13ABE8FA9AuL}, /* 996 */
        {0x0000000000000000uL, 0x0000A2F9836E4E44uL,
         0x1529FC2757D1F534uL}, /* 997 */
        {0x0000000000000000uL, 0x000145F306DC9C88uL,
         0x2A53F84EAFA3EA69uL}, /* 998 */
        {0x0000000000000000uL, 0x00028BE60DB93910uL,
         0x54A7F09D5F47D4D3uL}, /* 999 */
        {0x0000000000000000uL, 0x000517CC1B727220uL,
         0xA94FE13ABE8FA9A6uL}, /* 1000 */
        {0x0000000000000000uL, 0x000A2F9836E4E441uL,
         0x529FC2757D1F534DuL}, /* 1001 */
        {0x0000000000000000uL, 0x00145F306DC9C882uL,
         0xA53F84EAFA3EA69BuL}, /* 1002 */
        {0x0000000000000000uL, 0x0028BE60DB939105uL,
         0x4A7F09D5F47D4D37uL}, /* 1003 */
        {0x0000000000000000uL, 0x00517CC1B727220AuL,
         0x94FE13ABE8FA9A6EuL}, /* 1004 */
        {0x0000000000000000uL, 0x00A2F9836E4E4415uL,
         0x29FC2757D1F534DDuL}, /* 1005 */
        {0x0000000000000000uL, 0x0145F306DC9C882AuL,
         0x53F84EAFA3EA69BBuL}, /* 1006 */
        {0x0000000000000000uL, 0x028BE60DB9391054uL,
         0xA7F09D5F47D4D377uL}, /* 1007 */
        {0x0000000000000000uL, 0x0517CC1B727220A9uL,
         0x4FE13ABE8FA9A6EEuL}, /* 1008 */
        {0x0000000000000000uL, 0x0A2F9836E4E44152uL,
         0x9FC2757D1F534DDCuL}, /* 1009 */
        {0x0000000000000000uL, 0x145F306DC9C882A5uL,
         0x3F84EAFA3EA69BB8uL}, /* 1010 */
        {0x0000000000000000uL, 0x28BE60DB9391054AuL,
         0x7F09D5F47D4D3770uL}, /* 1011 */
        {0x0000000000000000uL, 0x517CC1B727220A94uL,
         0xFE13ABE8FA9A6EE0uL}, /* 1012 */
        {0x0000000000000000uL, 0xA2F9836E4E441529uL,
         0xFC2757D1F534DDC0uL}, /* 1013 */
        {0x0000000000000001uL, 0x45F306DC9C882A53uL,
         0xF84EAFA3EA69BB81uL}, /* 1014 */
        {0x0000000000000002uL, 0x8BE60DB9391054A7uL,
         0xF09D5F47D4D37703uL}, /* 1015 */
        {0x0000000000000005uL, 0x17CC1B727220A94FuL,
         0xE13ABE8FA9A6EE06uL}, /* 1016 */
        {0x000000000000000AuL, 0x2F9836E4E441529FuL,
         0xC2757D1F534DDC0DuL}, /* 1017 */
        {0x0000000000000014uL, 0x5F306DC9C882A53FuL,
         0x84EAFA3EA69BB81BuL}, /* 1018 */
        {0x0000000000000028uL, 0xBE60DB9391054A7FuL,
         0x09D5F47D4D377036uL}, /* 1019 */
        {0x0000000000000051uL, 0x7CC1B727220A94FEuL,
         0x13ABE8FA9A6EE06DuL}, /* 1020 */
        {0x00000000000000A2uL, 0xF9836E4E441529FCuL,
         0x2757D1F534DDC0DBuL}, /* 1021 */
        {0x0000000000000145uL, 0xF306DC9C882A53F8uL,
         0x4EAFA3EA69BB81B6uL}, /* 1022 */
        {0x000000000000028BuL, 0xE60DB9391054A7F0uL,
         0x9D5F47D4D377036DuL}, /* 1023 */
        {0x0000000000000517uL, 0xCC1B727220A94FE1uL,
         0x3ABE8FA9A6EE06DBuL}, /* 1024 */
        {0x0000000000000A2FuL, 0x9836E4E441529FC2uL,
         0x757D1F534DDC0DB6uL}, /* 1025 */
        {0x000000000000145FuL, 0x306DC9C882A53F84uL,
         0xEAFA3EA69BB81B6CuL}, /* 1026 */
        {0x00000000000028BEuL, 0x60DB9391054A7F09uL,
         0xD5F47D4D377036D8uL}, /* 1027 */
        {0x000000000000517CuL, 0xC1B727220A94FE13uL,
         0xABE8FA9A6EE06DB1uL}, /* 1028 */
        {0x000000000000A2F9uL, 0x836E4E441529FC27uL,
         0x57D1F534DDC0DB62uL}, /* 1029 */
        {0x00000000000145F3uL, 0x06DC9C882A53F84EuL,
         0xAFA3EA69BB81B6C5uL}, /* 1030 */
        {0x0000000000028BE6uL, 0x0DB9391054A7F09DuL,
         0x5F47D4D377036D8AuL}, /* 1031 */
        {0x00000000000517CCuL, 0x1B727220A94FE13AuL,
         0xBE8FA9A6EE06DB14uL}, /* 1032 */
        {0x00000000000A2F98uL, 0x36E4E441529FC275uL,
         0x7D1F534DDC0DB629uL}, /* 1033 */
        {0x0000000000145F30uL, 0x6DC9C882A53F84EAuL,
         0xFA3EA69BB81B6C52uL}, /* 1034 */
        {0x000000000028BE60uL, 0xDB9391054A7F09D5uL,
         0xF47D4D377036D8A5uL}, /* 1035 */
        {0x0000000000517CC1uL, 0xB727220A94FE13ABuL,
         0xE8FA9A6EE06DB14AuL}, /* 1036 */
        {0x0000000000A2F983uL, 0x6E4E441529FC2757uL,
         0xD1F534DDC0DB6295uL}, /* 1037 */
        {0x000000000145F306uL, 0xDC9C882A53F84EAFuL,
         0xA3EA69BB81B6C52BuL}, /* 1038 */
        {0x00000000028BE60DuL, 0xB9391054A7F09D5FuL,
         0x47D4D377036D8A56uL}, /* 1039 */
        {0x000000000517CC1BuL, 0x727220A94FE13ABEuL,
         0x8FA9A6EE06DB14ACuL}, /* 1040 */
        {0x000000000A2F9836uL, 0xE4E441529FC2757DuL,
         0x1F534DDC0DB62959uL}, /* 1041 */
        {0x00000000145F306DuL, 0xC9C882A53F84EAFAuL,
         0x3EA69BB81B6C52B3uL}, /* 1042 */
        {0x0000000028BE60DBuL, 0x9391054A7F09D5F4uL,
         0x7D4D377036D8A566uL}, /* 1043 */
        {0x00000000517CC1B7uL, 0x27220A94FE13ABE8uL,
         0xFA9A6EE06DB14ACCuL}, /* 1044 */
        {0x00000000A2F9836EuL, 0x4E441529FC2757D1uL,
         0xF534DDC0DB629599uL}, /* 1045 */
        {0x0000000145F306DCuL, 0x9C882A53F84EAFA3uL,
         0xEA69BB81B6C52B32uL}, /* 1046 */
        {0x000000028BE60DB9uL, 0x391054A7F09D5F47uL,
         0xD4D377036D8A5664uL}, /* 1047 */
        {0x0000000517CC1B72uL, 0x7220A94FE13ABE8FuL,
         0xA9A6EE06DB14ACC9uL}, /* 1048 */
        {0x0000000A2F9836E4uL, 0xE441529FC2757D1FuL,
         0x534DDC0DB6295993uL}, /* 1049 */
        {0x000000145F306DC9uL, 0xC882A53F84EAFA3EuL,
         0xA69BB81B6C52B327uL}, /* 1050 */
        {0x00000028BE60DB93uL, 0x91054A7F09D5F47DuL,
         0x4D377036D8A5664FuL}, /* 1051 */
        {0x000000517CC1B727uL, 0x220A94FE13ABE8FAuL,
         0x9A6EE06DB14ACC9EuL}, /* 1052 */
        {0x000000A2F9836E4EuL, 0x441529FC2757D1F5uL,
         0x34DDC0DB6295993CuL}, /* 1053 */
        {0x00000145F306DC9CuL, 0x882A53F84EAFA3EAuL,
         0x69BB81B6C52B3278uL}, /* 1054 */
        {0x0000028BE60DB939uL, 0x1054A7F09D5F47D4uL,
         0xD377036D8A5664F1uL}, /* 1055 */
        {0x00000517CC1B7272uL, 0x20A94FE13ABE8FA9uL,
         0xA6EE06DB14ACC9E2uL}, /* 1056 */
        {0x00000A2F9836E4E4uL, 0x41529FC2757D1F53uL,
         0x4DDC0DB6295993C4uL}, /* 1057 */
        {0x0000145F306DC9C8uL, 0x82A53F84EAFA3EA6uL,
         0x9BB81B6C52B32788uL}, /* 1058 */
        {0x000028BE60DB9391uL, 0x054A7F09D5F47D4DuL,
         0x377036D8A5664F10uL}, /* 1059 */
        {0x0000517CC1B72722uL, 0x0A94FE13ABE8FA9AuL,
         0x6EE06DB14ACC9E21uL}, /* 1060 */
        {0x0000A2F9836E4E44uL, 0x1529FC2757D1F534uL,
         0xDDC0DB6295993C43uL}, /* 1061 */
        {0x000145F306DC9C88uL, 0x2A53F84EAFA3EA69uL,
         0xBB81B6C52B327887uL}, /* 1062 */
        {0x00028BE60DB93910uL, 0x54A7F09D5F47D4D3uL,
         0x77036D8A5664F10EuL}, /* 1063 */
        {0x000517CC1B727220uL, 0xA94FE13ABE8FA9A6uL,
         0xEE06DB14ACC9E21CuL}, /* 1064 */
        {0x000A2F9836E4E441uL, 0x529FC2757D1F534DuL,
         0xDC0DB6295993C439uL}, /* 1065 */
        {0x00145F306DC9C882uL, 0xA53F84EAFA3EA69BuL,
         0xB81B6C52B3278872uL}, /* 1066 */
        {0x0028BE60DB939105uL, 0x4A7F09D5F47D4D37uL,
         0x7036D8A5664F10E4uL}, /* 1067 */
        {0x00517CC1B727220AuL, 0x94FE13ABE8FA9A6EuL,
         0xE06DB14ACC9E21C8uL}, /* 1068 */
        {0x00A2F9836E4E4415uL, 0x29FC2757D1F534DDuL,
         0xC0DB6295993C4390uL}, /* 1069 */
        {0x0145F306DC9C882AuL, 0x53F84EAFA3EA69BBuL,
         0x81B6C52B32788720uL}, /* 1070 */
        {0x028BE60DB9391054uL, 0xA7F09D5F47D4D377uL,
         0x036D8A5664F10E41uL}, /* 1071 */
        {0x0517CC1B727220A9uL, 0x4FE13ABE8FA9A6EEuL,
         0x06DB14ACC9E21C82uL}, /* 1072 */
        {0x0A2F9836E4E44152uL, 0x9FC2757D1F534DDCuL,
         0x0DB6295993C43904uL}, /* 1073 */
        {0x145F306DC9C882A5uL, 0x3F84EAFA3EA69BB8uL,
         0x1B6C52B327887208uL}, /* 1074 */
        {0x28BE60DB9391054AuL, 0x7F09D5F47D4D3770uL,
         0x36D8A5664F10E410uL}, /* 1075 */
        {0x517CC1B727220A94uL, 0xFE13ABE8FA9A6EE0uL,
         0x6DB14ACC9E21C820uL}, /* 1076 */
        {0xA2F9836E4E441529uL, 0xFC2757D1F534DDC0uL,
         0xDB6295993C439041uL}, /* 1077 */
        {0x45F306DC9C882A53uL, 0xF84EAFA3EA69BB81uL,
         0xB6C52B3278872083uL}, /* 1078 */
        {0x8BE60DB9391054A7uL, 0xF09D5F47D4D37703uL,
         0x6D8A5664F10E4107uL}, /* 1079 */
        {0x17CC1B727220A94FuL, 0xE13ABE8FA9A6EE06uL,
         0xDB14ACC9E21C820FuL}, /* 1080 */
        {0x2F9836E4E441529FuL, 0xC2757D1F534DDC0DuL,
         0xB6295993C439041FuL}, /* 1081 */
        {0x5F306DC9C882A53FuL, 0x84EAFA3EA69BB81BuL,
         0x6C52B3278872083FuL}, /* 1082 */
        {0xBE60DB9391054A7FuL, 0x09D5F47D4D377036uL,
         0xD8A5664F10E4107FuL}, /* 1083 */
        {0x7CC1B727220A94FEuL, 0x13ABE8FA9A6EE06DuL,
         0xB14ACC9E21C820FFuL}, /* 1084 */
        {0xF9836E4E441529FCuL, 0x2757D1F534DDC0DBuL,
         0x6295993C439041FEuL}, /* 1085 */
        {0xF306DC9C882A53F8uL, 0x4EAFA3EA69BB81B6uL,
         0xC52B3278872083FCuL}, /* 1086 */
        {0xE60DB9391054A7F0uL, 0x9D5F47D4D377036DuL,
         0x8A5664F10E4107F9uL}, /* 1087 */
        {0xCC1B727220A94FE1uL, 0x3ABE8FA9A6EE06DBuL,
         0x14ACC9E21C820FF2uL}, /* 1088 */
        {0x9836E4E441529FC2uL, 0x757D1F534DDC0DB6uL,
         0x295993C439041FE5uL}, /* 1089 */
        {0x306DC9C882A53F84uL, 0xEAFA3EA69BB81B6CuL,
         0x52B3278872083FCAuL}, /* 1090 */
        {0x60DB9391054A7F09uL, 0xD5F47D4D377036D8uL,
         0xA5664F10E4107F94uL}, /* 1091 */
        {0xC1B727220A94FE13uL, 0xABE8FA9A6EE06DB1uL,
         0x4ACC9E21C820FF28uL}, /* 1092 */
        {0x836E4E441529FC27uL, 0x57D1F534DDC0DB62uL,
         0x95993C439041FE51uL}, /* 1093 */
        {0x06DC9C882A53F84EuL, 0xAFA3EA69BB81B6C5uL,
         0x2B3278872083FCA2uL}, /* 1094 */
        {0x0DB9391054A7F09DuL, 0x5F47D4D377036D8AuL,
         0x5664F10E4107F945uL}, /* 1095 */
        {0x1B727220A94FE13AuL, 0xBE8FA9A6EE06DB14uL,
         0xACC9E21C820FF28BuL}, /* 1096 */
        {0x36E4E441529FC275uL, 0x7D1F534DDC0DB629uL,
         0x5993C439041FE516uL}, /* 1097 */
        {0x6DC9C882A53F84EAuL, 0xFA3EA69BB81B6C52uL,
         0xB3278872083FCA2CuL}, /* 1098 */
        {0xDB9391054A7F09D5uL, 0xF47D4D377036D8A5uL,
         0x664F10E4107F9458uL}, /* 1099 */
        {0xB727220A94FE13ABuL, 0xE8FA9A6EE06DB14AuL,
         0xCC9E21C820FF28B1uL}, /* 1100 */
        {0x6E4E441529FC2757uL, 0xD1F534DDC0DB6295uL,
         0x993C439041FE5163uL}, /* 1101 */
        {0xDC9C882A53F84EAFuL, 0xA3EA69BB81B6C52BuL,
         0x3278872083FCA2C7uL}, /* 1102 */
        {0xB9391054A7F09D5FuL, 0x47D4D377036D8A56uL,
         0x64F10E4107F9458EuL}, /* 1103 */
        {0x727220A94FE13ABEuL, 0x8FA9A6EE06DB14ACuL,
         0xC9E21C820FF28B1DuL}, /* 1104 */
        {0xE4E441529FC2757DuL, 0x1F534DDC0DB62959uL,
         0x93C439041FE5163AuL}, /* 1105 */
        {0xC9C882A53F84EAFAuL, 0x3EA69BB81B6C52B3uL,
         0x278872083FCA2C75uL}, /* 1106 */
        {0x9391054A7F09D5F4uL, 0x7D4D377036D8A566uL,
         0x4F10E4107F9458EAuL}, /* 1107 */
        {0x27220A94FE13ABE8uL, 0xFA9A6EE06DB14ACCuL,
         0x9E21C820FF28B1D5uL}, /* 1108 */
        {0x4E441529FC2757D1uL, 0xF534DDC0DB629599uL,
         0x3C439041FE5163ABuL}, /* 1109 */
        {0x9C882A53F84EAFA3uL, 0xEA69BB81B6C52B32uL,
         0x78872083FCA2C757uL}, /* 1110 */
        {0x391054A7F09D5F47uL, 0xD4D377036D8A5664uL,
         0xF10E4107F9458EAFuL}, /* 1111 */
        {0x7220A94FE13ABE8FuL, 0xA9A6EE06DB14ACC9uL,
         0xE21C820FF28B1D5EuL}, /* 1112 */
        {0xE441529FC2757D1FuL, 0x534DDC0DB6295993uL,
         0xC439041FE5163ABDuL}, /* 1113 */
        {0xC882A53F84EAFA3EuL, 0xA69BB81B6C52B327uL,
         0x8872083FCA2C757BuL}, /* 1114 */
        {0x91054A7F09D5F47DuL, 0x4D377036D8A5664FuL,
         0x10E4107F9458EAF7uL}, /* 1115 */
        {0x220A94FE13ABE8FAuL, 0x9A6EE06DB14ACC9EuL,
         0x21C820FF28B1D5EFuL}, /* 1116 */
        {0x441529FC2757D1F5uL, 0x34DDC0DB6295993CuL,
         0x439041FE5163ABDEuL}, /* 1117 */
        {0x882A53F84EAFA3EAuL, 0x69BB81B6C52B3278uL,
         0x872083FCA2C757BDuL}, /* 1118 */
        {0x1054A7F09D5F47D4uL, 0xD377036D8A5664F1uL,
         0x0E4107F9458EAF7AuL}, /* 1119 */
        {0x20A94FE13ABE8FA9uL, 0xA6EE06DB14ACC9E2uL,
         0x1C820FF28B1D5EF5uL}, /* 1120 */
        {0x41529FC2757D1F53uL, 0x4DDC0DB6295993C4uL,
         0x39041FE5163ABDEBuL}, /* 1121 */
        {0x82A53F84EAFA3EA6uL, 0x9BB81B6C52B32788uL,
         0x72083FCA2C757BD7uL}, /* 1122 */
        {0x054A7F09D5F47D4DuL, 0x377036D8A5664F10uL,
         0xE4107F9458EAF7AEuL}, /* 1123 */
        {0x0A94FE13ABE8FA9AuL, 0x6EE06DB14ACC9E21uL,
         0xC820FF28B1D5EF5DuL}, /* 1124 */
        {0x1529FC2757D1F534uL, 0xDDC0DB6295993C43uL,
         0x9041FE5163ABDEBBuL}, /* 1125 */
        {0x2A53F84EAFA3EA69uL, 0xBB81B6C52B327887uL,
         0x2083FCA2C757BD77uL}, /* 1126 */
        {0x54A7F09D5F47D4D3uL, 0x77036D8A5664F10EuL,
         0x4107F9458EAF7AEFuL}, /* 1127 */
        {0xA94FE13ABE8FA9A6uL, 0xEE06DB14ACC9E21CuL,
         0x820FF28B1D5EF5DEuL}, /* 1128 */
        {0x529FC2757D1F534DuL, 0xDC0DB6295993C439uL,
         0x041FE5163ABDEBBCuL}, /* 1129 */
        {0xA53F84EAFA3EA69BuL, 0xB81B6C52B3278872uL,
         0x083FCA2C757BD778uL}, /* 1130 */
        {0x4A7F09D5F47D4D37uL, 0x7036D8A5664F10E4uL,
         0x107F9458EAF7AEF1uL}, /* 1131 */
        {0x94FE13ABE8FA9A6EuL, 0xE06DB14ACC9E21C8uL,
         0x20FF28B1D5EF5DE2uL}, /* 1132 */
        {0x29FC2757D1F534DDuL, 0xC0DB6295993C4390uL,
         0x41FE5163ABDEBBC5uL}, /* 1133 */
        {0x53F84EAFA3EA69BBuL, 0x81B6C52B32788720uL,
         0x83FCA2C757BD778AuL}, /* 1134 */
        {0xA7F09D5F47D4D377uL, 0x036D8A5664F10E41uL,
         0x07F9458EAF7AEF15uL}, /* 1135 */
        {0x4FE13ABE8FA9A6EEuL, 0x06DB14ACC9E21C82uL,
         0x0FF28B1D5EF5DE2BuL}, /* 1136 */
        {0x9FC2757D1F534DDCuL, 0x0DB6295993C43904uL,
         0x1FE5163ABDEBBC56uL}, /* 1137 */
        {0x3F84EAFA3EA69BB8uL, 0x1B6C52B327887208uL,
         0x3FCA2C757BD778ACuL}, /* 1138 */
        {0x7F09D5F47D4D3770uL, 0x36D8A5664F10E410uL,
         0x7F9458EAF7AEF158uL}, /* 1139 */
        {0xFE13ABE8FA9A6EE0uL, 0x6DB14ACC9E21C820uL,
         0xFF28B1D5EF5DE2B0uL}, /* 1140 */
        {0xFC2757D1F534DDC0uL, 0xDB6295993C439041uL,
         0xFE5163ABDEBBC561uL}, /* 1141 */
        {0xF84EAFA3EA69BB81uL, 0xB6C52B3278872083uL,
         0xFCA2C757BD778AC3uL}, /* 1142 */
        {0xF09D5F47D4D37703uL, 0x6D8A5664F10E4107uL,
         0xF9458EAF7AEF1586uL}, /* 1143 */
        {0xE13ABE8FA9A6EE06uL, 0xDB14ACC9E21C820FuL,
         0xF28B1D5EF5DE2B0DuL}, /* 1144 */
        {0xC2757D1F534DDC0DuL, 0xB6295993C439041FuL,
         0xE5163ABDEBBC561BuL}, /* 1145 */
        {0x84EAFA3EA69BB81BuL, 0x6C52B3278872083FuL,
         0xCA2C757BD778AC36uL}, /* 1146 */
        {0x09D5F47D4D377036uL, 0xD8A5664F10E4107FuL,
         0x9458EAF7AEF1586DuL}, /* 1147 */
        {0x13ABE8FA9A6EE06DuL, 0xB14ACC9E21C820FFuL,
         0x28B1D5EF5DE2B0DBuL}, /* 1148 */
        {0x2757D1F534DDC0DBuL, 0x6295993C439041FEuL,
         0x5163ABDEBBC561B7uL}, /* 1149 */
        {0x4EAFA3EA69BB81B6uL, 0xC52B3278872083FCuL,
         0xA2C757BD778AC36EuL}, /* 1150 */
        {0x9D5F47D4D377036DuL, 0x8A5664F10E4107F9uL,
         0x458EAF7AEF1586DCuL}, /* 1151 */
        {0x3ABE8FA9A6EE06DBuL, 0x14ACC9E21C820FF2uL,
         0x8B1D5EF5DE2B0DB9uL}, /* 1152 */
        {0x757D1F534DDC0DB6uL, 0x295993C439041FE5uL,
         0x163ABDEBBC561B72uL}, /* 1153 */
        {0xEAFA3EA69BB81B6CuL, 0x52B3278872083FCAuL,
         0x2C757BD778AC36E4uL}, /* 1154 */
        {0xD5F47D4D377036D8uL, 0xA5664F10E4107F94uL,
         0x58EAF7AEF1586DC9uL}, /* 1155 */
        {0xABE8FA9A6EE06DB1uL, 0x4ACC9E21C820FF28uL,
         0xB1D5EF5DE2B0DB92uL}, /* 1156 */
        {0x57D1F534DDC0DB62uL, 0x95993C439041FE51uL,
         0x63ABDEBBC561B724uL}, /* 1157 */
        {0xAFA3EA69BB81B6C5uL, 0x2B3278872083FCA2uL,
         0xC757BD778AC36E48uL}, /* 1158 */
        {0x5F47D4D377036D8AuL, 0x5664F10E4107F945uL,
         0x8EAF7AEF1586DC91uL}, /* 1159 */
        {0xBE8FA9A6EE06DB14uL, 0xACC9E21C820FF28BuL,
         0x1D5EF5DE2B0DB923uL}, /* 1160 */
        {0x7D1F534DDC0DB629uL, 0x5993C439041FE516uL,
         0x3ABDEBBC561B7246uL}, /* 1161 */
        {0xFA3EA69BB81B6C52uL, 0xB3278872083FCA2CuL,
         0x757BD778AC36E48DuL}, /* 1162 */
        {0xF47D4D377036D8A5uL, 0x664F10E4107F9458uL,
         0xEAF7AEF1586DC91BuL}, /* 1163 */
        {0xE8FA9A6EE06DB14AuL, 0xCC9E21C820FF28B1uL,
         0xD5EF5DE2B0DB9237uL}, /* 1164 */
        {0xD1F534DDC0DB6295uL, 0x993C439041FE5163uL,
         0xABDEBBC561B7246EuL}, /* 1165 */
        {0xA3EA69BB81B6C52BuL, 0x3278872083FCA2C7uL,
         0x57BD778AC36E48DCuL}, /* 1166 */
        {0x47D4D377036D8A56uL, 0x64F10E4107F9458EuL,
         0xAF7AEF1586DC91B8uL}, /* 1167 */
        {0x8FA9A6EE06DB14ACuL, 0xC9E21C820FF28B1DuL,
         0x5EF5DE2B0DB92371uL}, /* 1168 */
        {0x1F534DDC0DB62959uL, 0x93C439041FE5163AuL,
         0xBDEBBC561B7246E3uL}, /* 1169 */
        {0x3EA69BB81B6C52B3uL, 0x278872083FCA2C75uL,
         0x7BD778AC36E48DC7uL}, /* 1170 */
        {0x7D4D377036D8A566uL, 0x4F10E4107F9458EAuL,
         0xF7AEF1586DC91B8EuL}, /* 1171 */
        {0xFA9A6EE06DB14ACCuL, 0x9E21C820FF28B1D5uL,
         0xEF5DE2B0DB92371DuL}, /* 1172 */
        {0xF534DDC0DB629599uL, 0x3C439041FE5163ABuL,
         0xDEBBC561B7246E3AuL}, /* 1173 */
        {0xEA69BB81B6C52B32uL, 0x78872083FCA2C757uL,
         0xBD778AC36E48DC74uL}, /* 1174 */
        {0xD4D377036D8A5664uL, 0xF10E4107F9458EAFuL,
         0x7AEF1586DC91B8E9uL}, /* 1175 */
        {0xA9A6EE06DB14ACC9uL, 0xE21C820FF28B1D5EuL,
         0xF5DE2B0DB92371D2uL}, /* 1176 */
        {0x534DDC0DB6295993uL, 0xC439041FE5163ABDuL,
         0xEBBC561B7246E3A4uL}, /* 1177 */
        {0xA69BB81B6C52B327uL, 0x8872083FCA2C757BuL,
         0xD778AC36E48DC748uL}, /* 1178 */
        {0x4D377036D8A5664FuL, 0x10E4107F9458EAF7uL,
         0xAEF1586DC91B8E90uL}, /* 1179 */
        {0x9A6EE06DB14ACC9EuL, 0x21C820FF28B1D5EFuL,
         0x5DE2B0DB92371D21uL}, /* 1180 */
        {0x34DDC0DB6295993CuL, 0x439041FE5163ABDEuL,
         0xBBC561B7246E3A42uL}, /* 1181 */
        {0x69BB81B6C52B3278uL, 0x872083FCA2C757BDuL,
         0x778AC36E48DC7484uL}, /* 1182 */
        {0xD377036D8A5664F1uL, 0x0E4107F9458EAF7AuL,
         0xEF1586DC91B8E909uL}, /* 1183 */
        {0xA6EE06DB14ACC9E2uL, 0x1C820FF28B1D5EF5uL,
         0xDE2B0DB92371D212uL}, /* 1184 */
        {0x4DDC0DB6295993C4uL, 0x39041FE5163ABDEBuL,
         0xBC561B7246E3A424uL}, /* 1185 */
        {0x9BB81B6C52B32788uL, 0x72083FCA2C757BD7uL,
         0x78AC36E48DC74849uL}, /* 1186 */
        {0x377036D8A5664F10uL, 0xE4107F9458EAF7AEuL,
         0xF1586DC91B8E9093uL}, /* 1187 */
        {0x6EE06DB14ACC9E21uL, 0xC820FF28B1D5EF5DuL,
         0xE2B0DB92371D2126uL}, /* 1188 */
        {0xDDC0DB6295993C43uL, 0x9041FE5163ABDEBBuL,
         0xC561B7246E3A424DuL}, /* 1189 */
        {0xBB81B6C52B327887uL, 0x2083FCA2C757BD77uL,
         0x8AC36E48DC74849BuL}, /* 1190 */
        {0x77036D8A5664F10EuL, 0x4107F9458EAF7AEFuL,
         0x1586DC91B8E90937uL}, /* 1191 */
        {0xEE06DB14ACC9E21CuL, 0x820FF28B1D5EF5DEuL,
         0x2B0DB92371D2126EuL}, /* 1192 */
        {0xDC0DB6295993C439uL, 0x041FE5163ABDEBBCuL,
         0x561B7246E3A424DDuL}, /* 1193 */
        {0xB81B6C52B3278872uL, 0x083FCA2C757BD778uL,
         0xAC36E48DC74849BAuL}, /* 1194 */
        {0x7036D8A5664F10E4uL, 0x107F9458EAF7AEF1uL,
         0x586DC91B8E909374uL}, /* 1195 */
        {0xE06DB14ACC9E21C8uL, 0x20FF28B1D5EF5DE2uL,
         0xB0DB92371D2126E9uL}, /* 1196 */
        {0xC0DB6295993C4390uL, 0x41FE5163ABDEBBC5uL,
         0x61B7246E3A424DD2uL}, /* 1197 */
        {0x81B6C52B32788720uL, 0x83FCA2C757BD778AuL,
         0xC36E48DC74849BA5uL}, /* 1198 */
        {0x036D8A5664F10E41uL, 0x07F9458EAF7AEF15uL,
         0x86DC91B8E909374BuL}, /* 1199 */
        {0x06DB14ACC9E21C82uL, 0x0FF28B1D5EF5DE2BuL,
         0x0DB92371D2126E97uL}, /* 1200 */
        {0x0DB6295993C43904uL, 0x1FE5163ABDEBBC56uL,
         0x1B7246E3A424DD2EuL}, /* 1201 */
        {0x1B6C52B327887208uL, 0x3FCA2C757BD778ACuL,
         0x36E48DC74849BA5CuL}, /* 1202 */
        {0x36D8A5664F10E410uL, 0x7F9458EAF7AEF158uL,
         0x6DC91B8E909374B8uL}, /* 1203 */
        {0x6DB14ACC9E21C820uL, 0xFF28B1D5EF5DE2B0uL,
         0xDB92371D2126E970uL}, /* 1204 */
        {0xDB6295993C439041uL, 0xFE5163ABDEBBC561uL,
         0xB7246E3A424DD2E0uL}, /* 1205 */
        {0xB6C52B3278872083uL, 0xFCA2C757BD778AC3uL,
         0x6E48DC74849BA5C0uL}, /* 1206 */
        {0x6D8A5664F10E4107uL, 0xF9458EAF7AEF1586uL,
         0xDC91B8E909374B80uL}, /* 1207 */
        {0xDB14ACC9E21C820FuL, 0xF28B1D5EF5DE2B0DuL,
         0xB92371D2126E9700uL}, /* 1208 */
        {0xB6295993C439041FuL, 0xE5163ABDEBBC561BuL,
         0x7246E3A424DD2E00uL}, /* 1209 */
        {0x6C52B3278872083FuL, 0xCA2C757BD778AC36uL,
         0xE48DC74849BA5C00uL}, /* 1210 */
        {0xD8A5664F10E4107FuL, 0x9458EAF7AEF1586DuL,
         0xC91B8E909374B801uL}, /* 1211 */
        {0xB14ACC9E21C820FFuL, 0x28B1D5EF5DE2B0DBuL,
         0x92371D2126E97003uL}, /* 1212 */
        {0x6295993C439041FEuL, 0x5163ABDEBBC561B7uL,
         0x246E3A424DD2E006uL}, /* 1213 */
        {0xC52B3278872083FCuL, 0xA2C757BD778AC36EuL,
         0x48DC74849BA5C00CuL}, /* 1214 */
        {0x8A5664F10E4107F9uL, 0x458EAF7AEF1586DCuL,
         0x91B8E909374B8019uL}, /* 1215 */
        {0x14ACC9E21C820FF2uL, 0x8B1D5EF5DE2B0DB9uL,
         0x2371D2126E970032uL}, /* 1216 */
        {0x295993C439041FE5uL, 0x163ABDEBBC561B72uL,
         0x46E3A424DD2E0064uL}, /* 1217 */
        {0x52B3278872083FCAuL, 0x2C757BD778AC36E4uL,
         0x8DC74849BA5C00C9uL}, /* 1218 */
        {0xA5664F10E4107F94uL, 0x58EAF7AEF1586DC9uL,
         0x1B8E909374B80192uL}, /* 1219 */
        {0x4ACC9E21C820FF28uL, 0xB1D5EF5DE2B0DB92uL,
         0x371D2126E9700324uL}, /* 1220 */
        {0x95993C439041FE51uL, 0x63ABDEBBC561B724uL,
         0x6E3A424DD2E00649uL}, /* 1221 */
        {0x2B3278872083FCA2uL, 0xC757BD778AC36E48uL,
         0xDC74849BA5C00C92uL}, /* 1222 */
        {0x5664F10E4107F945uL, 0x8EAF7AEF1586DC91uL,
         0xB8E909374B801924uL}, /* 1223 */
        {0xACC9E21C820FF28BuL, 0x1D5EF5DE2B0DB923uL,
         0x71D2126E97003249uL}, /* 1224 */
        {0x5993C439041FE516uL, 0x3ABDEBBC561B7246uL,
         0xE3A424DD2E006492uL}, /* 1225 */
        {0xB3278872083FCA2CuL, 0x757BD778AC36E48DuL,
         0xC74849BA5C00C925uL}, /* 1226 */
        {0x664F10E4107F9458uL, 0xEAF7AEF1586DC91BuL,
         0x8E909374B801924BuL}, /* 1227 */
        {0xCC9E21C820FF28B1uL, 0xD5EF5DE2B0DB9237uL,
         0x1D2126E970032497uL}, /* 1228 */
        {0x993C439041FE5163uL, 0xABDEBBC561B7246EuL,
         0x3A424DD2E006492EuL}, /* 1229 */
        {0x3278872083FCA2C7uL, 0x57BD778AC36E48DCuL,
         0x74849BA5C00C925DuL}, /* 1230 */
        {0x64F10E4107F9458EuL, 0xAF7AEF1586DC91B8uL,
         0xE909374B801924BBuL}, /* 1231 */
        {0xC9E21C820FF28B1DuL, 0x5EF5DE2B0DB92371uL,
         0xD2126E9700324977uL}, /* 1232 */
        {0x93C439041FE5163AuL, 0xBDEBBC561B7246E3uL,
         0xA424DD2E006492EEuL}, /* 1233 */
        {0x278872083FCA2C75uL, 0x7BD778AC36E48DC7uL,
         0x4849BA5C00C925DDuL}, /* 1234 */
        {0x4F10E4107F9458EAuL, 0xF7AEF1586DC91B8EuL,
         0x909374B801924BBAuL}, /* 1235 */
        {0x9E21C820FF28B1D5uL, 0xEF5DE2B0DB92371DuL,
         0x2126E97003249775uL}, /* 1236 */
        {0x3C439041FE5163ABuL, 0xDEBBC561B7246E3AuL,
         0x424DD2E006492EEAuL}, /* 1237 */
        {0x78872083FCA2C757uL, 0xBD778AC36E48DC74uL,
         0x849BA5C00C925DD4uL}, /* 1238 */
        {0xF10E4107F9458EAFuL, 0x7AEF1586DC91B8E9uL,
         0x09374B801924BBA8uL}, /* 1239 */
        {0xE21C820FF28B1D5EuL, 0xF5DE2B0DB92371D2uL,
         0x126E970032497750uL}, /* 1240 */
        {0xC439041FE5163ABDuL, 0xEBBC561B7246E3A4uL,
         0x24DD2E006492EEA0uL}, /* 1241 */
        {0x8872083FCA2C757BuL, 0xD778AC36E48DC748uL,
         0x49BA5C00C925DD41uL}, /* 1242 */
        {0x10E4107F9458EAF7uL, 0xAEF1586DC91B8E90uL,
         0x9374B801924BBA82uL}, /* 1243 */
        {0x21C820FF28B1D5EFuL, 0x5DE2B0DB92371D21uL,
         0x26E9700324977504uL}, /* 1244 */
        {0x439041FE5163ABDEuL, 0xBBC561B7246E3A42uL,
         0x4DD2E006492EEA09uL}, /* 1245 */
        {0x872083FCA2C757BDuL, 0x778AC36E48DC7484uL,
         0x9BA5C00C925DD413uL}, /* 1246 */
        {0x0E4107F9458EAF7AuL, 0xEF1586DC91B8E909uL,
         0x374B801924BBA827uL}, /* 1247 */
        {0x1C820FF28B1D5EF5uL, 0xDE2B0DB92371D212uL,
         0x6E9700324977504EuL}, /* 1248 */
        {0x39041FE5163ABDEBuL, 0xBC561B7246E3A424uL,
         0xDD2E006492EEA09DuL}, /* 1249 */
        {0x72083FCA2C757BD7uL, 0x78AC36E48DC74849uL,
         0xBA5C00C925DD413AuL}, /* 1250 */
        {0xE4107F9458EAF7AEuL, 0xF1586DC91B8E9093uL,
         0x74B801924BBA8274uL}, /* 1251 */
        {0xC820FF28B1D5EF5DuL, 0xE2B0DB92371D2126uL,
         0xE9700324977504E8uL}, /* 1252 */
        {0x9041FE5163ABDEBBuL, 0xC561B7246E3A424DuL,
         0xD2E006492EEA09D1uL}, /* 1253 */
        {0x2083FCA2C757BD77uL, 0x8AC36E48DC74849BuL,
         0xA5C00C925DD413A3uL}, /* 1254 */
        {0x4107F9458EAF7AEFuL, 0x1586DC91B8E90937uL,
         0x4B801924BBA82746uL}, /* 1255 */
        {0x820FF28B1D5EF5DEuL, 0x2B0DB92371D2126EuL,
         0x9700324977504E8CuL}, /* 1256 */
        {0x041FE5163ABDEBBCuL, 0x561B7246E3A424DDuL,
         0x2E006492EEA09D19uL}, /* 1257 */
        {0x083FCA2C757BD778uL, 0xAC36E48DC74849BAuL,
         0x5C00C925DD413A32uL}, /* 1258 */
        {0x107F9458EAF7AEF1uL, 0x586DC91B8E909374uL,
         0xB801924BBA827464uL}, /* 1259 */
        {0x20FF28B1D5EF5DE2uL, 0xB0DB92371D2126E9uL,
         0x700324977504E8C9uL}, /* 1260 */
        {0x41FE5163ABDEBBC5uL, 0x61B7246E3A424DD2uL,
         0xE006492EEA09D192uL}, /* 1261 */
        {0x83FCA2C757BD778AuL, 0xC36E48DC74849BA5uL,
         0xC00C925DD413A324uL}, /* 1262 */
        {0x07F9458EAF7AEF15uL, 0x86DC91B8E909374BuL,
         0x801924BBA8274648uL}, /* 1263 */
        {0x0FF28B1D5EF5DE2BuL, 0x0DB92371D2126E97uL,
         0x00324977504E8C90uL}, /* 1264 */
        {0x1FE5163ABDEBBC56uL, 0x1B7246E3A424DD2EuL,
         0x006492EEA09D1921uL}, /* 1265 */
        {0x3FCA2C757BD778ACuL, 0x36E48DC74849BA5CuL,
         0x00C925DD413A3243uL}, /* 1266 */
        {0x7F9458EAF7AEF158uL, 0x6DC91B8E909374B8uL,
         0x01924BBA82746487uL}, /* 1267 */
        {0xFF28B1D5EF5DE2B0uL, 0xDB92371D2126E970uL,
         0x0324977504E8C90EuL}, /* 1268 */
        {0xFE5163ABDEBBC561uL, 0xB7246E3A424DD2E0uL,
         0x06492EEA09D1921CuL}, /* 1269 */
        {0xFCA2C757BD778AC3uL, 0x6E48DC74849BA5C0uL,
         0x0C925DD413A32439uL}, /* 1270 */
        {0xF9458EAF7AEF1586uL, 0xDC91B8E909374B80uL,
         0x1924BBA827464873uL}, /* 1271 */
        {0xF28B1D5EF5DE2B0DuL, 0xB92371D2126E9700uL,
         0x324977504E8C90E7uL}, /* 1272 */
        {0xE5163ABDEBBC561BuL, 0x7246E3A424DD2E00uL,
         0x6492EEA09D1921CFuL}, /* 1273 */
        {0xCA2C757BD778AC36uL, 0xE48DC74849BA5C00uL,
         0xC925DD413A32439FuL}, /* 1274 */
        {0x9458EAF7AEF1586DuL, 0xC91B8E909374B801uL,
         0x924BBA827464873FuL}, /* 1275 */
        {0x28B1D5EF5DE2B0DBuL, 0x92371D2126E97003uL,
         0x24977504E8C90E7FuL}, /* 1276 */
        {0x5163ABDEBBC561B7uL, 0x246E3A424DD2E006uL,
         0x492EEA09D1921CFEuL}, /* 1277 */
        {0xA2C757BD778AC36EuL, 0x48DC74849BA5C00CuL,
         0x925DD413A32439FCuL}, /* 1278 */
        {0x458EAF7AEF1586DCuL, 0x91B8E909374B8019uL,
         0x24BBA827464873F8uL}, /* 1279 */
        {0x8B1D5EF5DE2B0DB9uL, 0x2371D2126E970032uL,
         0x4977504E8C90E7F0uL}, /* 1280 */
        {0x163ABDEBBC561B72uL, 0x46E3A424DD2E0064uL,
         0x92EEA09D1921CFE1uL}, /* 1281 */
        {0x2C757BD778AC36E4uL, 0x8DC74849BA5C00C9uL,
         0x25DD413A32439FC3uL}, /* 1282 */
        {0x58EAF7AEF1586DC9uL, 0x1B8E909374B80192uL,
         0x4BBA827464873F87uL}, /* 1283 */
        {0xB1D5EF5DE2B0DB92uL, 0x371D2126E9700324uL,
         0x977504E8C90E7F0EuL}, /* 1284 */
        {0x63ABDEBBC561B724uL, 0x6E3A424DD2E00649uL,
         0x2EEA09D1921CFE1DuL}, /* 1285 */
        {0xC757BD778AC36E48uL, 0xDC74849BA5C00C92uL,
         0x5DD413A32439FC3BuL}, /* 1286 */
        {0x8EAF7AEF1586DC91uL, 0xB8E909374B801924uL,
         0xBBA827464873F877uL}, /* 1287 */
        {0x1D5EF5DE2B0DB923uL, 0x71D2126E97003249uL,
         0x77504E8C90E7F0EFuL}, /* 1288 */
        {0x3ABDEBBC561B7246uL, 0xE3A424DD2E006492uL,
         0xEEA09D1921CFE1DEuL}, /* 1289 */
        {0x757BD778AC36E48DuL, 0xC74849BA5C00C925uL,
         0xDD413A32439FC3BDuL}, /* 1290 */
        {0xEAF7AEF1586DC91BuL, 0x8E909374B801924BuL,
         0xBA827464873F877AuL}, /* 1291 */
        {0xD5EF5DE2B0DB9237uL, 0x1D2126E970032497uL,
         0x7504E8C90E7F0EF5uL}, /* 1292 */
        {0xABDEBBC561B7246EuL, 0x3A424DD2E006492EuL,
         0xEA09D1921CFE1DEBuL}, /* 1293 */
        {0x57BD778AC36E48DCuL, 0x74849BA5C00C925DuL,
         0xD413A32439FC3BD6uL}, /* 1294 */
        {0xAF7AEF1586DC91B8uL, 0xE909374B801924BBuL,
         0xA827464873F877ACuL}, /* 1295 */
        {0x5EF5DE2B0DB92371uL, 0xD2126E9700324977uL,
         0x504E8C90E7F0EF58uL}, /* 1296 */
        {0xBDEBBC561B7246E3uL, 0xA424DD2E006492EEuL,
         0xA09D1921CFE1DEB1uL}, /* 1297 */
        {0x7BD778AC36E48DC7uL, 0x4849BA5C00C925DDuL,
         0x413A32439FC3BD63uL}, /* 1298 */
        {0xF7AEF1586DC91B8EuL, 0x909374B801924BBAuL,
         0x827464873F877AC7uL}, /* 1299 */
        {0xEF5DE2B0DB92371DuL, 0x2126E97003249775uL,
         0x04E8C90E7F0EF58EuL}, /* 1300 */
        {0xDEBBC561B7246E3AuL, 0x424DD2E006492EEAuL,
         0x09D1921CFE1DEB1CuL}, /* 1301 */
        {0xBD778AC36E48DC74uL, 0x849BA5C00C925DD4uL,
         0x13A32439FC3BD639uL}, /* 1302 */
        {0x7AEF1586DC91B8E9uL, 0x09374B801924BBA8uL,
         0x27464873F877AC72uL}, /* 1303 */
        {0xF5DE2B0DB92371D2uL, 0x126E970032497750uL,
         0x4E8C90E7F0EF58E5uL}, /* 1304 */
        {0xEBBC561B7246E3A4uL, 0x24DD2E006492EEA0uL,
         0x9D1921CFE1DEB1CBuL}, /* 1305 */
        {0xD778AC36E48DC748uL, 0x49BA5C00C925DD41uL,
         0x3A32439FC3BD6396uL}, /* 1306 */
        {0xAEF1586DC91B8E90uL, 0x9374B801924BBA82uL,
         0x7464873F877AC72CuL}, /* 1307 */
        {0x5DE2B0DB92371D21uL, 0x26E9700324977504uL,
         0xE8C90E7F0EF58E58uL}, /* 1308 */
        {0xBBC561B7246E3A42uL, 0x4DD2E006492EEA09uL,
         0xD1921CFE1DEB1CB1uL}, /* 1309 */
        {0x778AC36E48DC7484uL, 0x9BA5C00C925DD413uL,
         0xA32439FC3BD63962uL}, /* 1310 */
        {0xEF1586DC91B8E909uL, 0x374B801924BBA827uL,
         0x464873F877AC72C4uL}, /* 1311 */
        {0xDE2B0DB92371D212uL, 0x6E9700324977504EuL,
         0x8C90E7F0EF58E589uL}, /* 1312 */
        {0xBC561B7246E3A424uL, 0xDD2E006492EEA09DuL,
         0x1921CFE1DEB1CB12uL}, /* 1313 */
        {0x78AC36E48DC74849uL, 0xBA5C00C925DD413AuL,
         0x32439FC3BD639625uL}, /* 1314 */
        {0xF1586DC91B8E9093uL, 0x74B801924BBA8274uL,
         0x64873F877AC72C4AuL}, /* 1315 */
        {0xE2B0DB92371D2126uL, 0xE9700324977504E8uL,
         0xC90E7F0EF58E5894uL}, /* 1316 */
        {0xC561B7246E3A424DuL, 0xD2E006492EEA09D1uL,
         0x921CFE1DEB1CB129uL}, /* 1317 */
        {0x8AC36E48DC74849BuL, 0xA5C00C925DD413A3uL,
         0x2439FC3BD6396253uL}, /* 1318 */
        {0x1586DC91B8E90937uL, 0x4B801924BBA82746uL,
         0x4873F877AC72C4A6uL}, /* 1319 */
        {0x2B0DB92371D2126EuL, 0x9700324977504E8CuL,
         0x90E7F0EF58E5894DuL}, /* 1320 */
        {0x561B7246E3A424DDuL, 0x2E006492EEA09D19uL,
         0x21CFE1DEB1CB129AuL}, /* 1321 */
        {0xAC36E48DC74849BAuL, 0x5C00C925DD413A32uL,
         0x439FC3BD63962534uL}, /* 1322 */
        {0x586DC91B8E909374uL, 0xB801924BBA827464uL,
         0x873F877AC72C4A69uL}, /* 1323 */
        {0xB0DB92371D2126E9uL, 0x700324977504E8C9uL,
         0x0E7F0EF58E5894D3uL}, /* 1324 */
        {0x61B7246E3A424DD2uL, 0xE006492EEA09D192uL,
         0x1CFE1DEB1CB129A7uL}, /* 1325 */
        {0xC36E48DC74849BA5uL, 0xC00C925DD413A324uL,
         0x39FC3BD63962534EuL}, /* 1326 */
        {0x86DC91B8E909374BuL, 0x801924BBA8274648uL,
         0x73F877AC72C4A69CuL}, /* 1327 */
        {0x0DB92371D2126E97uL, 0x00324977504E8C90uL,
         0xE7F0EF58E5894D39uL}, /* 1328 */
        {0x1B7246E3A424DD2EuL, 0x006492EEA09D1921uL,
         0xCFE1DEB1CB129A73uL}, /* 1329 */
        {0x36E48DC74849BA5CuL, 0x00C925DD413A3243uL,
         0x9FC3BD63962534E7uL}, /* 1330 */
        {0x6DC91B8E909374B8uL, 0x01924BBA82746487uL,
         0x3F877AC72C4A69CFuL}, /* 1331 */
        {0xDB92371D2126E970uL, 0x0324977504E8C90EuL,
         0x7F0EF58E5894D39FuL}, /* 1332 */
        {0xB7246E3A424DD2E0uL, 0x06492EEA09D1921CuL,
         0xFE1DEB1CB129A73EuL}, /* 1333 */
        {0x6E48DC74849BA5C0uL, 0x0C925DD413A32439uL,
         0xFC3BD63962534E7DuL}, /* 1334 */
        {0xDC91B8E909374B80uL, 0x1924BBA827464873uL,
         0xF877AC72C4A69CFBuL}, /* 1335 */
        {0xB92371D2126E9700uL, 0x324977504E8C90E7uL,
         0xF0EF58E5894D39F7uL}, /* 1336 */
        {0x7246E3A424DD2E00uL, 0x6492EEA09D1921CFuL,
         0xE1DEB1CB129A73EEuL}, /* 1337 */
        {0xE48DC74849BA5C00uL, 0xC925DD413A32439FuL,
         0xC3BD63962534E7DDuL}, /* 1338 */
        {0xC91B8E909374B801uL, 0x924BBA827464873FuL,
         0x877AC72C4A69CFBAuL}, /* 1339 */
        {0x92371D2126E97003uL, 0x24977504E8C90E7FuL,
         0x0EF58E5894D39F74uL}, /* 1340 */
        {0x246E3A424DD2E006uL, 0x492EEA09D1921CFEuL,
         0x1DEB1CB129A73EE8uL}, /* 1341 */
        {0x48DC74849BA5C00CuL, 0x925DD413A32439FCuL,
         0x3BD63962534E7DD1uL}, /* 1342 */
        {0x91B8E909374B8019uL, 0x24BBA827464873F8uL,
         0x77AC72C4A69CFBA2uL}, /* 1343 */
        {0x2371D2126E970032uL, 0x4977504E8C90E7F0uL,
         0xEF58E5894D39F744uL}, /* 1344 */
        {0x46E3A424DD2E0064uL, 0x92EEA09D1921CFE1uL,
         0xDEB1CB129A73EE88uL}, /* 1345 */
        {0x8DC74849BA5C00C9uL, 0x25DD413A32439FC3uL,
         0xBD63962534E7DD10uL}, /* 1346 */
        {0x1B8E909374B80192uL, 0x4BBA827464873F87uL,
         0x7AC72C4A69CFBA20uL}, /* 1347 */
        {0x371D2126E9700324uL, 0x977504E8C90E7F0EuL,
         0xF58E5894D39F7441uL}, /* 1348 */
        {0x6E3A424DD2E00649uL, 0x2EEA09D1921CFE1DuL,
         0xEB1CB129A73EE882uL}, /* 1349 */
        {0xDC74849BA5C00C92uL, 0x5DD413A32439FC3BuL,
         0xD63962534E7DD104uL}, /* 1350 */
        {0xB8E909374B801924uL, 0xBBA827464873F877uL,
         0xAC72C4A69CFBA208uL}, /* 1351 */
        {0x71D2126E97003249uL, 0x77504E8C90E7F0EFuL,
         0x58E5894D39F74411uL}, /* 1352 */
        {0xE3A424DD2E006492uL, 0xEEA09D1921CFE1DEuL,
         0xB1CB129A73EE8823uL}, /* 1353 */
        {0xC74849BA5C00C925uL, 0xDD413A32439FC3BDuL,
         0x63962534E7DD1046uL}, /* 1354 */
        {0x8E909374B801924BuL, 0xBA827464873F877AuL,
         0xC72C4A69CFBA208DuL}, /* 1355 */
        {0x1D2126E970032497uL, 0x7504E8C90E7F0EF5uL,
         0x8E5894D39F74411AuL}, /* 1356 */
        {0x3A424DD2E006492EuL, 0xEA09D1921CFE1DEBuL,
         0x1CB129A73EE88235uL}, /* 1357 */
        {0x74849BA5C00C925DuL, 0xD413A32439FC3BD6uL,
         0x3962534E7DD1046BuL}, /* 1358 */
        {0xE909374B801924BBuL, 0xA827464873F877ACuL,
         0x72C4A69CFBA208D7uL}, /* 1359 */
        {0xD2126E9700324977uL, 0x504E8C90E7F0EF58uL,
         0xE5894D39F74411AFuL}, /* 1360 */
        {0xA424DD2E006492EEuL, 0xA09D1921CFE1DEB1uL,
         0xCB129A73EE88235FuL}, /* 1361 */
        {0x4849BA5C00C925DDuL, 0x413A32439FC3BD63uL,
         0x962534E7DD1046BEuL}, /* 1362 */
        {0x909374B801924BBAuL, 0x827464873F877AC7uL,
         0x2C4A69CFBA208D7DuL}, /* 1363 */
        {0x2126E97003249775uL, 0x04E8C90E7F0EF58EuL,
         0x5894D39F74411AFAuL}, /* 1364 */
        {0x424DD2E006492EEAuL, 0x09D1921CFE1DEB1CuL,
         0xB129A73EE88235F5uL}, /* 1365 */
        {0x849BA5C00C925DD4uL, 0x13A32439FC3BD639uL,
         0x62534E7DD1046BEAuL}, /* 1366 */
        {0x09374B801924BBA8uL, 0x27464873F877AC72uL,
         0xC4A69CFBA208D7D4uL}, /* 1367 */
        {0x126E970032497750uL, 0x4E8C90E7F0EF58E5uL,
         0x894D39F74411AFA9uL}, /* 1368 */
        {0x24DD2E006492EEA0uL, 0x9D1921CFE1DEB1CBuL,
         0x129A73EE88235F52uL}, /* 1369 */
        {0x49BA5C00C925DD41uL, 0x3A32439FC3BD6396uL,
         0x2534E7DD1046BEA5uL}, /* 1370 */
        {0x9374B801924BBA82uL, 0x7464873F877AC72CuL,
         0x4A69CFBA208D7D4BuL}, /* 1371 */
        {0x26E9700324977504uL, 0xE8C90E7F0EF58E58uL,
         0x94D39F74411AFA97uL}, /* 1372 */
        {0x4DD2E006492EEA09uL, 0xD1921CFE1DEB1CB1uL,
         0x29A73EE88235F52EuL}, /* 1373 */
        {0x9BA5C00C925DD413uL, 0xA32439FC3BD63962uL,
         0x534E7DD1046BEA5DuL}, /* 1374 */
        {0x374B801924BBA827uL, 0x464873F877AC72C4uL,
         0xA69CFBA208D7D4BAuL}, /* 1375 */
        {0x6E9700324977504EuL, 0x8C90E7F0EF58E589uL,
         0x4D39F74411AFA975uL}, /* 1376 */
        {0xDD2E006492EEA09DuL, 0x1921CFE1DEB1CB12uL,
         0x9A73EE88235F52EBuL}, /* 1377 */
        {0xBA5C00C925DD413AuL, 0x32439FC3BD639625uL,
         0x34E7DD1046BEA5D7uL}, /* 1378 */
        {0x74B801924BBA8274uL, 0x64873F877AC72C4AuL,
         0x69CFBA208D7D4BAEuL}, /* 1379 */
        {0xE9700324977504E8uL, 0xC90E7F0EF58E5894uL,
         0xD39F74411AFA975DuL}, /* 1380 */
        {0xD2E006492EEA09D1uL, 0x921CFE1DEB1CB129uL,
         0xA73EE88235F52EBBuL}, /* 1381 */
        {0xA5C00C925DD413A3uL, 0x2439FC3BD6396253uL,
         0x4E7DD1046BEA5D76uL}, /* 1382 */
        {0x4B801924BBA82746uL, 0x4873F877AC72C4A6uL,
         0x9CFBA208D7D4BAEDuL}, /* 1383 */
        {0x9700324977504E8CuL, 0x90E7F0EF58E5894DuL,
         0x39F74411AFA975DAuL}, /* 1384 */
        {0x2E006492EEA09D19uL, 0x21CFE1DEB1CB129AuL,
         0x73EE88235F52EBB4uL}, /* 1385 */
        {0x5C00C925DD413A32uL, 0x439FC3BD63962534uL,
         0xE7DD1046BEA5D768uL}, /* 1386 */
        {0xB801924BBA827464uL, 0x873F877AC72C4A69uL,
         0xCFBA208D7D4BAED1uL}, /* 1387 */
        {0x700324977504E8C9uL, 0x0E7F0EF58E5894D3uL,
         0x9F74411AFA975DA2uL}, /* 1388 */
        {0xE006492EEA09D192uL, 0x1CFE1DEB1CB129A7uL,
         0x3EE88235F52EBB44uL}, /* 1389 */
        {0xC00C925DD413A324uL, 0x39FC3BD63962534EuL,
         0x7DD1046BEA5D7689uL}, /* 1390 */
        {0x801924BBA8274648uL, 0x73F877AC72C4A69CuL,
         0xFBA208D7D4BAED12uL}, /* 1391 */
        {0x00324977504E8C90uL, 0xE7F0EF58E5894D39uL,
         0xF74411AFA975DA24uL}, /* 1392 */
        {0x006492EEA09D1921uL, 0xCFE1DEB1CB129A73uL,
         0xEE88235F52EBB448uL}, /* 1393 */
        {0x00C925DD413A3243uL, 0x9FC3BD63962534E7uL,
         0xDD1046BEA5D76890uL}, /* 1394 */
        {0x01924BBA82746487uL, 0x3F877AC72C4A69CFuL,
         0xBA208D7D4BAED121uL}, /* 1395 */
        {0x0324977504E8C90EuL, 0x7F0EF58E5894D39FuL,
         0x74411AFA975DA242uL}, /* 1396 */
        {0x06492EEA09D1921CuL, 0xFE1DEB1CB129A73EuL,
         0xE88235F52EBB4484uL}, /* 1397 */
        {0x0C925DD413A32439uL, 0xFC3BD63962534E7DuL,
         0xD1046BEA5D768909uL}, /* 1398 */
        {0x1924BBA827464873uL, 0xF877AC72C4A69CFBuL,
         0xA208D7D4BAED1213uL}, /* 1399 */
        {0x324977504E8C90E7uL, 0xF0EF58E5894D39F7uL,
         0x4411AFA975DA2427uL}, /* 1400 */
        {0x6492EEA09D1921CFuL, 0xE1DEB1CB129A73EEuL,
         0x88235F52EBB4484EuL}, /* 1401 */
        {0xC925DD413A32439FuL, 0xC3BD63962534E7DDuL,
         0x1046BEA5D768909DuL}, /* 1402 */
        {0x924BBA827464873FuL, 0x877AC72C4A69CFBAuL,
         0x208D7D4BAED1213AuL}, /* 1403 */
        {0x24977504E8C90E7FuL, 0x0EF58E5894D39F74uL,
         0x411AFA975DA24274uL}, /* 1404 */
        {0x492EEA09D1921CFEuL, 0x1DEB1CB129A73EE8uL,
         0x8235F52EBB4484E9uL}, /* 1405 */
        {0x925DD413A32439FCuL, 0x3BD63962534E7DD1uL,
         0x046BEA5D768909D3uL}, /* 1406 */
        {0x24BBA827464873F8uL, 0x77AC72C4A69CFBA2uL,
         0x08D7D4BAED1213A6uL}, /* 1407 */
        {0x4977504E8C90E7F0uL, 0xEF58E5894D39F744uL,
         0x11AFA975DA24274CuL}, /* 1408 */
        {0x92EEA09D1921CFE1uL, 0xDEB1CB129A73EE88uL,
         0x235F52EBB4484E99uL}, /* 1409 */
        {0x25DD413A32439FC3uL, 0xBD63962534E7DD10uL,
         0x46BEA5D768909D33uL}, /* 1410 */
        {0x4BBA827464873F87uL, 0x7AC72C4A69CFBA20uL,
         0x8D7D4BAED1213A67uL}, /* 1411 */
        {0x977504E8C90E7F0EuL, 0xF58E5894D39F7441uL,
         0x1AFA975DA24274CEuL}, /* 1412 */
        {0x2EEA09D1921CFE1DuL, 0xEB1CB129A73EE882uL,
         0x35F52EBB4484E99CuL}, /* 1413 */
        {0x5DD413A32439FC3BuL, 0xD63962534E7DD104uL,
         0x6BEA5D768909D338uL}, /* 1414 */
        {0xBBA827464873F877uL, 0xAC72C4A69CFBA208uL,
         0xD7D4BAED1213A671uL}, /* 1415 */
        {0x77504E8C90E7F0EFuL, 0x58E5894D39F74411uL,
         0xAFA975DA24274CE3uL}, /* 1416 */
        {0xEEA09D1921CFE1DEuL, 0xB1CB129A73EE8823uL,
         0x5F52EBB4484E99C7uL}, /* 1417 */
        {0xDD413A32439FC3BDuL, 0x63962534E7DD1046uL,
         0xBEA5D768909D338EuL}, /* 1418 */
        {0xBA827464873F877AuL, 0xC72C4A69CFBA208DuL,
         0x7D4BAED1213A671CuL}, /* 1419 */
        {0x7504E8C90E7F0EF5uL, 0x8E5894D39F74411AuL,
         0xFA975DA24274CE38uL}, /* 1420 */
        {0xEA09D1921CFE1DEBuL, 0x1CB129A73EE88235uL,
         0xF52EBB4484E99C70uL}, /* 1421 */
        {0xD413A32439FC3BD6uL, 0x3962534E7DD1046BuL,
         0xEA5D768909D338E0uL}, /* 1422 */
        {0xA827464873F877ACuL, 0x72C4A69CFBA208D7uL,
         0xD4BAED1213A671C0uL}, /* 1423 */
        {0x504E8C90E7F0EF58uL, 0xE5894D39F74411AFuL,
         0xA975DA24274CE381uL}, /* 1424 */
        {0xA09D1921CFE1DEB1uL, 0xCB129A73EE88235FuL,
         0x52EBB4484E99C702uL}, /* 1425 */
        {0x413A32439FC3BD63uL, 0x962534E7DD1046BEuL,
         0xA5D768909D338E04uL}, /* 1426 */
        {0x827464873F877AC7uL, 0x2C4A69CFBA208D7DuL,
         0x4BAED1213A671C09uL}, /* 1427 */
        {0x04E8C90E7F0EF58EuL, 0x5894D39F74411AFAuL,
         0x975DA24274CE3813uL}, /* 1428 */
        {0x09D1921CFE1DEB1CuL, 0xB129A73EE88235F5uL,
         0x2EBB4484E99C7026uL}, /* 1429 */
        {0x13A32439FC3BD639uL, 0x62534E7DD1046BEAuL,
         0x5D768909D338E04DuL}, /* 1430 */
        {0x27464873F877AC72uL, 0xC4A69CFBA208D7D4uL,
         0xBAED1213A671C09AuL}, /* 1431 */
        {0x4E8C90E7F0EF58E5uL, 0x894D39F74411AFA9uL,
         0x75DA24274CE38135uL}, /* 1432 */
        {0x9D1921CFE1DEB1CBuL, 0x129A73EE88235F52uL,
         0xEBB4484E99C7026BuL}, /* 1433 */
        {0x3A32439FC3BD6396uL, 0x2534E7DD1046BEA5uL,
         0xD768909D338E04D6uL}, /* 1434 */
        {0x7464873F877AC72CuL, 0x4A69CFBA208D7D4BuL,
         0xAED1213A671C09ADuL}, /* 1435 */
        {0xE8C90E7F0EF58E58uL, 0x94D39F74411AFA97uL,
         0x5DA24274CE38135AuL}, /* 1436 */
        {0xD1921CFE1DEB1CB1uL, 0x29A73EE88235F52EuL,
         0xBB4484E99C7026B4uL}, /* 1437 */
        {0xA32439FC3BD63962uL, 0x534E7DD1046BEA5DuL,
         0x768909D338E04D68uL}, /* 1438 */
        {0x464873F877AC72C4uL, 0xA69CFBA208D7D4BAuL,
         0xED1213A671C09AD1uL}, /* 1439 */
        {0x8C90E7F0EF58E589uL, 0x4D39F74411AFA975uL,
         0xDA24274CE38135A2uL}, /* 1440 */
        {0x1921CFE1DEB1CB12uL, 0x9A73EE88235F52EBuL,
         0xB4484E99C7026B45uL}, /* 1441 */
        {0x32439FC3BD639625uL, 0x34E7DD1046BEA5D7uL,
         0x68909D338E04D68BuL}, /* 1442 */
        {0x64873F877AC72C4AuL, 0x69CFBA208D7D4BAEuL,
         0xD1213A671C09AD17uL}, /* 1443 */
        {0xC90E7F0EF58E5894uL, 0xD39F74411AFA975DuL,
         0xA24274CE38135A2FuL}, /* 1444 */
        {0x921CFE1DEB1CB129uL, 0xA73EE88235F52EBBuL,
         0x4484E99C7026B45FuL}, /* 1445 */
        {0x2439FC3BD6396253uL, 0x4E7DD1046BEA5D76uL,
         0x8909D338E04D68BEuL}, /* 1446 */
        {0x4873F877AC72C4A6uL, 0x9CFBA208D7D4BAEDuL,
         0x1213A671C09AD17DuL}, /* 1447 */
        {0x90E7F0EF58E5894DuL, 0x39F74411AFA975DAuL,
         0x24274CE38135A2FBuL}, /* 1448 */
        {0x21CFE1DEB1CB129AuL, 0x73EE88235F52EBB4uL,
         0x484E99C7026B45F7uL}, /* 1449 */
        {0x439FC3BD63962534uL, 0xE7DD1046BEA5D768uL,
         0x909D338E04D68BEFuL}, /* 1450 */
        {0x873F877AC72C4A69uL, 0xCFBA208D7D4BAED1uL,
         0x213A671C09AD17DFuL}, /* 1451 */
        {0x0E7F0EF58E5894D3uL, 0x9F74411AFA975DA2uL,
         0x4274CE38135A2FBFuL}, /* 1452 */
        {0x1CFE1DEB1CB129A7uL, 0x3EE88235F52EBB44uL,
         0x84E99C7026B45F7EuL}, /* 1453 */
        {0x39FC3BD63962534EuL, 0x7DD1046BEA5D7689uL,
         0x09D338E04D68BEFCuL}, /* 1454 */
        {0x73F877AC72C4A69CuL, 0xFBA208D7D4BAED12uL,
         0x13A671C09AD17DF9uL}, /* 1455 */
        {0xE7F0EF58E5894D39uL, 0xF74411AFA975DA24uL,
         0x274CE38135A2FBF2uL}, /* 1456 */
        {0xCFE1DEB1CB129A73uL, 0xEE88235F52EBB448uL,
         0x4E99C7026B45F7E4uL}, /* 1457 */
        {0x9FC3BD63962534E7uL, 0xDD1046BEA5D76890uL,
         0x9D338E04D68BEFC8uL}, /* 1458 */
        {0x3F877AC72C4A69CFuL, 0xBA208D7D4BAED121uL,
         0x3A671C09AD17DF90uL}, /* 1459 */
        {0x7F0EF58E5894D39FuL, 0x74411AFA975DA242uL,
         0x74CE38135A2FBF20uL}, /* 1460 */
        {0xFE1DEB1CB129A73EuL, 0xE88235F52EBB4484uL,
         0xE99C7026B45F7E41uL}, /* 1461 */
        {0xFC3BD63962534E7DuL, 0xD1046BEA5D768909uL,
         0xD338E04D68BEFC82uL}, /* 1462 */
        {0xF877AC72C4A69CFBuL, 0xA208D7D4BAED1213uL,
         0xA671C09AD17DF904uL}, /* 1463 */
        {0xF0EF58E5894D39F7uL, 0x4411AFA975DA2427uL,
         0x4CE38135A2FBF209uL}, /* 1464 */
        {0xE1DEB1CB129A73EEuL, 0x88235F52EBB4484EuL,
         0x99C7026B45F7E413uL}, /* 1465 */
        {0xC3BD63962534E7DDuL, 0x1046BEA5D768909DuL,
         0x338E04D68BEFC827uL}, /* 1466 */
        {0x877AC72C4A69CFBAuL, 0x208D7D4BAED1213AuL,
         0x671C09AD17DF904EuL}, /* 1467 */
        {0x0EF58E5894D39F74uL, 0x411AFA975DA24274uL,
         0xCE38135A2FBF209CuL}, /* 1468 */
        {0x1DEB1CB129A73EE8uL, 0x8235F52EBB4484E9uL,
         0x9C7026B45F7E4139uL}, /* 1469 */
        {0x3BD63962534E7DD1uL, 0x046BEA5D768909D3uL,
         0x38E04D68BEFC8273uL}, /* 1470 */
        {0x77AC72C4A69CFBA2uL, 0x08D7D4BAED1213A6uL,
         0x71C09AD17DF904E6uL}, /* 1471 */
        {0xEF58E5894D39F744uL, 0x11AFA975DA24274CuL,
         0xE38135A2FBF209CCuL}, /* 1472 */
        {0xDEB1CB129A73EE88uL, 0x235F52EBB4484E99uL,
         0xC7026B45F7E41399uL}, /* 1473 */
        {0xBD63962534E7DD10uL, 0x46BEA5D768909D33uL,
         0x8E04D68BEFC82732uL}, /* 1474 */
        {0x7AC72C4A69CFBA20uL, 0x8D7D4BAED1213A67uL,
         0x1C09AD17DF904E64uL}, /* 1475 */
        {0xF58E5894D39F7441uL, 0x1AFA975DA24274CEuL,
         0x38135A2FBF209CC8uL}, /* 1476 */
        {0xEB1CB129A73EE882uL, 0x35F52EBB4484E99CuL,
         0x7026B45F7E413991uL}, /* 1477 */
        {0xD63962534E7DD104uL, 0x6BEA5D768909D338uL,
         0xE04D68BEFC827323uL}, /* 1478 */
        {0xAC72C4A69CFBA208uL, 0xD7D4BAED1213A671uL,
         0xC09AD17DF904E647uL}, /* 1479 */
        {0x58E5894D39F74411uL, 0xAFA975DA24274CE3uL,
         0x8135A2FBF209CC8EuL}, /* 1480 */
        {0xB1CB129A73EE8823uL, 0x5F52EBB4484E99C7uL,
         0x026B45F7E413991DuL}, /* 1481 */
        {0x63962534E7DD1046uL, 0xBEA5D768909D338EuL,
         0x04D68BEFC827323AuL}, /* 1482 */
        {0xC72C4A69CFBA208DuL, 0x7D4BAED1213A671CuL,
         0x09AD17DF904E6475uL}, /* 1483 */
        {0x8E5894D39F74411AuL, 0xFA975DA24274CE38uL,
         0x135A2FBF209CC8EBuL}, /* 1484 */
        {0x1CB129A73EE88235uL, 0xF52EBB4484E99C70uL,
         0x26B45F7E413991D6uL}, /* 1485 */
        {0x3962534E7DD1046BuL, 0xEA5D768909D338E0uL,
         0x4D68BEFC827323ACuL}, /* 1486 */
        {0x72C4A69CFBA208D7uL, 0xD4BAED1213A671C0uL,
         0x9AD17DF904E64758uL}, /* 1487 */
        {0xE5894D39F74411AFuL, 0xA975DA24274CE381uL,
         0x35A2FBF209CC8EB1uL}, /* 1488 */
        {0xCB129A73EE88235FuL, 0x52EBB4484E99C702uL,
         0x6B45F7E413991D63uL}, /* 1489 */
        {0x962534E7DD1046BEuL, 0xA5D768909D338E04uL,
         0xD68BEFC827323AC7uL}, /* 1490 */
        {0x2C4A69CFBA208D7DuL, 0x4BAED1213A671C09uL,
         0xAD17DF904E64758EuL}, /* 1491 */
        {0x5894D39F74411AFAuL, 0x975DA24274CE3813uL,
         0x5A2FBF209CC8EB1CuL}, /* 1492 */
        {0xB129A73EE88235F5uL, 0x2EBB4484E99C7026uL,
         0xB45F7E413991D639uL}, /* 1493 */
        {0x62534E7DD1046BEAuL, 0x5D768909D338E04DuL,
         0x68BEFC827323AC73uL}, /* 1494 */
        {0xC4A69CFBA208D7D4uL, 0xBAED1213A671C09AuL,
         0xD17DF904E64758E6uL}, /* 1495 */
        {0x894D39F74411AFA9uL, 0x75DA24274CE38135uL,
         0xA2FBF209CC8EB1CCuL}, /* 1496 */
        {0x129A73EE88235F52uL, 0xEBB4484E99C7026BuL,
         0x45F7E413991D6398uL}, /* 1497 */
        {0x2534E7DD1046BEA5uL, 0xD768909D338E04D6uL,
         0x8BEFC827323AC730uL}, /* 1498 */
        {0x4A69CFBA208D7D4BuL, 0xAED1213A671C09ADuL,
         0x17DF904E64758E60uL}, /* 1499 */
        {0x94D39F74411AFA97uL, 0x5DA24274CE38135AuL,
         0x2FBF209CC8EB1CC1uL}, /* 1500 */
        {0x29A73EE88235F52EuL, 0xBB4484E99C7026B4uL,
         0x5F7E413991D63983uL}, /* 1501 */
        {0x534E7DD1046BEA5DuL, 0x768909D338E04D68uL,
         0xBEFC827323AC7306uL}, /* 1502 */
        {0xA69CFBA208D7D4BAuL, 0xED1213A671C09AD1uL,
         0x7DF904E64758E60DuL}, /* 1503 */
        {0x4D39F74411AFA975uL, 0xDA24274CE38135A2uL,
         0xFBF209CC8EB1CC1AuL}, /* 1504 */
        {0x9A73EE88235F52EBuL, 0xB4484E99C7026B45uL,
         0xF7E413991D639835uL}, /* 1505 */
        {0x34E7DD1046BEA5D7uL, 0x68909D338E04D68BuL,
         0xEFC827323AC7306AuL}, /* 1506 */
        {0x69CFBA208D7D4BAEuL, 0xD1213A671C09AD17uL,
         0xDF904E64758E60D4uL}, /* 1507 */
        {0xD39F74411AFA975DuL, 0xA24274CE38135A2FuL,
         0xBF209CC8EB1CC1A9uL}, /* 1508 */
        {0xA73EE88235F52EBBuL, 0x4484E99C7026B45FuL,
         0x7E413991D6398353uL}, /* 1509 */
        {0x4E7DD1046BEA5D76uL, 0x8909D338E04D68BEuL,
         0xFC827323AC7306A6uL}, /* 1510 */
        {0x9CFBA208D7D4BAEDuL, 0x1213A671C09AD17DuL,
         0xF904E64758E60D4CuL}, /* 1511 */
        {0x39F74411AFA975DAuL, 0x24274CE38135A2FBuL,
         0xF209CC8EB1CC1A99uL}, /* 1512 */
        {0x73EE88235F52EBB4uL, 0x484E99C7026B45F7uL,
         0xE413991D63983533uL}, /* 1513 */
        {0xE7DD1046BEA5D768uL, 0x909D338E04D68BEFuL,
         0xC827323AC7306A67uL}, /* 1514 */
        {0xCFBA208D7D4BAED1uL, 0x213A671C09AD17DFuL,
         0x904E64758E60D4CEuL}, /* 1515 */
        {0x9F74411AFA975DA2uL, 0x4274CE38135A2FBFuL,
         0x209CC8EB1CC1A99CuL}, /* 1516 */
        {0x3EE88235F52EBB44uL, 0x84E99C7026B45F7EuL,
         0x413991D639835339uL}, /* 1517 */
        {0x7DD1046BEA5D7689uL, 0x09D338E04D68BEFCuL,
         0x827323AC7306A673uL}, /* 1518 */
        {0xFBA208D7D4BAED12uL, 0x13A671C09AD17DF9uL,
         0x04E64758E60D4CE7uL}, /* 1519 */
        {0xF74411AFA975DA24uL, 0x274CE38135A2FBF2uL,
         0x09CC8EB1CC1A99CFuL}, /* 1520 */
        {0xEE88235F52EBB448uL, 0x4E99C7026B45F7E4uL,
         0x13991D639835339FuL}, /* 1521 */
        {0xDD1046BEA5D76890uL, 0x9D338E04D68BEFC8uL,
         0x27323AC7306A673EuL}, /* 1522 */
        {0xBA208D7D4BAED121uL, 0x3A671C09AD17DF90uL,
         0x4E64758E60D4CE7DuL}, /* 1523 */
        {0x74411AFA975DA242uL, 0x74CE38135A2FBF20uL,
         0x9CC8EB1CC1A99CFAuL}, /* 1524 */
        {0xE88235F52EBB4484uL, 0xE99C7026B45F7E41uL,
         0x3991D639835339F4uL}, /* 1525 */
        {0xD1046BEA5D768909uL, 0xD338E04D68BEFC82uL,
         0x7323AC7306A673E9uL}, /* 1526 */
        {0xA208D7D4BAED1213uL, 0xA671C09AD17DF904uL,
         0xE64758E60D4CE7D2uL}, /* 1527 */
        {0x4411AFA975DA2427uL, 0x4CE38135A2FBF209uL,
         0xCC8EB1CC1A99CFA4uL}, /* 1528 */
        {0x88235F52EBB4484EuL, 0x99C7026B45F7E413uL,
         0x991D639835339F49uL}, /* 1529 */
        {0x1046BEA5D768909DuL, 0x338E04D68BEFC827uL,
         0x323AC7306A673E93uL}, /* 1530 */
        {0x208D7D4BAED1213AuL, 0x671C09AD17DF904EuL,
         0x64758E60D4CE7D27uL}, /* 1531 */
        {0x411AFA975DA24274uL, 0xCE38135A2FBF209CuL,
         0xC8EB1CC1A99CFA4EuL}, /* 1532 */
        {0x8235F52EBB4484E9uL, 0x9C7026B45F7E4139uL,
         0x91D639835339F49CuL}, /* 1533 */
        {0x046BEA5D768909D3uL, 0x38E04D68BEFC8273uL,
         0x23AC7306A673E939uL}, /* 1534 */
        {0x08D7D4BAED1213A6uL, 0x71C09AD17DF904E6uL,
         0x4758E60D4CE7D272uL}, /* 1535 */
        {0x11AFA975DA24274CuL, 0xE38135A2FBF209CCuL,
         0x8EB1CC1A99CFA4E4uL}, /* 1536 */
        {0x235F52EBB4484E99uL, 0xC7026B45F7E41399uL,
         0x1D639835339F49C8uL}, /* 1537 */
        {0x46BEA5D768909D33uL, 0x8E04D68BEFC82732uL,
         0x3AC7306A673E9390uL}, /* 1538 */
        {0x8D7D4BAED1213A67uL, 0x1C09AD17DF904E64uL,
         0x758E60D4CE7D2721uL}, /* 1539 */
        {0x1AFA975DA24274CEuL, 0x38135A2FBF209CC8uL,
         0xEB1CC1A99CFA4E42uL}, /* 1540 */
        {0x35F52EBB4484E99CuL, 0x7026B45F7E413991uL,
         0xD639835339F49C84uL}, /* 1541 */
        {0x6BEA5D768909D338uL, 0xE04D68BEFC827323uL,
         0xAC7306A673E93908uL}, /* 1542 */
        {0xD7D4BAED1213A671uL, 0xC09AD17DF904E647uL,
         0x58E60D4CE7D27211uL}, /* 1543 */
        {0xAFA975DA24274CE3uL, 0x8135A2FBF209CC8EuL,
         0xB1CC1A99CFA4E422uL}, /* 1544 */
        {0x5F52EBB4484E99C7uL, 0x026B45F7E413991DuL,
         0x639835339F49C845uL}, /* 1545 */
        {0xBEA5D768909D338EuL, 0x04D68BEFC827323AuL,
         0xC7306A673E93908BuL}, /* 1546 */
        {0x7D4BAED1213A671CuL, 0x09AD17DF904E6475uL,
         0x8E60D4CE7D272117uL}, /* 1547 */
        {0xFA975DA24274CE38uL, 0x135A2FBF209CC8EBuL,
         0x1CC1A99CFA4E422FuL}, /* 1548 */
        {0xF52EBB4484E99C70uL, 0x26B45F7E413991D6uL,
         0x39835339F49C845FuL}, /* 1549 */
        {0xEA5D768909D338E0uL, 0x4D68BEFC827323ACuL,
         0x7306A673E93908BFuL}, /* 1550 */
        {0xD4BAED1213A671C0uL, 0x9AD17DF904E64758uL,
         0xE60D4CE7D272117EuL}, /* 1551 */
        {0xA975DA24274CE381uL, 0x35A2FBF209CC8EB1uL,
         0xCC1A99CFA4E422FCuL}, /* 1552 */
        {0x52EBB4484E99C702uL, 0x6B45F7E413991D63uL,
         0x9835339F49C845F8uL}, /* 1553 */
        {0xA5D768909D338E04uL, 0xD68BEFC827323AC7uL,
         0x306A673E93908BF1uL}, /* 1554 */
        {0x4BAED1213A671C09uL, 0xAD17DF904E64758EuL,
         0x60D4CE7D272117E2uL}, /* 1555 */
        {0x975DA24274CE3813uL, 0x5A2FBF209CC8EB1CuL,
         0xC1A99CFA4E422FC5uL}, /* 1556 */
        {0x2EBB4484E99C7026uL, 0xB45F7E413991D639uL,
         0x835339F49C845F8BuL}, /* 1557 */
        {0x5D768909D338E04DuL, 0x68BEFC827323AC73uL,
         0x06A673E93908BF17uL}, /* 1558 */
        {0xBAED1213A671C09AuL, 0xD17DF904E64758E6uL,
         0x0D4CE7D272117E2EuL}, /* 1559 */
        {0x75DA24274CE38135uL, 0xA2FBF209CC8EB1CCuL,
         0x1A99CFA4E422FC5DuL}, /* 1560 */
        {0xEBB4484E99C7026BuL, 0x45F7E413991D6398uL,
         0x35339F49C845F8BBuL}, /* 1561 */
        {0xD768909D338E04D6uL, 0x8BEFC827323AC730uL,
         0x6A673E93908BF177uL}, /* 1562 */
        {0xAED1213A671C09ADuL, 0x17DF904E64758E60uL,
         0xD4CE7D272117E2EFuL}, /* 1563 */
        {0x5DA24274CE38135AuL, 0x2FBF209CC8EB1CC1uL,
         0xA99CFA4E422FC5DEuL}, /* 1564 */
        {0xBB4484E99C7026B4uL, 0x5F7E413991D63983uL,
         0x5339F49C845F8BBDuL}, /* 1565 */
        {0x768909D338E04D68uL, 0xBEFC827323AC7306uL,
         0xA673E93908BF177BuL}, /* 1566 */
        {0xED1213A671C09AD1uL, 0x7DF904E64758E60DuL,
         0x4CE7D272117E2EF7uL}, /* 1567 */
        {0xDA24274CE38135A2uL, 0xFBF209CC8EB1CC1AuL,
         0x99CFA4E422FC5DEFuL}, /* 1568 */
        {0xB4484E99C7026B45uL, 0xF7E413991D639835uL,
         0x339F49C845F8BBDFuL}, /* 1569 */
        {0x68909D338E04D68BuL, 0xEFC827323AC7306AuL,
         0x673E93908BF177BFuL}, /* 1570 */
        {0xD1213A671C09AD17uL, 0xDF904E64758E60D4uL,
         0xCE7D272117E2EF7EuL}, /* 1571 */
        {0xA24274CE38135A2FuL, 0xBF209CC8EB1CC1A9uL,
         0x9CFA4E422FC5DEFCuL}, /* 1572 */
        {0x4484E99C7026B45FuL, 0x7E413991D6398353uL,
         0x39F49C845F8BBDF9uL}, /* 1573 */
        {0x8909D338E04D68BEuL, 0xFC827323AC7306A6uL,
         0x73E93908BF177BF2uL}, /* 1574 */
        {0x1213A671C09AD17DuL, 0xF904E64758E60D4CuL,
         0xE7D272117E2EF7E4uL}, /* 1575 */
        {0x24274CE38135A2FBuL, 0xF209CC8EB1CC1A99uL,
         0xCFA4E422FC5DEFC9uL}, /* 1576 */
        {0x484E99C7026B45F7uL, 0xE413991D63983533uL,
         0x9F49C845F8BBDF92uL}, /* 1577 */
        {0x909D338E04D68BEFuL, 0xC827323AC7306A67uL,
         0x3E93908BF177BF25uL}, /* 1578 */
        {0x213A671C09AD17DFuL, 0x904E64758E60D4CEuL,
         0x7D272117E2EF7E4AuL}, /* 1579 */
        {0x4274CE38135A2FBFuL, 0x209CC8EB1CC1A99CuL,
         0xFA4E422FC5DEFC94uL}, /* 1580 */
        {0x84E99C7026B45F7EuL, 0x413991D639835339uL,
         0xF49C845F8BBDF928uL}, /* 1581 */
        {0x09D338E04D68BEFCuL, 0x827323AC7306A673uL,
         0xE93908BF177BF250uL}, /* 1582 */
        {0x13A671C09AD17DF9uL, 0x04E64758E60D4CE7uL,
         0xD272117E2EF7E4A0uL}, /* 1583 */
        {0x274CE38135A2FBF2uL, 0x09CC8EB1CC1A99CFuL,
         0xA4E422FC5DEFC941uL}, /* 1584 */
        {0x4E99C7026B45F7E4uL, 0x13991D639835339FuL,
         0x49C845F8BBDF9283uL}, /* 1585 */
        {0x9D338E04D68BEFC8uL, 0x27323AC7306A673EuL,
         0x93908BF177BF2507uL}, /* 1586 */
        {0x3A671C09AD17DF90uL, 0x4E64758E60D4CE7DuL,
         0x272117E2EF7E4A0EuL}, /* 1587 */
        {0x74CE38135A2FBF20uL, 0x9CC8EB1CC1A99CFAuL,
         0x4E422FC5DEFC941DuL}, /* 1588 */
        {0xE99C7026B45F7E41uL, 0x3991D639835339F4uL,
         0x9C845F8BBDF9283BuL}, /* 1589 */
        {0xD338E04D68BEFC82uL, 0x7323AC7306A673E9uL,
         0x3908BF177BF25076uL}, /* 1590 */
        {0xA671C09AD17DF904uL, 0xE64758E60D4CE7D2uL,
         0x72117E2EF7E4A0ECuL}, /* 1591 */
        {0x4CE38135A2FBF209uL, 0xCC8EB1CC1A99CFA4uL,
         0xE422FC5DEFC941D8uL}, /* 1592 */
        {0x99C7026B45F7E413uL, 0x991D639835339F49uL,
         0xC845F8BBDF9283B1uL}, /* 1593 */
        {0x338E04D68BEFC827uL, 0x323AC7306A673E93uL,
         0x908BF177BF250763uL}, /* 1594 */
        {0x671C09AD17DF904EuL, 0x64758E60D4CE7D27uL,
         0x2117E2EF7E4A0EC7uL}, /* 1595 */
        {0xCE38135A2FBF209CuL, 0xC8EB1CC1A99CFA4EuL,
         0x422FC5DEFC941D8FuL}, /* 1596 */
        {0x9C7026B45F7E4139uL, 0x91D639835339F49CuL,
         0x845F8BBDF9283B1FuL}, /* 1597 */
        {0x38E04D68BEFC8273uL, 0x23AC7306A673E939uL,
         0x08BF177BF250763FuL}, /* 1598 */
        {0x71C09AD17DF904E6uL, 0x4758E60D4CE7D272uL,
         0x117E2EF7E4A0EC7FuL}, /* 1599 */
        {0xE38135A2FBF209CCuL, 0x8EB1CC1A99CFA4E4uL,
         0x22FC5DEFC941D8FFuL}, /* 1600 */
        {0xC7026B45F7E41399uL, 0x1D639835339F49C8uL,
         0x45F8BBDF9283B1FFuL}, /* 1601 */
        {0x8E04D68BEFC82732uL, 0x3AC7306A673E9390uL,
         0x8BF177BF250763FFuL}, /* 1602 */
        {0x1C09AD17DF904E64uL, 0x758E60D4CE7D2721uL,
         0x17E2EF7E4A0EC7FEuL}, /* 1603 */
        {0x38135A2FBF209CC8uL, 0xEB1CC1A99CFA4E42uL,
         0x2FC5DEFC941D8FFCuL}, /* 1604 */
        {0x7026B45F7E413991uL, 0xD639835339F49C84uL,
         0x5F8BBDF9283B1FF8uL}, /* 1605 */
        {0xE04D68BEFC827323uL, 0xAC7306A673E93908uL,
         0xBF177BF250763FF1uL}, /* 1606 */
        {0xC09AD17DF904E647uL, 0x58E60D4CE7D27211uL,
         0x7E2EF7E4A0EC7FE2uL}, /* 1607 */
        {0x8135A2FBF209CC8EuL, 0xB1CC1A99CFA4E422uL,
         0xFC5DEFC941D8FFC4uL}, /* 1608 */
        {0x026B45F7E413991DuL, 0x639835339F49C845uL,
         0xF8BBDF9283B1FF89uL}, /* 1609 */
        {0x04D68BEFC827323AuL, 0xC7306A673E93908BuL,
         0xF177BF250763FF12uL}, /* 1610 */
        {0x09AD17DF904E6475uL, 0x8E60D4CE7D272117uL,
         0xE2EF7E4A0EC7FE25uL}, /* 1611 */
        {0x135A2FBF209CC8EBuL, 0x1CC1A99CFA4E422FuL,
         0xC5DEFC941D8FFC4BuL}, /* 1612 */
        {0x26B45F7E413991D6uL, 0x39835339F49C845FuL,
         0x8BBDF9283B1FF897uL}, /* 1613 */
        {0x4D68BEFC827323ACuL, 0x7306A673E93908BFuL,
         0x177BF250763FF12FuL}, /* 1614 */
        {0x9AD17DF904E64758uL, 0xE60D4CE7D272117EuL,
         0x2EF7E4A0EC7FE25FuL}, /* 1615 */
        {0x35A2FBF209CC8EB1uL, 0xCC1A99CFA4E422FCuL,
         0x5DEFC941D8FFC4BFuL}, /* 1616 */
        {0x6B45F7E413991D63uL, 0x9835339F49C845F8uL,
         0xBBDF9283B1FF897FuL}, /* 1617 */
        {0xD68BEFC827323AC7uL, 0x306A673E93908BF1uL,
         0x77BF250763FF12FFuL}, /* 1618 */
        {0xAD17DF904E64758EuL, 0x60D4CE7D272117E2uL,
         0xEF7E4A0EC7FE25FFuL}, /* 1619 */
        {0x5A2FBF209CC8EB1CuL, 0xC1A99CFA4E422FC5uL,
         0xDEFC941D8FFC4BFFuL}, /* 1620 */
        {0xB45F7E413991D639uL, 0x835339F49C845F8BuL,
         0xBDF9283B1FF897FFuL}, /* 1621 */
        {0x68BEFC827323AC73uL, 0x06A673E93908BF17uL,
         0x7BF250763FF12FFFuL}, /* 1622 */
        {0xD17DF904E64758E6uL, 0x0D4CE7D272117E2EuL,
         0xF7E4A0EC7FE25FFFuL}, /* 1623 */
        {0xA2FBF209CC8EB1CCuL, 0x1A99CFA4E422FC5DuL,
         0xEFC941D8FFC4BFFEuL}, /* 1624 */
        {0x45F7E413991D6398uL, 0x35339F49C845F8BBuL,
         0xDF9283B1FF897FFDuL}, /* 1625 */
        {0x8BEFC827323AC730uL, 0x6A673E93908BF177uL,
         0xBF250763FF12FFFBuL}, /* 1626 */
        {0x17DF904E64758E60uL, 0xD4CE7D272117E2EFuL,
         0x7E4A0EC7FE25FFF7uL}, /* 1627 */
        {0x2FBF209CC8EB1CC1uL, 0xA99CFA4E422FC5DEuL,
         0xFC941D8FFC4BFFEFuL}, /* 1628 */
        {0x5F7E413991D63983uL, 0x5339F49C845F8BBDuL,
         0xF9283B1FF897FFDEuL}, /* 1629 */
        {0xBEFC827323AC7306uL, 0xA673E93908BF177BuL,
         0xF250763FF12FFFBCuL}, /* 1630 */
        {0x7DF904E64758E60DuL, 0x4CE7D272117E2EF7uL,
         0xE4A0EC7FE25FFF78uL}, /* 1631 */
        {0xFBF209CC8EB1CC1AuL, 0x99CFA4E422FC5DEFuL,
         0xC941D8FFC4BFFEF0uL}, /* 1632 */
        {0xF7E413991D639835uL, 0x339F49C845F8BBDFuL,
         0x9283B1FF897FFDE0uL}, /* 1633 */
        {0xEFC827323AC7306AuL, 0x673E93908BF177BFuL,
         0x250763FF12FFFBC0uL}, /* 1634 */
        {0xDF904E64758E60D4uL, 0xCE7D272117E2EF7EuL,
         0x4A0EC7FE25FFF781uL}, /* 1635 */
        {0xBF209CC8EB1CC1A9uL, 0x9CFA4E422FC5DEFCuL,
         0x941D8FFC4BFFEF02uL}, /* 1636 */
        {0x7E413991D6398353uL, 0x39F49C845F8BBDF9uL,
         0x283B1FF897FFDE05uL}, /* 1637 */
        {0xFC827323AC7306A6uL, 0x73E93908BF177BF2uL,
         0x50763FF12FFFBC0BuL}, /* 1638 */
        {0xF904E64758E60D4CuL, 0xE7D272117E2EF7E4uL,
         0xA0EC7FE25FFF7816uL}, /* 1639 */
        {0xF209CC8EB1CC1A99uL, 0xCFA4E422FC5DEFC9uL,
         0x41D8FFC4BFFEF02CuL}, /* 1640 */
        {0xE413991D63983533uL, 0x9F49C845F8BBDF92uL,
         0x83B1FF897FFDE059uL}, /* 1641 */
        {0xC827323AC7306A67uL, 0x3E93908BF177BF25uL,
         0x0763FF12FFFBC0B3uL}, /* 1642 */
        {0x904E64758E60D4CEuL, 0x7D272117E2EF7E4AuL,
         0x0EC7FE25FFF78166uL}, /* 1643 */
        {0x209CC8EB1CC1A99CuL, 0xFA4E422FC5DEFC94uL,
         0x1D8FFC4BFFEF02CCuL}, /* 1644 */
        {0x413991D639835339uL, 0xF49C845F8BBDF928uL,
         0x3B1FF897FFDE0598uL}, /* 1645 */
        {0x827323AC7306A673uL, 0xE93908BF177BF250uL,
         0x763FF12FFFBC0B30uL}, /* 1646 */
        {0x04E64758E60D4CE7uL, 0xD272117E2EF7E4A0uL,
         0xEC7FE25FFF781660uL}, /* 1647 */
        {0x09CC8EB1CC1A99CFuL, 0xA4E422FC5DEFC941uL,
         0xD8FFC4BFFEF02CC0uL}, /* 1648 */
        {0x13991D639835339FuL, 0x49C845F8BBDF9283uL,
         0xB1FF897FFDE05980uL}, /* 1649 */
        {0x27323AC7306A673EuL, 0x93908BF177BF2507uL,
         0x63FF12FFFBC0B301uL}, /* 1650 */
        {0x4E64758E60D4CE7DuL, 0x272117E2EF7E4A0EuL,
         0xC7FE25FFF7816603uL}, /* 1651 */
        {0x9CC8EB1CC1A99CFAuL, 0x4E422FC5DEFC941DuL,
         0x8FFC4BFFEF02CC07uL}, /* 1652 */
        {0x3991D639835339F4uL, 0x9C845F8BBDF9283BuL,
         0x1FF897FFDE05980FuL}, /* 1653 */
        {0x7323AC7306A673E9uL, 0x3908BF177BF25076uL,
         0x3FF12FFFBC0B301FuL}, /* 1654 */
        {0xE64758E60D4CE7D2uL, 0x72117E2EF7E4A0ECuL,
         0x7FE25FFF7816603FuL}, /* 1655 */
        {0xCC8EB1CC1A99CFA4uL, 0xE422FC5DEFC941D8uL,
         0xFFC4BFFEF02CC07FuL}, /* 1656 */
        {0x991D639835339F49uL, 0xC845F8BBDF9283B1uL,
         0xFF897FFDE05980FEuL}, /* 1657 */
        {0x323AC7306A673E93uL, 0x908BF177BF250763uL,
         0xFF12FFFBC0B301FDuL}, /* 1658 */
        {0x64758E60D4CE7D27uL, 0x2117E2EF7E4A0EC7uL,
         0xFE25FFF7816603FBuL}, /* 1659 */
        {0xC8EB1CC1A99CFA4EuL, 0x422FC5DEFC941D8FuL,
         0xFC4BFFEF02CC07F7uL}, /* 1660 */
        {0x91D639835339F49CuL, 0x845F8BBDF9283B1FuL,
         0xF897FFDE05980FEFuL}, /* 1661 */
        {0x23AC7306A673E939uL, 0x08BF177BF250763FuL,
         0xF12FFFBC0B301FDEuL}, /* 1662 */
        {0x4758E60D4CE7D272uL, 0x117E2EF7E4A0EC7FuL,
         0xE25FFF7816603FBCuL}, /* 1663 */
        {0x8EB1CC1A99CFA4E4uL, 0x22FC5DEFC941D8FFuL,
         0xC4BFFEF02CC07F79uL}, /* 1664 */
        {0x1D639835339F49C8uL, 0x45F8BBDF9283B1FFuL,
         0x897FFDE05980FEF2uL}, /* 1665 */
        {0x3AC7306A673E9390uL, 0x8BF177BF250763FFuL,
         0x12FFFBC0B301FDE5uL}, /* 1666 */
        {0x758E60D4CE7D2721uL, 0x17E2EF7E4A0EC7FEuL,
         0x25FFF7816603FBCBuL}, /* 1667 */
        {0xEB1CC1A99CFA4E42uL, 0x2FC5DEFC941D8FFCuL,
         0x4BFFEF02CC07F797uL}, /* 1668 */
        {0xD639835339F49C84uL, 0x5F8BBDF9283B1FF8uL,
         0x97FFDE05980FEF2FuL}, /* 1669 */
        {0xAC7306A673E93908uL, 0xBF177BF250763FF1uL,
         0x2FFFBC0B301FDE5EuL}, /* 1670 */
        {0x58E60D4CE7D27211uL, 0x7E2EF7E4A0EC7FE2uL,
         0x5FFF7816603FBCBCuL}, /* 1671 */
        {0xB1CC1A99CFA4E422uL, 0xFC5DEFC941D8FFC4uL,
         0xBFFEF02CC07F7978uL}, /* 1672 */
        {0x639835339F49C845uL, 0xF8BBDF9283B1FF89uL,
         0x7FFDE05980FEF2F1uL}, /* 1673 */
        {0xC7306A673E93908BuL, 0xF177BF250763FF12uL,
         0xFFFBC0B301FDE5E2uL}, /* 1674 */
        {0x8E60D4CE7D272117uL, 0xE2EF7E4A0EC7FE25uL,
         0xFFF7816603FBCBC4uL}, /* 1675 */
        {0x1CC1A99CFA4E422FuL, 0xC5DEFC941D8FFC4BuL,
         0xFFEF02CC07F79788uL}, /* 1676 */
        {0x39835339F49C845FuL, 0x8BBDF9283B1FF897uL,
         0xFFDE05980FEF2F11uL}, /* 1677 */
        {0x7306A673E93908BFuL, 0x177BF250763FF12FuL,
         0xFFBC0B301FDE5E23uL}, /* 1678 */
        {0xE60D4CE7D272117EuL, 0x2EF7E4A0EC7FE25FuL,
         0xFF7816603FBCBC46uL}, /* 1679 */
        {0xCC1A99CFA4E422FCuL, 0x5DEFC941D8FFC4BFuL,
         0xFEF02CC07F79788CuL}, /* 1680 */
        {0x9835339F49C845F8uL, 0xBBDF9283B1FF897FuL,
         0xFDE05980FEF2F118uL}, /* 1681 */
        {0x306A673E93908BF1uL, 0x77BF250763FF12FFuL,
         0xFBC0B301FDE5E231uL}, /* 1682 */
        {0x60D4CE7D272117E2uL, 0xEF7E4A0EC7FE25FFuL,
         0xF7816603FBCBC462uL}, /* 1683 */
        {0xC1A99CFA4E422FC5uL, 0xDEFC941D8FFC4BFFuL,
         0xEF02CC07F79788C5uL}, /* 1684 */
        {0x835339F49C845F8BuL, 0xBDF9283B1FF897FFuL,
         0xDE05980FEF2F118BuL}, /* 1685 */
        {0x06A673E93908BF17uL, 0x7BF250763FF12FFFuL,
         0xBC0B301FDE5E2316uL}, /* 1686 */
        {0x0D4CE7D272117E2EuL, 0xF7E4A0EC7FE25FFFuL,
         0x7816603FBCBC462DuL}, /* 1687 */
        {0x1A99CFA4E422FC5DuL, 0xEFC941D8FFC4BFFEuL,
         0xF02CC07F79788C5AuL}, /* 1688 */
        {0x35339F49C845F8BBuL, 0xDF9283B1FF897FFDuL,
         0xE05980FEF2F118B5uL}, /* 1689 */
        {0x6A673E93908BF177uL, 0xBF250763FF12FFFBuL,
         0xC0B301FDE5E2316BuL}, /* 1690 */
        {0xD4CE7D272117E2EFuL, 0x7E4A0EC7FE25FFF7uL,
         0x816603FBCBC462D6uL}, /* 1691 */
        {0xA99CFA4E422FC5DEuL, 0xFC941D8FFC4BFFEFuL,
         0x02CC07F79788C5ADuL}, /* 1692 */
        {0x5339F49C845F8BBDuL, 0xF9283B1FF897FFDEuL,
         0x05980FEF2F118B5AuL}, /* 1693 */
        {0xA673E93908BF177BuL, 0xF250763FF12FFFBCuL,
         0x0B301FDE5E2316B4uL}, /* 1694 */
        {0x4CE7D272117E2EF7uL, 0xE4A0EC7FE25FFF78uL,
         0x16603FBCBC462D68uL}, /* 1695 */
        {0x99CFA4E422FC5DEFuL, 0xC941D8FFC4BFFEF0uL,
         0x2CC07F79788C5AD0uL}, /* 1696 */
        {0x339F49C845F8BBDFuL, 0x9283B1FF897FFDE0uL,
         0x5980FEF2F118B5A0uL}, /* 1697 */
        {0x673E93908BF177BFuL, 0x250763FF12FFFBC0uL,
         0xB301FDE5E2316B41uL}, /* 1698 */
        {0xCE7D272117E2EF7EuL, 0x4A0EC7FE25FFF781uL,
         0x6603FBCBC462D682uL}, /* 1699 */
        {0x9CFA4E422FC5DEFCuL, 0x941D8FFC4BFFEF02uL,
         0xCC07F79788C5AD05uL}, /* 1700 */
        {0x39F49C845F8BBDF9uL, 0x283B1FF897FFDE05uL,
         0x980FEF2F118B5A0AuL}, /* 1701 */
        {0x73E93908BF177BF2uL, 0x50763FF12FFFBC0BuL,
         0x301FDE5E2316B414uL}, /* 1702 */
        {0xE7D272117E2EF7E4uL, 0xA0EC7FE25FFF7816uL,
         0x603FBCBC462D6829uL}, /* 1703 */
        {0xCFA4E422FC5DEFC9uL, 0x41D8FFC4BFFEF02CuL,
         0xC07F79788C5AD053uL}, /* 1704 */
        {0x9F49C845F8BBDF92uL, 0x83B1FF897FFDE059uL,
         0x80FEF2F118B5A0A6uL}, /* 1705 */
        {0x3E93908BF177BF25uL, 0x0763FF12FFFBC0B3uL,
         0x01FDE5E2316B414DuL}, /* 1706 */
        {0x7D272117E2EF7E4AuL, 0x0EC7FE25FFF78166uL,
         0x03FBCBC462D6829BuL}, /* 1707 */
        {0xFA4E422FC5DEFC94uL, 0x1D8FFC4BFFEF02CCuL,
         0x07F79788C5AD0536uL}, /* 1708 */
        {0xF49C845F8BBDF928uL, 0x3B1FF897FFDE0598uL,
         0x0FEF2F118B5A0A6DuL}, /* 1709 */
        {0xE93908BF177BF250uL, 0x763FF12FFFBC0B30uL,
         0x1FDE5E2316B414DAuL}, /* 1710 */
        {0xD272117E2EF7E4A0uL, 0xEC7FE25FFF781660uL,
         0x3FBCBC462D6829B4uL}, /* 1711 */
        {0xA4E422FC5DEFC941uL, 0xD8FFC4BFFEF02CC0uL,
         0x7F79788C5AD05368uL}, /* 1712 */
        {0x49C845F8BBDF9283uL, 0xB1FF897FFDE05980uL,
         0xFEF2F118B5A0A6D1uL}, /* 1713 */
        {0x93908BF177BF2507uL, 0x63FF12FFFBC0B301uL,
         0xFDE5E2316B414DA3uL}, /* 1714 */
        {0x272117E2EF7E4A0EuL, 0xC7FE25FFF7816603uL,
         0xFBCBC462D6829B47uL}, /* 1715 */
        {0x4E422FC5DEFC941DuL, 0x8FFC4BFFEF02CC07uL,
         0xF79788C5AD05368FuL}, /* 1716 */
        {0x9C845F8BBDF9283BuL, 0x1FF897FFDE05980FuL,
         0xEF2F118B5A0A6D1FuL}, /* 1717 */
        {0x3908BF177BF25076uL, 0x3FF12FFFBC0B301FuL,
         0xDE5E2316B414DA3EuL}, /* 1718 */
        {0x72117E2EF7E4A0ECuL, 0x7FE25FFF7816603FuL,
         0xBCBC462D6829B47DuL}, /* 1719 */
        {0xE422FC5DEFC941D8uL, 0xFFC4BFFEF02CC07FuL,
         0x79788C5AD05368FBuL}, /* 1720 */
        {0xC845F8BBDF9283B1uL, 0xFF897FFDE05980FEuL,
         0xF2F118B5A0A6D1F6uL}, /* 1721 */
        {0x908BF177BF250763uL, 0xFF12FFFBC0B301FDuL,
         0xE5E2316B414DA3EDuL}, /* 1722 */
        {0x2117E2EF7E4A0EC7uL, 0xFE25FFF7816603FBuL,
         0xCBC462D6829B47DBuL}, /* 1723 */
        {0x422FC5DEFC941D8FuL, 0xFC4BFFEF02CC07F7uL,
         0x9788C5AD05368FB6uL}, /* 1724 */
        {0x845F8BBDF9283B1FuL, 0xF897FFDE05980FEFuL,
         0x2F118B5A0A6D1F6DuL}, /* 1725 */
        {0x08BF177BF250763FuL, 0xF12FFFBC0B301FDEuL,
         0x5E2316B414DA3EDAuL}, /* 1726 */
        {0x117E2EF7E4A0EC7FuL, 0xE25FFF7816603FBCuL,
         0xBC462D6829B47DB4uL}, /* 1727 */
        {0x22FC5DEFC941D8FFuL, 0xC4BFFEF02CC07F79uL,
         0x788C5AD05368FB69uL}, /* 1728 */
        {0x45F8BBDF9283B1FFuL, 0x897FFDE05980FEF2uL,
         0xF118B5A0A6D1F6D3uL}, /* 1729 */
        {0x8BF177BF250763FFuL, 0x12FFFBC0B301FDE5uL,
         0xE2316B414DA3EDA6uL}, /* 1730 */
        {0x17E2EF7E4A0EC7FEuL, 0x25FFF7816603FBCBuL,
         0xC462D6829B47DB4DuL}, /* 1731 */
        {0x2FC5DEFC941D8FFCuL, 0x4BFFEF02CC07F797uL,
         0x88C5AD05368FB69BuL}, /* 1732 */
        {0x5F8BBDF9283B1FF8uL, 0x97FFDE05980FEF2FuL,
         0x118B5A0A6D1F6D36uL}, /* 1733 */
        {0xBF177BF250763FF1uL, 0x2FFFBC0B301FDE5EuL,
         0x2316B414DA3EDA6CuL}, /* 1734 */
        {0x7E2EF7E4A0EC7FE2uL, 0x5FFF7816603FBCBCuL,
         0x462D6829B47DB4D9uL}, /* 1735 */
        {0xFC5DEFC941D8FFC4uL, 0xBFFEF02CC07F7978uL,
         0x8C5AD05368FB69B3uL}, /* 1736 */
        {0xF8BBDF9283B1FF89uL, 0x7FFDE05980FEF2F1uL,
         0x18B5A0A6D1F6D367uL}, /* 1737 */
        {0xF177BF250763FF12uL, 0xFFFBC0B301FDE5E2uL,
         0x316B414DA3EDA6CFuL}, /* 1738 */
        {0xE2EF7E4A0EC7FE25uL, 0xFFF7816603FBCBC4uL,
         0x62D6829B47DB4D9FuL}, /* 1739 */
        {0xC5DEFC941D8FFC4BuL, 0xFFEF02CC07F79788uL,
         0xC5AD05368FB69B3FuL}, /* 1740 */
        {0x8BBDF9283B1FF897uL, 0xFFDE05980FEF2F11uL,
         0x8B5A0A6D1F6D367EuL}, /* 1741 */
        {0x177BF250763FF12FuL, 0xFFBC0B301FDE5E23uL,
         0x16B414DA3EDA6CFDuL}, /* 1742 */
        {0x2EF7E4A0EC7FE25FuL, 0xFF7816603FBCBC46uL,
         0x2D6829B47DB4D9FBuL}, /* 1743 */
        {0x5DEFC941D8FFC4BFuL, 0xFEF02CC07F79788CuL,
         0x5AD05368FB69B3F6uL}, /* 1744 */
        {0xBBDF9283B1FF897FuL, 0xFDE05980FEF2F118uL,
         0xB5A0A6D1F6D367ECuL}, /* 1745 */
        {0x77BF250763FF12FFuL, 0xFBC0B301FDE5E231uL,
         0x6B414DA3EDA6CFD9uL}, /* 1746 */
        {0xEF7E4A0EC7FE25FFuL, 0xF7816603FBCBC462uL,
         0xD6829B47DB4D9FB3uL}, /* 1747 */
        {0xDEFC941D8FFC4BFFuL, 0xEF02CC07F79788C5uL,
         0xAD05368FB69B3F67uL}, /* 1748 */
        {0xBDF9283B1FF897FFuL, 0xDE05980FEF2F118BuL,
         0x5A0A6D1F6D367ECFuL}, /* 1749 */
        {0x7BF250763FF12FFFuL, 0xBC0B301FDE5E2316uL,
         0xB414DA3EDA6CFD9EuL}, /* 1750 */
        {0xF7E4A0EC7FE25FFFuL, 0x7816603FBCBC462DuL,
         0x6829B47DB4D9FB3CuL}, /* 1751 */
        {0xEFC941D8FFC4BFFEuL, 0xF02CC07F79788C5AuL,
         0xD05368FB69B3F679uL}, /* 1752 */
        {0xDF9283B1FF897FFDuL, 0xE05980FEF2F118B5uL,
         0xA0A6D1F6D367ECF2uL}, /* 1753 */
        {0xBF250763FF12FFFBuL, 0xC0B301FDE5E2316BuL,
         0x414DA3EDA6CFD9E4uL}, /* 1754 */
        {0x7E4A0EC7FE25FFF7uL, 0x816603FBCBC462D6uL,
         0x829B47DB4D9FB3C9uL}, /* 1755 */
        {0xFC941D8FFC4BFFEFuL, 0x02CC07F79788C5ADuL,
         0x05368FB69B3F6793uL}, /* 1756 */
        {0xF9283B1FF897FFDEuL, 0x05980FEF2F118B5AuL,
         0x0A6D1F6D367ECF27uL}, /* 1757 */
        {0xF250763FF12FFFBCuL, 0x0B301FDE5E2316B4uL,
         0x14DA3EDA6CFD9E4FuL}, /* 1758 */
        {0xE4A0EC7FE25FFF78uL, 0x16603FBCBC462D68uL,
         0x29B47DB4D9FB3C9FuL}, /* 1759 */
        {0xC941D8FFC4BFFEF0uL, 0x2CC07F79788C5AD0uL,
         0x5368FB69B3F6793EuL}, /* 1760 */
        {0x9283B1FF897FFDE0uL, 0x5980FEF2F118B5A0uL,
         0xA6D1F6D367ECF27CuL}, /* 1761 */
        {0x250763FF12FFFBC0uL, 0xB301FDE5E2316B41uL,
         0x4DA3EDA6CFD9E4F9uL}, /* 1762 */
        {0x4A0EC7FE25FFF781uL, 0x6603FBCBC462D682uL,
         0x9B47DB4D9FB3C9F2uL}, /* 1763 */
        {0x941D8FFC4BFFEF02uL, 0xCC07F79788C5AD05uL,
         0x368FB69B3F6793E5uL}, /* 1764 */
        {0x283B1FF897FFDE05uL, 0x980FEF2F118B5A0AuL,
         0x6D1F6D367ECF27CBuL}, /* 1765 */
        {0x50763FF12FFFBC0BuL, 0x301FDE5E2316B414uL,
         0xDA3EDA6CFD9E4F96uL}, /* 1766 */
        {0xA0EC7FE25FFF7816uL, 0x603FBCBC462D6829uL,
         0xB47DB4D9FB3C9F2CuL}, /* 1767 */
        {0x41D8FFC4BFFEF02CuL, 0xC07F79788C5AD053uL,
         0x68FB69B3F6793E58uL}, /* 1768 */
        {0x83B1FF897FFDE059uL, 0x80FEF2F118B5A0A6uL,
         0xD1F6D367ECF27CB0uL}, /* 1769 */
        {0x0763FF12FFFBC0B3uL, 0x01FDE5E2316B414DuL,
         0xA3EDA6CFD9E4F961uL}, /* 1770 */
        {0x0EC7FE25FFF78166uL, 0x03FBCBC462D6829BuL,
         0x47DB4D9FB3C9F2C2uL}, /* 1771 */
        {0x1D8FFC4BFFEF02CCuL, 0x07F79788C5AD0536uL,
         0x8FB69B3F6793E584uL}, /* 1772 */
        {0x3B1FF897FFDE0598uL, 0x0FEF2F118B5A0A6DuL,
         0x1F6D367ECF27CB09uL}, /* 1773 */
        {0x763FF12FFFBC0B30uL, 0x1FDE5E2316B414DAuL,
         0x3EDA6CFD9E4F9613uL}, /* 1774 */
        {0xEC7FE25FFF781660uL, 0x3FBCBC462D6829B4uL,
         0x7DB4D9FB3C9F2C26uL}, /* 1775 */
        {0xD8FFC4BFFEF02CC0uL, 0x7F79788C5AD05368uL,
         0xFB69B3F6793E584DuL}, /* 1776 */
        {0xB1FF897FFDE05980uL, 0xFEF2F118B5A0A6D1uL,
         0xF6D367ECF27CB09BuL}, /* 1777 */
        {0x63FF12FFFBC0B301uL, 0xFDE5E2316B414DA3uL,
         0xEDA6CFD9E4F96136uL}, /* 1778 */
        {0xC7FE25FFF7816603uL, 0xFBCBC462D6829B47uL,
         0xDB4D9FB3C9F2C26DuL}, /* 1779 */
        {0x8FFC4BFFEF02CC07uL, 0xF79788C5AD05368FuL,
         0xB69B3F6793E584DBuL}, /* 1780 */
        {0x1FF897FFDE05980FuL, 0xEF2F118B5A0A6D1FuL,
         0x6D367ECF27CB09B7uL}, /* 1781 */
        {0x3FF12FFFBC0B301FuL, 0xDE5E2316B414DA3EuL,
         0xDA6CFD9E4F96136EuL}, /* 1782 */
        {0x7FE25FFF7816603FuL, 0xBCBC462D6829B47DuL,
         0xB4D9FB3C9F2C26DDuL}, /* 1783 */
        {0xFFC4BFFEF02CC07FuL, 0x79788C5AD05368FBuL,
         0x69B3F6793E584DBAuL}, /* 1784 */
        {0xFF897FFDE05980FEuL, 0xF2F118B5A0A6D1F6uL,
         0xD367ECF27CB09B74uL}, /* 1785 */
        {0xFF12FFFBC0B301FDuL, 0xE5E2316B414DA3EDuL,
         0xA6CFD9E4F96136E9uL}, /* 1786 */
        {0xFE25FFF7816603FBuL, 0xCBC462D6829B47DBuL,
         0x4D9FB3C9F2C26DD3uL}, /* 1787 */
        {0xFC4BFFEF02CC07F7uL, 0x9788C5AD05368FB6uL,
         0x9B3F6793E584DBA7uL}, /* 1788 */
        {0xF897FFDE05980FEFuL, 0x2F118B5A0A6D1F6DuL,
         0x367ECF27CB09B74FuL}, /* 1789 */
        {0xF12FFFBC0B301FDEuL, 0x5E2316B414DA3EDAuL,
         0x6CFD9E4F96136E9EuL}, /* 1790 */
        {0xE25FFF7816603FBCuL, 0xBC462D6829B47DB4uL,
         0xD9FB3C9F2C26DD3DuL}, /* 1791 */
        {0xC4BFFEF02CC07F79uL, 0x788C5AD05368FB69uL,
         0xB3F6793E584DBA7AuL}, /* 1792 */
        {0x897FFDE05980FEF2uL, 0xF118B5A0A6D1F6D3uL,
         0x67ECF27CB09B74F4uL}, /* 1793 */
        {0x12FFFBC0B301FDE5uL, 0xE2316B414DA3EDA6uL,
         0xCFD9E4F96136E9E8uL}, /* 1794 */
        {0x25FFF7816603FBCBuL, 0xC462D6829B47DB4DuL,
         0x9FB3C9F2C26DD3D1uL}, /* 1795 */
        {0x4BFFEF02CC07F797uL, 0x88C5AD05368FB69BuL,
         0x3F6793E584DBA7A3uL}, /* 1796 */
        {0x97FFDE05980FEF2FuL, 0x118B5A0A6D1F6D36uL,
         0x7ECF27CB09B74F46uL}, /* 1797 */
        {0x2FFFBC0B301FDE5EuL, 0x2316B414DA3EDA6CuL,
         0xFD9E4F96136E9E8CuL}, /* 1798 */
        {0x5FFF7816603FBCBCuL, 0x462D6829B47DB4D9uL,
         0xFB3C9F2C26DD3D18uL}, /* 1799 */
        {0xBFFEF02CC07F7978uL, 0x8C5AD05368FB69B3uL,
         0xF6793E584DBA7A31uL}, /* 1800 */
        {0x7FFDE05980FEF2F1uL, 0x18B5A0A6D1F6D367uL,
         0xECF27CB09B74F463uL}, /* 1801 */
        {0xFFFBC0B301FDE5E2uL, 0x316B414DA3EDA6CFuL,
         0xD9E4F96136E9E8C7uL}, /* 1802 */
        {0xFFF7816603FBCBC4uL, 0x62D6829B47DB4D9FuL,
         0xB3C9F2C26DD3D18FuL}, /* 1803 */
        {0xFFEF02CC07F79788uL, 0xC5AD05368FB69B3FuL,
         0x6793E584DBA7A31FuL}, /* 1804 */
        {0xFFDE05980FEF2F11uL, 0x8B5A0A6D1F6D367EuL,
         0xCF27CB09B74F463FuL}, /* 1805 */
        {0xFFBC0B301FDE5E23uL, 0x16B414DA3EDA6CFDuL,
         0x9E4F96136E9E8C7EuL}, /* 1806 */
        {0xFF7816603FBCBC46uL, 0x2D6829B47DB4D9FBuL,
         0x3C9F2C26DD3D18FDuL}, /* 1807 */
        {0xFEF02CC07F79788CuL, 0x5AD05368FB69B3F6uL,
         0x793E584DBA7A31FBuL}, /* 1808 */
        {0xFDE05980FEF2F118uL, 0xB5A0A6D1F6D367ECuL,
         0xF27CB09B74F463F6uL}, /* 1809 */
        {0xFBC0B301FDE5E231uL, 0x6B414DA3EDA6CFD9uL,
         0xE4F96136E9E8C7ECuL}, /* 1810 */
        {0xF7816603FBCBC462uL, 0xD6829B47DB4D9FB3uL,
         0xC9F2C26DD3D18FD9uL}, /* 1811 */
        {0xEF02CC07F79788C5uL, 0xAD05368FB69B3F67uL,
         0x93E584DBA7A31FB3uL}, /* 1812 */
        {0xDE05980FEF2F118BuL, 0x5A0A6D1F6D367ECFuL,
         0x27CB09B74F463F66uL}, /* 1813 */
        {0xBC0B301FDE5E2316uL, 0xB414DA3EDA6CFD9EuL,
         0x4F96136E9E8C7ECDuL}, /* 1814 */
        {0x7816603FBCBC462DuL, 0x6829B47DB4D9FB3CuL,
         0x9F2C26DD3D18FD9AuL}, /* 1815 */
        {0xF02CC07F79788C5AuL, 0xD05368FB69B3F679uL,
         0x3E584DBA7A31FB34uL}, /* 1816 */
        {0xE05980FEF2F118B5uL, 0xA0A6D1F6D367ECF2uL,
         0x7CB09B74F463F669uL}, /* 1817 */
        {0xC0B301FDE5E2316BuL, 0x414DA3EDA6CFD9E4uL,
         0xF96136E9E8C7ECD3uL}, /* 1818 */
        {0x816603FBCBC462D6uL, 0x829B47DB4D9FB3C9uL,
         0xF2C26DD3D18FD9A7uL}, /* 1819 */
        {0x02CC07F79788C5ADuL, 0x05368FB69B3F6793uL,
         0xE584DBA7A31FB34FuL}, /* 1820 */
        {0x05980FEF2F118B5AuL, 0x0A6D1F6D367ECF27uL,
         0xCB09B74F463F669EuL}, /* 1821 */
        {0x0B301FDE5E2316B4uL, 0x14DA3EDA6CFD9E4FuL,
         0x96136E9E8C7ECD3CuL}, /* 1822 */
        {0x16603FBCBC462D68uL, 0x29B47DB4D9FB3C9FuL,
         0x2C26DD3D18FD9A79uL}, /* 1823 */
        {0x2CC07F79788C5AD0uL, 0x5368FB69B3F6793EuL,
         0x584DBA7A31FB34F2uL}, /* 1824 */
        {0x5980FEF2F118B5A0uL, 0xA6D1F6D367ECF27CuL,
         0xB09B74F463F669E5uL}, /* 1825 */
        {0xB301FDE5E2316B41uL, 0x4DA3EDA6CFD9E4F9uL,
         0x6136E9E8C7ECD3CBuL}, /* 1826 */
        {0x6603FBCBC462D682uL, 0x9B47DB4D9FB3C9F2uL,
         0xC26DD3D18FD9A797uL}, /* 1827 */
        {0xCC07F79788C5AD05uL, 0x368FB69B3F6793E5uL,
         0x84DBA7A31FB34F2FuL}, /* 1828 */
        {0x980FEF2F118B5A0AuL, 0x6D1F6D367ECF27CBuL,
         0x09B74F463F669E5FuL}, /* 1829 */
        {0x301FDE5E2316B414uL, 0xDA3EDA6CFD9E4F96uL,
         0x136E9E8C7ECD3CBFuL}, /* 1830 */
        {0x603FBCBC462D6829uL, 0xB47DB4D9FB3C9F2CuL,
         0x26DD3D18FD9A797FuL}, /* 1831 */
        {0xC07F79788C5AD053uL, 0x68FB69B3F6793E58uL,
         0x4DBA7A31FB34F2FFuL}, /* 1832 */
        {0x80FEF2F118B5A0A6uL, 0xD1F6D367ECF27CB0uL,
         0x9B74F463F669E5FEuL}, /* 1833 */
        {0x01FDE5E2316B414DuL, 0xA3EDA6CFD9E4F961uL,
         0x36E9E8C7ECD3CBFDuL}, /* 1834 */
        {0x03FBCBC462D6829BuL, 0x47DB4D9FB3C9F2C2uL,
         0x6DD3D18FD9A797FAuL}, /* 1835 */
        {0x07F79788C5AD0536uL, 0x8FB69B3F6793E584uL,
         0xDBA7A31FB34F2FF5uL}, /* 1836 */
        {0x0FEF2F118B5A0A6DuL, 0x1F6D367ECF27CB09uL,
         0xB74F463F669E5FEAuL}, /* 1837 */
        {0x1FDE5E2316B414DAuL, 0x3EDA6CFD9E4F9613uL,
         0x6E9E8C7ECD3CBFD4uL}, /* 1838 */
        {0x3FBCBC462D6829B4uL, 0x7DB4D9FB3C9F2C26uL,
         0xDD3D18FD9A797FA8uL}, /* 1839 */
        {0x7F79788C5AD05368uL, 0xFB69B3F6793E584DuL,
         0xBA7A31FB34F2FF51uL}, /* 1840 */
        {0xFEF2F118B5A0A6D1uL, 0xF6D367ECF27CB09BuL,
         0x74F463F669E5FEA2uL}, /* 1841 */
        {0xFDE5E2316B414DA3uL, 0xEDA6CFD9E4F96136uL,
         0xE9E8C7ECD3CBFD45uL}, /* 1842 */
        {0xFBCBC462D6829B47uL, 0xDB4D9FB3C9F2C26DuL,
         0xD3D18FD9A797FA8BuL}, /* 1843 */
        {0xF79788C5AD05368FuL, 0xB69B3F6793E584DBuL,
         0xA7A31FB34F2FF516uL}, /* 1844 */
        {0xEF2F118B5A0A6D1FuL, 0x6D367ECF27CB09B7uL,
         0x4F463F669E5FEA2DuL}, /* 1845 */
        {0xDE5E2316B414DA3EuL, 0xDA6CFD9E4F96136EuL,
         0x9E8C7ECD3CBFD45AuL}, /* 1846 */
        {0xBCBC462D6829B47DuL, 0xB4D9FB3C9F2C26DDuL,
         0x3D18FD9A797FA8B5uL}, /* 1847 */
        {0x79788C5AD05368FBuL, 0x69B3F6793E584DBAuL,
         0x7A31FB34F2FF516BuL}, /* 1848 */
        {0xF2F118B5A0A6D1F6uL, 0xD367ECF27CB09B74uL,
         0xF463F669E5FEA2D7uL}, /* 1849 */
        {0xE5E2316B414DA3EDuL, 0xA6CFD9E4F96136E9uL,
         0xE8C7ECD3CBFD45AEuL}, /* 1850 */
        {0xCBC462D6829B47DBuL, 0x4D9FB3C9F2C26DD3uL,
         0xD18FD9A797FA8B5DuL}, /* 1851 */
        {0x9788C5AD05368FB6uL, 0x9B3F6793E584DBA7uL,
         0xA31FB34F2FF516BAuL}, /* 1852 */
        {0x2F118B5A0A6D1F6DuL, 0x367ECF27CB09B74FuL,
         0x463F669E5FEA2D75uL}, /* 1853 */
        {0x5E2316B414DA3EDAuL, 0x6CFD9E4F96136E9EuL,
         0x8C7ECD3CBFD45AEAuL}, /* 1854 */
        {0xBC462D6829B47DB4uL, 0xD9FB3C9F2C26DD3DuL,
         0x18FD9A797FA8B5D4uL}, /* 1855 */
        {0x788C5AD05368FB69uL, 0xB3F6793E584DBA7AuL,
         0x31FB34F2FF516BA9uL}, /* 1856 */
        {0xF118B5A0A6D1F6D3uL, 0x67ECF27CB09B74F4uL,
         0x63F669E5FEA2D752uL}, /* 1857 */
        {0xE2316B414DA3EDA6uL, 0xCFD9E4F96136E9E8uL,
         0xC7ECD3CBFD45AEA4uL}, /* 1858 */
        {0xC462D6829B47DB4DuL, 0x9FB3C9F2C26DD3D1uL,
         0x8FD9A797FA8B5D49uL}, /* 1859 */
        {0x88C5AD05368FB69BuL, 0x3F6793E584DBA7A3uL,
         0x1FB34F2FF516BA93uL}, /* 1860 */
        {0x118B5A0A6D1F6D36uL, 0x7ECF27CB09B74F46uL,
         0x3F669E5FEA2D7527uL}, /* 1861 */
        {0x2316B414DA3EDA6CuL, 0xFD9E4F96136E9E8CuL,
         0x7ECD3CBFD45AEA4FuL}, /* 1862 */
        {0x462D6829B47DB4D9uL, 0xFB3C9F2C26DD3D18uL,
         0xFD9A797FA8B5D49EuL}, /* 1863 */
        {0x8C5AD05368FB69B3uL, 0xF6793E584DBA7A31uL,
         0xFB34F2FF516BA93DuL}, /* 1864 */
        {0x18B5A0A6D1F6D367uL, 0xECF27CB09B74F463uL,
         0xF669E5FEA2D7527BuL}, /* 1865 */
        {0x316B414DA3EDA6CFuL, 0xD9E4F96136E9E8C7uL,
         0xECD3CBFD45AEA4F7uL}, /* 1866 */
        {0x62D6829B47DB4D9FuL, 0xB3C9F2C26DD3D18FuL,
         0xD9A797FA8B5D49EEuL}, /* 1867 */
        {0xC5AD05368FB69B3FuL, 0x6793E584DBA7A31FuL,
         0xB34F2FF516BA93DDuL}, /* 1868 */
        {0x8B5A0A6D1F6D367EuL, 0xCF27CB09B74F463FuL,
         0x669E5FEA2D7527BAuL}, /* 1869 */
        {0x16B414DA3EDA6CFDuL, 0x9E4F96136E9E8C7EuL,
         0xCD3CBFD45AEA4F75uL}, /* 1870 */
        {0x2D6829B47DB4D9FBuL, 0x3C9F2C26DD3D18FDuL,
         0x9A797FA8B5D49EEBuL}, /* 1871 */
        {0x5AD05368FB69B3F6uL, 0x793E584DBA7A31FBuL,
         0x34F2FF516BA93DD6uL}, /* 1872 */
        {0xB5A0A6D1F6D367ECuL, 0xF27CB09B74F463F6uL,
         0x69E5FEA2D7527BACuL}, /* 1873 */
        {0x6B414DA3EDA6CFD9uL, 0xE4F96136E9E8C7ECuL,
         0xD3CBFD45AEA4F758uL}, /* 1874 */
        {0xD6829B47DB4D9FB3uL, 0xC9F2C26DD3D18FD9uL,
         0xA797FA8B5D49EEB1uL}, /* 1875 */
        {0xAD05368FB69B3F67uL, 0x93E584DBA7A31FB3uL,
         0x4F2FF516BA93DD63uL}, /* 1876 */
        {0x5A0A6D1F6D367ECFuL, 0x27CB09B74F463F66uL,
         0x9E5FEA2D7527BAC7uL}, /* 1877 */
        {0xB414DA3EDA6CFD9EuL, 0x4F96136E9E8C7ECDuL,
         0x3CBFD45AEA4F758FuL}, /* 1878 */
        {0x6829B47DB4D9FB3CuL, 0x9F2C26DD3D18FD9AuL,
         0x797FA8B5D49EEB1FuL}, /* 1879 */
        {0xD05368FB69B3F679uL, 0x3E584DBA7A31FB34uL,
         0xF2FF516BA93DD63FuL}, /* 1880 */
        {0xA0A6D1F6D367ECF2uL, 0x7CB09B74F463F669uL,
         0xE5FEA2D7527BAC7EuL}, /* 1881 */
        {0x414DA3EDA6CFD9E4uL, 0xF96136E9E8C7ECD3uL,
         0xCBFD45AEA4F758FDuL}, /* 1882 */
        {0x829B47DB4D9FB3C9uL, 0xF2C26DD3D18FD9A7uL,
         0x97FA8B5D49EEB1FAuL}, /* 1883 */
        {0x05368FB69B3F6793uL, 0xE584DBA7A31FB34FuL,
         0x2FF516BA93DD63F5uL}, /* 1884 */
        {0x0A6D1F6D367ECF27uL, 0xCB09B74F463F669EuL,
         0x5FEA2D7527BAC7EBuL}, /* 1885 */
        {0x14DA3EDA6CFD9E4FuL, 0x96136E9E8C7ECD3CuL,
         0xBFD45AEA4F758FD7uL}, /* 1886 */
        {0x29B47DB4D9FB3C9FuL, 0x2C26DD3D18FD9A79uL,
         0x7FA8B5D49EEB1FAFuL}, /* 1887 */
        {0x5368FB69B3F6793EuL, 0x584DBA7A31FB34F2uL,
         0xFF516BA93DD63F5FuL}, /* 1888 */
        {0xA6D1F6D367ECF27CuL, 0xB09B74F463F669E5uL,
         0xFEA2D7527BAC7EBEuL}, /* 1889 */
        {0x4DA3EDA6CFD9E4F9uL, 0x6136E9E8C7ECD3CBuL,
         0xFD45AEA4F758FD7CuL}, /* 1890 */
        {0x9B47DB4D9FB3C9F2uL, 0xC26DD3D18FD9A797uL,
         0xFA8B5D49EEB1FAF9uL}, /* 1891 */
        {0x368FB69B3F6793E5uL, 0x84DBA7A31FB34F2FuL,
         0xF516BA93DD63F5F2uL}, /* 1892 */
        {0x6D1F6D367ECF27CBuL, 0x09B74F463F669E5FuL,
         0xEA2D7527BAC7EBE5uL}, /* 1893 */
        {0xDA3EDA6CFD9E4F96uL, 0x136E9E8C7ECD3CBFuL,
         0xD45AEA4F758FD7CBuL}, /* 1894 */
        {0xB47DB4D9FB3C9F2CuL, 0x26DD3D18FD9A797FuL,
         0xA8B5D49EEB1FAF97uL}, /* 1895 */
        {0x68FB69B3F6793E58uL, 0x4DBA7A31FB34F2FFuL,
         0x516BA93DD63F5F2FuL}, /* 1896 */
        {0xD1F6D367ECF27CB0uL, 0x9B74F463F669E5FEuL,
         0xA2D7527BAC7EBE5FuL}, /* 1897 */
        {0xA3EDA6CFD9E4F961uL, 0x36E9E8C7ECD3CBFDuL,
         0x45AEA4F758FD7CBEuL}, /* 1898 */
        {0x47DB4D9FB3C9F2C2uL, 0x6DD3D18FD9A797FAuL,
         0x8B5D49EEB1FAF97CuL}, /* 1899 */
        {0x8FB69B3F6793E584uL, 0xDBA7A31FB34F2FF5uL,
         0x16BA93DD63F5F2F8uL}, /* 1900 */
        {0x1F6D367ECF27CB09uL, 0xB74F463F669E5FEAuL,
         0x2D7527BAC7EBE5F1uL}, /* 1901 */
        {0x3EDA6CFD9E4F9613uL, 0x6E9E8C7ECD3CBFD4uL,
         0x5AEA4F758FD7CBE2uL}, /* 1902 */
        {0x7DB4D9FB3C9F2C26uL, 0xDD3D18FD9A797FA8uL,
         0xB5D49EEB1FAF97C5uL}, /* 1903 */
        {0xFB69B3F6793E584DuL, 0xBA7A31FB34F2FF51uL,
         0x6BA93DD63F5F2F8BuL}, /* 1904 */
        {0xF6D367ECF27CB09BuL, 0x74F463F669E5FEA2uL,
         0xD7527BAC7EBE5F17uL}, /* 1905 */
        {0xEDA6CFD9E4F96136uL, 0xE9E8C7ECD3CBFD45uL,
         0xAEA4F758FD7CBE2FuL}, /* 1906 */
        {0xDB4D9FB3C9F2C26DuL, 0xD3D18FD9A797FA8BuL,
         0x5D49EEB1FAF97C5EuL}, /* 1907 */
        {0xB69B3F6793E584DBuL, 0xA7A31FB34F2FF516uL,
         0xBA93DD63F5F2F8BDuL}, /* 1908 */
        {0x6D367ECF27CB09B7uL, 0x4F463F669E5FEA2DuL,
         0x7527BAC7EBE5F17BuL}, /* 1909 */
        {0xDA6CFD9E4F96136EuL, 0x9E8C7ECD3CBFD45AuL,
         0xEA4F758FD7CBE2F6uL}, /* 1910 */
        {0xB4D9FB3C9F2C26DDuL, 0x3D18FD9A797FA8B5uL,
         0xD49EEB1FAF97C5ECuL}, /* 1911 */
        {0x69B3F6793E584DBAuL, 0x7A31FB34F2FF516BuL,
         0xA93DD63F5F2F8BD9uL}, /* 1912 */
        {0xD367ECF27CB09B74uL, 0xF463F669E5FEA2D7uL,
         0x527BAC7EBE5F17B3uL}, /* 1913 */
        {0xA6CFD9E4F96136E9uL, 0xE8C7ECD3CBFD45AEuL,
         0xA4F758FD7CBE2F67uL}, /* 1914 */
        {0x4D9FB3C9F2C26DD3uL, 0xD18FD9A797FA8B5DuL,
         0x49EEB1FAF97C5ECFuL}, /* 1915 */
        {0x9B3F6793E584DBA7uL, 0xA31FB34F2FF516BAuL,
         0x93DD63F5F2F8BD9EuL}, /* 1916 */
        {0x367ECF27CB09B74FuL, 0x463F669E5FEA2D75uL,
         0x27BAC7EBE5F17B3DuL}, /* 1917 */
        {0x6CFD9E4F96136E9EuL, 0x8C7ECD3CBFD45AEAuL,
         0x4F758FD7CBE2F67AuL}, /* 1918 */
        {0xD9FB3C9F2C26DD3DuL, 0x18FD9A797FA8B5D4uL,
         0x9EEB1FAF97C5ECF4uL}, /* 1919 */
        {0xB3F6793E584DBA7AuL, 0x31FB34F2FF516BA9uL,
         0x3DD63F5F2F8BD9E8uL}, /* 1920 */
        {0x67ECF27CB09B74F4uL, 0x63F669E5FEA2D752uL,
         0x7BAC7EBE5F17B3D0uL}, /* 1921 */
        {0xCFD9E4F96136E9E8uL, 0xC7ECD3CBFD45AEA4uL,
         0xF758FD7CBE2F67A0uL}, /* 1922 */
        {0x9FB3C9F2C26DD3D1uL, 0x8FD9A797FA8B5D49uL,
         0xEEB1FAF97C5ECF41uL}, /* 1923 */
        {0x3F6793E584DBA7A3uL, 0x1FB34F2FF516BA93uL,
         0xDD63F5F2F8BD9E83uL}, /* 1924 */
        {0x7ECF27CB09B74F46uL, 0x3F669E5FEA2D7527uL,
         0xBAC7EBE5F17B3D07uL}, /* 1925 */
        {0xFD9E4F96136E9E8CuL, 0x7ECD3CBFD45AEA4FuL,
         0x758FD7CBE2F67A0EuL}, /* 1926 */
        {0xFB3C9F2C26DD3D18uL, 0xFD9A797FA8B5D49EuL,
         0xEB1FAF97C5ECF41CuL}, /* 1927 */
        {0xF6793E584DBA7A31uL, 0xFB34F2FF516BA93DuL,
         0xD63F5F2F8BD9E839uL}, /* 1928 */
        {0xECF27CB09B74F463uL, 0xF669E5FEA2D7527BuL,
         0xAC7EBE5F17B3D073uL}, /* 1929 */
        {0xD9E4F96136E9E8C7uL, 0xECD3CBFD45AEA4F7uL,
         0x58FD7CBE2F67A0E7uL}, /* 1930 */
        {0xB3C9F2C26DD3D18FuL, 0xD9A797FA8B5D49EEuL,
         0xB1FAF97C5ECF41CEuL}, /* 1931 */
        {0x6793E584DBA7A31FuL, 0xB34F2FF516BA93DDuL,
         0x63F5F2F8BD9E839CuL}, /* 1932 */
        {0xCF27CB09B74F463FuL, 0x669E5FEA2D7527BAuL,
         0xC7EBE5F17B3D0739uL}, /* 1933 */
        {0x9E4F96136E9E8C7EuL, 0xCD3CBFD45AEA4F75uL,
         0x8FD7CBE2F67A0E73uL}, /* 1934 */
        {0x3C9F2C26DD3D18FDuL, 0x9A797FA8B5D49EEBuL,
         0x1FAF97C5ECF41CE7uL}, /* 1935 */
        {0x793E584DBA7A31FBuL, 0x34F2FF516BA93DD6uL,
         0x3F5F2F8BD9E839CFuL}, /* 1936 */
        {0xF27CB09B74F463F6uL, 0x69E5FEA2D7527BACuL,
         0x7EBE5F17B3D0739FuL}, /* 1937 */
        {0xE4F96136E9E8C7ECuL, 0xD3CBFD45AEA4F758uL,
         0xFD7CBE2F67A0E73EuL}, /* 1938 */
        {0xC9F2C26DD3D18FD9uL, 0xA797FA8B5D49EEB1uL,
         0xFAF97C5ECF41CE7DuL}, /* 1939 */
        {0x93E584DBA7A31FB3uL, 0x4F2FF516BA93DD63uL,
         0xF5F2F8BD9E839CFBuL}, /* 1940 */
        {0x27CB09B74F463F66uL, 0x9E5FEA2D7527BAC7uL,
         0xEBE5F17B3D0739F7uL}, /* 1941 */
        {0x4F96136E9E8C7ECDuL, 0x3CBFD45AEA4F758FuL,
         0xD7CBE2F67A0E73EFuL}, /* 1942 */
        {0x9F2C26DD3D18FD9AuL, 0x797FA8B5D49EEB1FuL,
         0xAF97C5ECF41CE7DEuL}, /* 1943 */
        {0x3E584DBA7A31FB34uL, 0xF2FF516BA93DD63FuL,
         0x5F2F8BD9E839CFBCuL}, /* 1944 */
        {0x7CB09B74F463F669uL, 0xE5FEA2D7527BAC7EuL,
         0xBE5F17B3D0739F78uL}, /* 1945 */
        {0xF96136E9E8C7ECD3uL, 0xCBFD45AEA4F758FDuL,
         0x7CBE2F67A0E73EF1uL}, /* 1946 */
        {0xF2C26DD3D18FD9A7uL, 0x97FA8B5D49EEB1FAuL,
         0xF97C5ECF41CE7DE2uL}, /* 1947 */
        {0xE584DBA7A31FB34FuL, 0x2FF516BA93DD63F5uL,
         0xF2F8BD9E839CFBC5uL}, /* 1948 */
        {0xCB09B74F463F669EuL, 0x5FEA2D7527BAC7EBuL,
         0xE5F17B3D0739F78AuL}, /* 1949 */
        {0x96136E9E8C7ECD3CuL, 0xBFD45AEA4F758FD7uL,
         0xCBE2F67A0E73EF14uL}, /* 1950 */
        {0x2C26DD3D18FD9A79uL, 0x7FA8B5D49EEB1FAFuL,
         0x97C5ECF41CE7DE29uL}, /* 1951 */
        {0x584DBA7A31FB34F2uL, 0xFF516BA93DD63F5FuL,
         0x2F8BD9E839CFBC52uL}, /* 1952 */
        {0xB09B74F463F669E5uL, 0xFEA2D7527BAC7EBEuL,
         0x5F17B3D0739F78A5uL}, /* 1953 */
        {0x6136E9E8C7ECD3CBuL, 0xFD45AEA4F758FD7CuL,
         0xBE2F67A0E73EF14AuL}, /* 1954 */
        {0xC26DD3D18FD9A797uL, 0xFA8B5D49EEB1FAF9uL,
         0x7C5ECF41CE7DE294uL}, /* 1955 */
        {0x84DBA7A31FB34F2FuL, 0xF516BA93DD63F5F2uL,
         0xF8BD9E839CFBC529uL}, /* 1956 */
        {0x09B74F463F669E5FuL, 0xEA2D7527BAC7EBE5uL,
         0xF17B3D0739F78A52uL}, /* 1957 */
        {0x136E9E8C7ECD3CBFuL, 0xD45AEA4F758FD7CBuL,
         0xE2F67A0E73EF14A5uL}, /* 1958 */
        {0x26DD3D18FD9A797FuL, 0xA8B5D49EEB1FAF97uL,
         0xC5ECF41CE7DE294AuL}, /* 1959 */
        {0x4DBA7A31FB34F2FFuL, 0x516BA93DD63F5F2FuL,
         0x8BD9E839CFBC5294uL}, /* 1960 */
        {0x9B74F463F669E5FEuL, 0xA2D7527BAC7EBE5FuL,
         0x17B3D0739F78A529uL}, /* 1961 */
        {0x36E9E8C7ECD3CBFDuL, 0x45AEA4F758FD7CBEuL,
         0x2F67A0E73EF14A52uL}, /* 1962 */
        {0x6DD3D18FD9A797FAuL, 0x8B5D49EEB1FAF97CuL,
         0x5ECF41CE7DE294A4uL}, /* 1963 */
        {0xDBA7A31FB34F2FF5uL, 0x16BA93DD63F5F2F8uL,
         0xBD9E839CFBC52949uL}, /* 1964 */
        {0xB74F463F669E5FEAuL, 0x2D7527BAC7EBE5F1uL,
         0x7B3D0739F78A5292uL}, /* 1965 */
        {0x6E9E8C7ECD3CBFD4uL, 0x5AEA4F758FD7CBE2uL,
         0xF67A0E73EF14A525uL}, /* 1966 */
        {0xDD3D18FD9A797FA8uL, 0xB5D49EEB1FAF97C5uL,
         0xECF41CE7DE294A4BuL}, /* 1967 */
        {0xBA7A31FB34F2FF51uL, 0x6BA93DD63F5F2F8BuL,
         0xD9E839CFBC529497uL}, /* 1968 */
        {0x74F463F669E5FEA2uL, 0xD7527BAC7EBE5F17uL,
         0xB3D0739F78A5292EuL}, /* 1969 */
        {0xE9E8C7ECD3CBFD45uL, 0xAEA4F758FD7CBE2FuL,
         0x67A0E73EF14A525DuL}, /* 1970 */
        {0xD3D18FD9A797FA8BuL, 0x5D49EEB1FAF97C5EuL,
         0xCF41CE7DE294A4BAuL}, /* 1971 */
        {0xA7A31FB34F2FF516uL, 0xBA93DD63F5F2F8BDuL,
         0x9E839CFBC5294975uL}, /* 1972 */
        {0x4F463F669E5FEA2DuL, 0x7527BAC7EBE5F17BuL,
         0x3D0739F78A5292EAuL}, /* 1973 */
        {0x9E8C7ECD3CBFD45AuL, 0xEA4F758FD7CBE2F6uL,
         0x7A0E73EF14A525D4uL}, /* 1974 */
        {0x3D18FD9A797FA8B5uL, 0xD49EEB1FAF97C5ECuL,
         0xF41CE7DE294A4BA9uL}, /* 1975 */
        {0x7A31FB34F2FF516BuL, 0xA93DD63F5F2F8BD9uL,
         0xE839CFBC52949753uL}, /* 1976 */
        {0xF463F669E5FEA2D7uL, 0x527BAC7EBE5F17B3uL,
         0xD0739F78A5292EA6uL}, /* 1977 */
        {0xE8C7ECD3CBFD45AEuL, 0xA4F758FD7CBE2F67uL,
         0xA0E73EF14A525D4DuL}, /* 1978 */
        {0xD18FD9A797FA8B5DuL, 0x49EEB1FAF97C5ECFuL,
         0x41CE7DE294A4BA9AuL}, /* 1979 */
        {0xA31FB34F2FF516BAuL, 0x93DD63F5F2F8BD9EuL,
         0x839CFBC529497535uL}, /* 1980 */
        {0x463F669E5FEA2D75uL, 0x27BAC7EBE5F17B3DuL,
         0x0739F78A5292EA6BuL}, /* 1981 */
        {0x8C7ECD3CBFD45AEAuL, 0x4F758FD7CBE2F67AuL,
         0x0E73EF14A525D4D7uL}, /* 1982 */
        {0x18FD9A797FA8B5D4uL, 0x9EEB1FAF97C5ECF4uL,
         0x1CE7DE294A4BA9AFuL}, /* 1983 */
        {0x31FB34F2FF516BA9uL, 0x3DD63F5F2F8BD9E8uL,
         0x39CFBC529497535FuL}, /* 1984 */
        {0x63F669E5FEA2D752uL, 0x7BAC7EBE5F17B3D0uL,
         0x739F78A5292EA6BFuL}, /* 1985 */
        {0xC7ECD3CBFD45AEA4uL, 0xF758FD7CBE2F67A0uL,
         0xE73EF14A525D4D7FuL}, /* 1986 */
        {0x8FD9A797FA8B5D49uL, 0xEEB1FAF97C5ECF41uL,
         0xCE7DE294A4BA9AFEuL}, /* 1987 */
        {0x1FB34F2FF516BA93uL, 0xDD63F5F2F8BD9E83uL,
         0x9CFBC529497535FDuL}, /* 1988 */
        {0x3F669E5FEA2D7527uL, 0xBAC7EBE5F17B3D07uL,
         0x39F78A5292EA6BFBuL}, /* 1989 */
        {0x7ECD3CBFD45AEA4FuL, 0x758FD7CBE2F67A0EuL,
         0x73EF14A525D4D7F6uL}, /* 1990 */
        {0xFD9A797FA8B5D49EuL, 0xEB1FAF97C5ECF41CuL,
         0xE7DE294A4BA9AFEDuL}, /* 1991 */
        {0xFB34F2FF516BA93DuL, 0xD63F5F2F8BD9E839uL,
         0xCFBC529497535FDAuL}, /* 1992 */
        {0xF669E5FEA2D7527BuL, 0xAC7EBE5F17B3D073uL,
         0x9F78A5292EA6BFB5uL}, /* 1993 */
        {0xECD3CBFD45AEA4F7uL, 0x58FD7CBE2F67A0E7uL,
         0x3EF14A525D4D7F6BuL}, /* 1994 */
        {0xD9A797FA8B5D49EEuL, 0xB1FAF97C5ECF41CEuL,
         0x7DE294A4BA9AFED7uL}, /* 1995 */
        {0xB34F2FF516BA93DDuL, 0x63F5F2F8BD9E839CuL,
         0xFBC529497535FDAFuL}, /* 1996 */
        {0x669E5FEA2D7527BAuL, 0xC7EBE5F17B3D0739uL,
         0xF78A5292EA6BFB5FuL}, /* 1997 */
        {0xCD3CBFD45AEA4F75uL, 0x8FD7CBE2F67A0E73uL,
         0xEF14A525D4D7F6BFuL}, /* 1998 */
        {0x9A797FA8B5D49EEBuL, 0x1FAF97C5ECF41CE7uL,
         0xDE294A4BA9AFED7EuL}, /* 1999 */
        {0x34F2FF516BA93DD6uL, 0x3F5F2F8BD9E839CFuL,
         0xBC529497535FDAFDuL}, /* 2000 */
        {0x69E5FEA2D7527BACuL, 0x7EBE5F17B3D0739FuL,
         0x78A5292EA6BFB5FBuL}, /* 2001 */
        {0xD3CBFD45AEA4F758uL, 0xFD7CBE2F67A0E73EuL,
         0xF14A525D4D7F6BF6uL}, /* 2002 */
        {0xA797FA8B5D49EEB1uL, 0xFAF97C5ECF41CE7DuL,
         0xE294A4BA9AFED7ECuL}, /* 2003 */
        {0x4F2FF516BA93DD63uL, 0xF5F2F8BD9E839CFBuL,
         0xC529497535FDAFD8uL}, /* 2004 */
        {0x9E5FEA2D7527BAC7uL, 0xEBE5F17B3D0739F7uL,
         0x8A5292EA6BFB5FB1uL}, /* 2005 */
        {0x3CBFD45AEA4F758FuL, 0xD7CBE2F67A0E73EFuL,
         0x14A525D4D7F6BF62uL}, /* 2006 */
        {0x797FA8B5D49EEB1FuL, 0xAF97C5ECF41CE7DEuL,
         0x294A4BA9AFED7EC4uL}, /* 2007 */
        {0xF2FF516BA93DD63FuL, 0x5F2F8BD9E839CFBCuL,
         0x529497535FDAFD88uL}, /* 2008 */
        {0xE5FEA2D7527BAC7EuL, 0xBE5F17B3D0739F78uL,
         0xA5292EA6BFB5FB11uL}, /* 2009 */
        {0xCBFD45AEA4F758FDuL, 0x7CBE2F67A0E73EF1uL,
         0x4A525D4D7F6BF623uL}, /* 2010 */
        {0x97FA8B5D49EEB1FAuL, 0xF97C5ECF41CE7DE2uL,
         0x94A4BA9AFED7EC47uL}, /* 2011 */
        {0x2FF516BA93DD63F5uL, 0xF2F8BD9E839CFBC5uL,
         0x29497535FDAFD88FuL}, /* 2012 */
        {0x5FEA2D7527BAC7EBuL, 0xE5F17B3D0739F78AuL,
         0x5292EA6BFB5FB11FuL}, /* 2013 */
        {0xBFD45AEA4F758FD7uL, 0xCBE2F67A0E73EF14uL,
         0xA525D4D7F6BF623FuL}, /* 2014 */
        {0x7FA8B5D49EEB1FAFuL, 0x97C5ECF41CE7DE29uL,
         0x4A4BA9AFED7EC47EuL}, /* 2015 */
        {0xFF516BA93DD63F5FuL, 0x2F8BD9E839CFBC52uL,
         0x9497535FDAFD88FCuL}, /* 2016 */
        {0xFEA2D7527BAC7EBEuL, 0x5F17B3D0739F78A5uL,
         0x292EA6BFB5FB11F8uL}, /* 2017 */
        {0xFD45AEA4F758FD7CuL, 0xBE2F67A0E73EF14AuL,
         0x525D4D7F6BF623F1uL}, /* 2018 */
        {0xFA8B5D49EEB1FAF9uL, 0x7C5ECF41CE7DE294uL,
         0xA4BA9AFED7EC47E3uL}, /* 2019 */
        {0xF516BA93DD63F5F2uL, 0xF8BD9E839CFBC529uL,
         0x497535FDAFD88FC6uL}, /* 2020 */
        {0xEA2D7527BAC7EBE5uL, 0xF17B3D0739F78A52uL,
         0x92EA6BFB5FB11F8DuL}, /* 2021 */
        {0xD45AEA4F758FD7CBuL, 0xE2F67A0E73EF14A5uL,
         0x25D4D7F6BF623F1AuL}, /* 2022 */
        {0xA8B5D49EEB1FAF97uL, 0xC5ECF41CE7DE294AuL,
         0x4BA9AFED7EC47E35uL}, /* 2023 */
        {0x516BA93DD63F5F2FuL, 0x8BD9E839CFBC5294uL,
         0x97535FDAFD88FC6AuL}, /* 2024 */
        {0xA2D7527BAC7EBE5FuL, 0x17B3D0739F78A529uL,
         0x2EA6BFB5FB11F8D5uL}, /* 2025 */
        {0x45AEA4F758FD7CBEuL, 0x2F67A0E73EF14A52uL,
         0x5D4D7F6BF623F1ABuL}, /* 2026 */
        {0x8B5D49EEB1FAF97CuL, 0x5ECF41CE7DE294A4uL,
         0xBA9AFED7EC47E357uL}, /* 2027 */
        {0x16BA93DD63F5F2F8uL, 0xBD9E839CFBC52949uL,
         0x7535FDAFD88FC6AEuL}, /* 2028 */
        {0x2D7527BAC7EBE5F1uL, 0x7B3D0739F78A5292uL,
         0xEA6BFB5FB11F8D5DuL}, /* 2029 */
        {0x5AEA4F758FD7CBE2uL, 0xF67A0E73EF14A525uL,
         0xD4D7F6BF623F1ABAuL}, /* 2030 */
        {0xB5D49EEB1FAF97C5uL, 0xECF41CE7DE294A4BuL,
         0xA9AFED7EC47E3574uL}, /* 2031 */
        {0x6BA93DD63F5F2F8BuL, 0xD9E839CFBC529497uL,
         0x535FDAFD88FC6AE8uL}, /* 2032 */
        {0xD7527BAC7EBE5F17uL, 0xB3D0739F78A5292EuL,
         0xA6BFB5FB11F8D5D0uL}, /* 2033 */
        {0xAEA4F758FD7CBE2FuL, 0x67A0E73EF14A525DuL,
         0x4D7F6BF623F1ABA1uL}, /* 2034 */
        {0x5D49EEB1FAF97C5EuL, 0xCF41CE7DE294A4BAuL,
         0x9AFED7EC47E35742uL}, /* 2035 */
        {0xBA93DD63F5F2F8BDuL, 0x9E839CFBC5294975uL,
         0x35FDAFD88FC6AE84uL}, /* 2036 */
        {0x7527BAC7EBE5F17BuL, 0x3D0739F78A5292EAuL,
         0x6BFB5FB11F8D5D08uL}, /* 2037 */
        {0xEA4F758FD7CBE2F6uL, 0x7A0E73EF14A525D4uL,
         0xD7F6BF623F1ABA10uL}, /* 2038 */
        {0xD49EEB1FAF97C5ECuL, 0xF41CE7DE294A4BA9uL,
         0xAFED7EC47E357421uL}, /* 2039 */
        {0xA93DD63F5F2F8BD9uL, 0xE839CFBC52949753uL,
         0x5FDAFD88FC6AE842uL}, /* 2040 */
        {0x527BAC7EBE5F17B3uL, 0xD0739F78A5292EA6uL,
         0xBFB5FB11F8D5D085uL}, /* 2041 */
        {0xA4F758FD7CBE2F67uL, 0xA0E73EF14A525D4DuL,
         0x7F6BF623F1ABA10AuL}, /* 2042 */
        {0x49EEB1FAF97C5ECFuL, 0x41CE7DE294A4BA9AuL,
         0xFED7EC47E3574215uL}, /* 2043 */
        {0x93DD63F5F2F8BD9EuL, 0x839CFBC529497535uL,
         0xFDAFD88FC6AE842BuL}, /* 2044 */
        {0x27BAC7EBE5F17B3DuL, 0x0739F78A5292EA6BuL,
         0xFB5FB11F8D5D0856uL}, /* 2045 */
        {0x4F758FD7CBE2F67AuL, 0x0E73EF14A525D4D7uL,
         0xF6BF623F1ABA10ACuL}, /* 2046 */
        {0x9EEB1FAF97C5ECF4uL, 0x1CE7DE294A4BA9AFuL,
         0xED7EC47E35742158uL} /* 2047 */
    },
}; /*dReduction_Table*/
   /* Table parameters */
typedef struct {
  unsigned long _dInvPI_uisa;
  unsigned long _dPI1_uisa;
  unsigned long _dPI2_uisa;
  unsigned long _dPI3_uisa;
  unsigned long _dPI2_ha_uisa;
  unsigned long _dPI3_ha_uisa;
  unsigned long Th_tbl_uisa[16];
  unsigned long Tl_tbl_uisa[16];
  unsigned long _dPC3_uisa;
  unsigned long _dPC5_uisa;
  unsigned long _dPC7_uisa;
  unsigned long _dPC9_uisa;
  unsigned long _dPC11_uisa;
  unsigned long _dOne_uisa;
  unsigned long _dAbsMask;
  unsigned long _dRangeVal;
  unsigned long _dRShift;
  unsigned long _dPIoHi;
  unsigned long _dPIoLo;
  unsigned long _dPIoTail;
  unsigned long _lIndexMask;
  unsigned long _dPIu;
  unsigned long _dCoeffs[256][13];
  unsigned long _dReductionRangeVal;
} __ocl_svml_internal_dtan_ha_data_t;
static __ocl_svml_internal_dtan_ha_data_t __ocl_svml_internal_dtan_ha_data = {
    /* UISA */
    0x40145F306DC9C883uL, /* _dInvPI_uisa */
    0x3fc921fb54442d18uL, /* _dPI1_uisa */
    0x3c61a62633145c06uL, /* _dPI2_uisa */
    0x391c1cd129024e09uL, /* _dPI3_uisa */
    0x3c61a62633000000uL, /* _dPI2_ha_uisa */
    0x3a645c06e0e68948uL, /* _dPI3_ha_uisa */
    /*for i from 0 to 15 do printdouble(tan(i*Pi/16));*/
    {
        /*== Th_tbl_uisa ==*/
        0x8000000000000000uL,
        0x3fc975f5e0553158uL,
        0x3fda827999fcef32uL,
        0x3fe561b82ab7f990uL,
        0x3ff0000000000000uL,
        0x3ff7f218e25a7461uL,
        0x4003504f333f9de6uL,
        0x40141bfee2424771uL,
        0xffefffffffffffffuL,
        0xc0141bfee2424771uL,
        0xc003504f333f9de6uL,
        0xbff7f218e25a7461uL,
        0xbff0000000000000uL,
        0xbfe561b82ab7f990uL,
        0xbfda827999fcef32uL,
        0xbfc975f5e0553158uL,
    }, /* Th_tbl_uisa */
    {
        /*== Tl_tbl_uisa ==*/
        0x8000000000000000uL,
        0x3c2ef5d367441946uL,
        0x3c708b2fb1366ea9uL,
        0x3c87a8c52172b675uL,
        0x0000000000000000uL,
        0x3c9419fa6954928fuL,
        0x3ca21165f626cdd5uL,
        0x3c810706fed37f0euL,
        0xfca0000000000000uL,
        0xbc810706fed37f0euL,
        0xbca21165f626cdd5uL,
        0xbc9419fa6954928fuL,
        0x0000000000000000uL,
        0xbc87a8c52172b675uL,
        0xbc708b2fb1366ea9uL,
        0xbc2ef5d367441946uL,
    },                    /* Tl_tbl_uisa */
    0x3fd55555555555dcuL, /* _dPC3_uisa */
    0x3fc11111110b0802uL, /* _dPC5_uisa */
    0x3faba1ba489d25cauL, /* _dPC7_uisa */
    0x3f9664ab664efba9uL, /* _dPC9_uisa */
    0x3f825cccc7c9fa5duL, /* _dPC11_uisa */
    0x3ff0000000000000uL, /* _dOne_uisa */
    /* Shared value*/
    0x7FFFFFFFFFFFFFFFuL, /* _dAbsMask  */
    0x7FF0000000000000uL, /* _dRangeVal  */
    0x4338000000000000uL, /* _dRShift  */
    0x3F8921FB54400000uL, /* PIoHi = 1.22718463026103563607e-02 */
    0x3D60B4611A600000uL, /* PIoLo = 4.74773477054997341922e-13 */
    0x3B33198A2E037073uL, /* PIoTail = 1.57989550687183643089e-23 */
    0x00000000000001FFuL, /* lIndexMask  */
    0x40545F306DC9C883uL, /* PIu = 8.14873308630504169514e+01 */
                          /* _dCoeffs */
    {
        // Breakpoint B = 0 * pi/256, function tan(B + x)
        {
            0x3FF921FB54442D18uL, // B' = pi/2 - B (high double)
            0x3C91A62633145C07uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x0000000000000000uL, // c0 (high double)
            0x0000000000000000uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x0000000000000000uL, // c1 (low double)
            0x0000000000000000uL, // c2
            0x3FD5555555555559uL, // c3
            0x0000000000000000uL, // c4
            0x3FC1111110FE825DuL, // c5
            0x0000000000000000uL, // c6
            0x3FABA1EEC0271523uL, // c7
        },
        // Breakpoint B = 1 * pi/256, function tan(B + x)
        {
            0x3FF8EFB75D9BA4BEuL, // B' = pi/2 - B (high double)
            0x3C760B679AB8CD3BuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3F89224E047E368EuL, // c0 (high double)
            0xBC1A268EB2C67108uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F23BDBEAD8CF800uL, // c1 (low double)
            0x3F8923461AE96D62uL, // c2
            0x3FD5589FB822D5EEuL, // c3
            0x3F80C326DCDCFCD0uL, // c4
            0x3FC116A999125B7FuL, // c5
            0x3F7300CC26596ADAuL, // c6
            0x3FAB7B6F1DE289A8uL, // c7
        },
        // Breakpoint B = 2 * pi/256, function tan(B + x)
        {
            0x3FF8BD7366F31C64uL, // B' = pi/2 - B (high double)
            0xBC7A81C996DFD5A5uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3F992346247A91F0uL, // c0 (high double)
            0xBBF126D920C15240uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F43BF447B091000uL, // c1 (low double)
            0x3F992726F0F98D7FuL, // c2
            0x3FD5628108EFA55CuL, // c3
            0x3F90C8A60639C114uL, // c4
            0x3FC127789EC0BD38uL, // c5
            0x3F830BA6F46C58B7uL, // c6
            0x3FABBEE2C1E62B81uL, // c7
        },
        // Breakpoint B = 3 * pi/256, function tan(B + x)
        {
            0x3FF88B2F704A940AuL, // B' = pi/2 - B (high double)
            0xBC92C3BEB21E1E21uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FA2DBAAE9A05DB0uL, // c0 (high double)
            0xBC4371F88E84877CuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F563A086B860B00uL, // c1 (low double)
            0x3FA2E23785EEB488uL, // c2
            0x3FD572FFC353B741uL, // c3
            0x3F993ABD2276BF6AuL, // c4
            0x3FC1439260886581uL, // c5
            0x3F8CAC359DCF5E16uL, // c6
            0x3FAB56F9A068D4C5uL, // c7
        },
        // Breakpoint B = 4 * pi/256, function tan(B + x)
        {
            0x3FF858EB79A20BB0uL, // B' = pi/2 - B (high double)
            0xBC9EE70AFE8446D9uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FA927278A3B1162uL, // c0 (high double)
            0x3BFB01BD7A51D5C0uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F63C55DB08E1680uL, // c1 (low double)
            0x3FA936B1EC652161uL, // c2
            0x3FD58A26BD6CCCCDuL, // c3
            0x3FA0DEB5445B4098uL, // c4
            0x3FC16B146127E154uL, // c5
            0x3F9336A98365D094uL, // c6
            0x3FAC19302EB1C4A9uL, // c7
        },
        // Breakpoint B = 5 * pi/256, function tan(B + x)
        {
            0x3FF826A782F98355uL, // B' = pi/2 - B (high double)
            0x3C94F5A8B515906FuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FAF7495EA3F3783uL, // c0 (high double)
            0xBC407F10D50F4C0AuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F6EEB8B377B1A00uL, // c1 (low double)
            0x3FAF92FABFD0B7B3uL, // c2
            0x3FD5A805341437FFuL, // c3
            0x3FA52B295E2DCF92uL, // c4
            0x3FC19E2C59B8C033uL, // c5
            0x3F982CF88466EFECuL, // c6
            0x3FAD27509D6C3FEAuL, // c7
        },
        // Breakpoint B = 6 * pi/256, function tan(B + x)
        {
            0x3FF7F4638C50FAFBuL, // B' = pi/2 - B (high double)
            0x3C81A4B8D15ECF6DuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FB2E239CCFF3831uL, // c0 (high double)
            0x3C5879ED3FF4EEAEuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F764980AEEE3740uL, // c1 (low double)
            0x3FB2FC879C81123CuL, // c2
            0x3FD5CCAEDC1DC96EuL, // c3
            0x3FA98599B71D86B1uL, // c4
            0x3FC1DD150D55E5CEuL, // c5
            0x3F9D3EB9450DBD01uL, // c6
            0x3FADD6CFBE9AA234uL, // c7
        },
        // Breakpoint B = 7 * pi/256, function tan(B + x)
        {
            0x3FF7C21F95A872A1uL, // B' = pi/2 - B (high double)
            0xBC6A877F1DB6080CuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FB60B9F7597FDECuL, // c0 (high double)
            0x3C3D1E3FBF3B4E18uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F7E5FFEF4D8DFC0uL, // c1 (low double)
            0x3FB6357984E124B6uL, // c2
            0x3FD5F83BF8B56900uL, // c3
            0x3FADF0F50CAA06C2uL, // c4
            0x3FC2281476904258uL, // c5
            0x3FA138F0F640FD68uL, // c6
            0x3FAF8D1BCDD8D11FuL, // c7
        },
        // Breakpoint B = 8 * pi/256, function tan(B + x)
        {
            0x3FF78FDB9EFFEA47uL, // B' = pi/2 - B (high double)
            0xBC8EE8786039D373uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FB936BB8C5B2DA2uL, // c0 (high double)
            0x3C587D23392C2BA2uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F83DDE2A8935DE0uL, // c1 (low double)
            0x3FB97558DC230BA8uL, // c2
            0x3FD62AC976FBC1B8uL, // c3
            0x3FB1381EB60159F6uL, // c4
            0x3FC27F84438D5029uL, // c5
            0x3FA3E63B3A462E45uL, // c6
            0x3FAF0AE614B2D473uL, // c7
        },
        // Breakpoint B = 9 * pi/256, function tan(B + x)
        {
            0x3FF75D97A85761EDuL, // B' = pi/2 - B (high double)
            0xBC9B97887C831272uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FBC63CE377FC802uL, // c0 (high double)
            0xBC543F7D42DBAC49uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F892FE02A47EFA0uL, // c1 (low double)
            0x3FBCBD3001C956C5uL, // c2
            0x3FD664790F293C47uL, // c3
            0x3FB38345DE7C4B0BuL, // c4
            0x3FC2E3C4C02FFDFBuL, // c5
            0x3FA6AA7F0C13520BuL, // c6
            0x3FAFE3C754A9F092uL, // c7
        },
        // Breakpoint B = 10 * pi/256, function tan(B + x)
        {
            0x3FF72B53B1AED992uL, // B' = pi/2 - B (high double)
            0x3C98452B3716C4D6uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FBF93183A8DB9E9uL, // c0 (high double)
            0xBC4124AC452772E2uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F8F27A3181EE680uL, // c1 (low double)
            0x3FC00707549D7ECCuL, // c2
            0x3FD6A5716B500730uL, // c3
            0x3FB5DB89BB6B245FuL, // c4
            0x3FC3554867F59648uL, // c5
            0x3FA98919BF6D176AuL, // c6
            0x3FB1AF9A1823147BuL, // c7
        },
        // Breakpoint B = 11 * pi/256, function tan(B + x)
        {
            0x3FF6F90FBB065138uL, // B' = pi/2 - B (high double)
            0x3C8843BDD561383DuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FC1626D85A91E70uL, // c0 (high double)
            0xBC50084844291CB4uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F92E3863C9A69E0uL, // c1 (low double)
            0x3FC1B4854BB2B147uL, // c2
            0x3FD6EDDE5416DFFBuL, // c3
            0x3FB8429341CCA17AuL, // c4
            0x3FC3D49AC2C9F46FuL, // c5
            0x3FAC85843914CF90uL, // c6
            0x3FB1B1DC3FA33DDDuL, // c7
        },
        // Breakpoint B = 12 * pi/256, function tan(B + x)
        {
            0x3FF6C6CBC45DC8DEuL, // B' = pi/2 - B (high double)
            0xBBD6D61B58C99C43uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FC2FCAC73A60640uL, // c0 (high double)
            0x3C52CF95F70B55BAuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F96881A43B33590uL, // c1 (low double)
            0x3FC367A033B52F5EuL, // c2
            0x3FD73DF0E3CC557AuL, // c3
            0x3FBABA1CAC4F18CDuL, // c4
            0x3FC4624C36BE941BuL, // c5
            0x3FAFA38B2F8876EFuL, // c6
            0x3FB215B1F404D921uL, // c7
        },
        // Breakpoint B = 13 * pi/256, function tan(B + x)
        {
            0x3FF69487CDB54084uL, // B' = pi/2 - B (high double)
            0xBC8849735C376AA4uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FC4986A74CF4E57uL, // c0 (high double)
            0x3C69AB0A24392F72uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F9A82B60D75F5C0uL, // c1 (low double)
            0x3FC520EA2BA6A0AEuL, // c2
            0x3FD795DFBFFEE39EuL, // c3
            0x3FBD43F3A7862CC7uL, // c4
            0x3FC4FF04247D700AuL, // c5
            0x3FB1739726786F4FuL, // c6
            0x3FB2FAC67533455BuL, // c7
        },
        // Breakpoint B = 14 * pi/256, function tan(B + x)
        {
            0x3FF66243D70CB82AuL, // B' = pi/2 - B (high double)
            0xBC984805FA81DE0AuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FC635C990CE0D36uL, // c0 (high double)
            0xBC59DFC0C48C1978uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F9ED49F1F8144C0uL, // c1 (low double)
            0x3FC6E0F98ED00C32uL, // c2
            0x3FD7F5E75A2DD66EuL, // c3
            0x3FBFE1FBA10F6B0EuL, // c4
            0x3FC5AB7CD1905EB8uL, // c5
            0x3FB32A57A4FAE3F9uL, // c6
            0x3FB4BA60BA656817uL, // c7
        },
        // Breakpoint B = 15 * pi/256, function tan(B + x)
        {
            0x3FF62FFFE0642FCFuL, // B' = pi/2 - B (high double)
            0x3C9B94ADB917F93EuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FC7D4EC54FB5968uL, // c0 (high double)
            0x3C4FB65648F8ABF4uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FA1BF9C7C60B940uL, // c1 (low double)
            0x3FC8A86964A1796BuL, // c2
            0x3FD85E4A37F23352uL, // c3
            0x3FC14B181F5BF6AEuL, // c4
            0x3FC66888E3A5D3B3uL, // c5
            0x3FB4F84D7AC0F5DAuL, // c6
            0x3FB62BA6F082700CuL, // c7
        },
        // Breakpoint B = 16 * pi/256, function tan(B + x)
        {
            0x3FF5FDBBE9BBA775uL, // B' = pi/2 - B (high double)
            0x3C8EE2C2D963A10CuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FC975F5E0553158uL, // c0 (high double)
            0x3C2EEEA31ABB0A10uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FA4420303DEC550uL, // c1 (low double)
            0x3FCA77D9D62CA610uL, // c2
            0x3FD8CF5143323608uL, // c3
            0x3FC2B15401455507uL, // c4
            0x3FC7370FF0965570uL, // c5
            0x3FB6DFEACA56EACBuL, // c6
            0x3FB6B75DF89DE1ECuL, // c7
        },
        // Breakpoint B = 17 * pi/256, function tan(B + x)
        {
            0x3FF5CB77F3131F1BuL, // B' = pi/2 - B (high double)
            0x3C6A70A9025D3E70uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FCB1909EFD8B762uL, // c0 (high double)
            0x3C5843049E1321E0uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FA6F2545C478E00uL, // c1 (low double)
            0x3FCC4FF0A9D74F70uL, // c2
            0x3FD9494C22EF2A54uL, // c3
            0x3FC424CB909A6722uL, // c4
            0x3FC8180A481D5934uL, // c5
            0x3FB8E3D90C36FA83uL, // c6
            0x3FB8D27EAFF0DF70uL, // c7
        },
        // Breakpoint B = 18 * pi/256, function tan(B + x)
        {
            0x3FF59933FC6A96C1uL, // B' = pi/2 - B (high double)
            0xBC81AA6E583501D4uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FCCBE4CEB4B4CF2uL, // c0 (high double)
            0x3C64A514F790E428uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FA9D1724DE6C940uL, // c1 (low double)
            0x3FCE3159C5ED3A4EuL, // c2
            0x3FD9CC919D4271B2uL, // c3
            0x3FC5A6A9489AA7F4uL, // c4
            0x3FC90C961E2B49F6uL, // c5
            0x3FBB06EE7CFA3693uL, // c6
            0x3FB9E9447AB1D521uL, // c7
        },
        // Breakpoint B = 19 * pi/256, function tan(B + x)
        {
            0x3FF566F005C20E67uL, // B' = pi/2 - B (high double)
            0xBC94F8837880A9A2uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FCE65E3F27C9F2AuL, // c0 (high double)
            0x3C6F62A8B8B40082uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FACE04FD3EC1C90uL, // c1 (low double)
            0x3FD00E63DD639A45uL, // c2
            0x3FDA598003751442uL, // c3
            0x3FC73829AC409F93uL, // c4
            0x3FCA15E3A26F9D65uL, // c5
            0x3FBD4C4574E0CB34uL, // c6
            0x3FBB9C5881865EC1uL, // c7
        },
        // Breakpoint B = 20 * pi/256, function tan(B + x)
        {
            0x3FF534AC0F19860CuL, // B' = pi/2 - B (high double)
            0x3C9EE4303B192DA6uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FD007FA758626AEuL, // c0 (high double)
            0x3C65A5EB9CA505B6uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FB00FF8E583BEBCuL, // c1 (low double)
            0x3FD1097A2AA570F5uL, // c2
            0x3FDAF07DA8CAEBA7uL, // c3
            0x3FC8DA9D107525ADuL, // c4
            0x3FCB3545F2ADC1D6uL, // c5
            0x3FBFB739DB465891uL, // c6
            0x3FBDA2C23752D365uL, // c7
        },
        // Breakpoint B = 21 * pi/256, function tan(B + x)
        {
            0x3FF502681870FDB2uL, // B' = pi/2 - B (high double)
            0x3C92C0E3EEB304EEuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FD0DE53475F3B3CuL, // c0 (high double)
            0x3C67B2CC0A83B8D0uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FB1C8B7D76C7EE0uL, // c1 (low double)
            0x3FD20A509D4BB6C1uL, // c2
            0x3FDB91F96510DF38uL, // c3
            0x3FCA8F698D62832EuL, // c4
            0x3FCC6C340829132AuL, // c5
            0x3FC125B6D8F6980AuL, // c6
            0x3FBE80E8F3AEF448uL, // c7
        },
        // Breakpoint B = 22 * pi/256, function tan(B + x)
        {
            0x3FF4D02421C87558uL, // B' = pi/2 - B (high double)
            0x3C7A765E893370D7uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FD1B6103D3597E9uL, // c0 (high double)
            0x3C6CBCB5FED7BFB0uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FB39AFA3392E164uL, // c1 (low double)
            0x3FD3114C46E2AEF3uL, // c2
            0x3FDC3E6B23EF73B5uL, // c3
            0x3FCC580D1B42EE7FuL, // c4
            0x3FCDBC3D6E14D1FFuL, // c5
            0x3FC28671B66B7EBDuL, // c6
            0x3FC0BFBCFE643140uL, // c7
        },
        // Breakpoint B = 23 * pi/256, function tan(B + x)
        {
            0x3FF49DE02B1FECFEuL, // B' = pi/2 - B (high double)
            0xBC7616D2A8653209uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FD28F459ECAD74DuL, // c0 (high double)
            0x3C302C86D50F5140uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FB5875F92EDBDF4uL, // c1 (low double)
            0x3FD41ED6D05A0010uL, // c2
            0x3FDCF65481FC5BEBuL, // c3
            0x3FCE361FE028368DuL, // c4
            0x3FCF2728B941710DuL, // c5
            0x3FC3FFF67496086BuL, // c6
            0x3FC1A12109F041C7uL, // c7
        },
        // Breakpoint B = 24 * pi/256, function tan(B + x)
        {
            0x3FF46B9C347764A4uL, // B' = pi/2 - B (high double)
            0xBC91A900F67F753AuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FD36A08355C63DCuL, // c0 (high double)
            0xBC6D7145B6A0EB84uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FB78E922B82AB64uL, // c1 (low double)
            0x3FD5335ED87AB0D5uL, // c2
            0x3FDDBA417913D89CuL, // c3
            0x3FD015AB59BD876BuL, // c4
            0x3FD0576D1EF5DB86uL, // c5
            0x3FC594A6DD2CA170uL, // c6
            0x3FC2F51FBC1D0708uL, // c7
        },
        // Breakpoint B = 25 * pi/256, function tan(B + x)
        {
            0x3FF439583DCEDC4AuL, // B' = pi/2 - B (high double)
            0xBC9DCC4D42E59DF2uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FD4466D542BAC92uL, // c0 (high double)
            0xBC705073A04E3471uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FB9B14751F8BA20uL, // c1 (low double)
            0x3FD64F58590170A2uL, // c2
            0x3FDE8AC91D15CFCFuL, // c3
            0x3FD11CC2EF09C664uL, // c4
            0x3FD12AB5AB65B965uL, // c5
            0x3FC7471C9AC8155BuL, // c6
            0x3FC4826BA97C3E62uL, // c7
        },
        // Breakpoint B = 26 * pi/256, function tan(B + x)
        {
            0x3FF40714472653EFuL, // B' = pi/2 - B (high double)
            0x3C96106670B43955uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FD5248AE1701B17uL, // c0 (high double)
            0x3C7BE8CC944DBC57uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FBBF04005723CCCuL, // c1 (low double)
            0x3FD7733D1300258AuL, // c2
            0x3FDF688E6ACC7DF5uL, // c3
            0x3FD231520EA793C8uL, // c4
            0x3FD20E93C5183A32uL, // c5
            0x3FC91A331D55B3EEuL, // c6
            0x3FC651721E153056uL, // c7
        },
        // Breakpoint B = 27 * pi/256, function tan(B + x)
        {
            0x3FF3D4D0507DCB95uL, // B' = pi/2 - B (high double)
            0x3C83DA34489C213BuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FD604775FBB27DFuL, // c0 (high double)
            0x3C6AC647C9CF38D0uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FBE4C49866D2D78uL, // c1 (low double)
            0x3FD89F8D033946D5uL, // c2
            0x3FE02A20954EE23AuL, // c3
            0x3FD35466F88C5ABCuL, // c4
            0x3FD3044BAC5DC8C8uL, // c5
            0x3FCB110C3D6FEB5EuL, // c6
            0x3FC7761A8DBF67E9uL, // c7
        },
        // Breakpoint B = 28 * pi/256, function tan(B + x)
        {
            0x3FF3A28C59D5433BuL, // B' = pi/2 - B (high double)
            0xBC61B19140C0C0D5uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FD6E649F7D78649uL, // c0 (high double)
            0x3C678D2AC032531AuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FC0631EFCBE0D16uL, // c1 (low double)
            0x3FD9D4CEDF38DFCDuL, // c2
            0x3FE0A74F7488C3A6uL, // c3
            0x3FD487249EA3BD6CuL, // c4
            0x3FD40D3A9BDEFA27uL, // c5
            0x3FCD2F1CBD1B58BDuL, // c6
            0x3FC98CBDCEBDACE8uL, // c7
        },
        // Breakpoint B = 29 * pi/256, function tan(B + x)
        {
            0x3FF37048632CBAE1uL, // B' = pi/2 - B (high double)
            0xBC8CB2FCE8FC81A5uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FD7CA1A832D0F84uL, // c0 (high double)
            0xBC7EE5A71208FD7EuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FC1AF828B5ED674uL, // c1 (low double)
            0x3FDB13909C0269A2uL, // c2
            0x3FE12C3A0394B7FAuL, // c3
            0x3FD5CAC4A421216CuL, // c4
            0x3FD52AE625505253uL, // c5
            0x3FCF7831018A6159uL, // c6
            0x3FCB98AB42F54A08uL, // c7
        },
        // Breakpoint B = 30 * pi/256, function tan(B + x)
        {
            0x3FF33E046C843287uL, // B' = pi/2 - B (high double)
            0xBC9A7CCAC0E4698BuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FD8B00196B3D022uL, // c0 (high double)
            0xBC67341EE3041CF4uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FC30BCA73878656uL, // c1 (low double)
            0x3FDC5C67FF403B73uL, // c2
            0x3FE1B94E72981059uL, // c3
            0x3FD7209990BAE1F6uL, // c4
            0x3FD65EF72AC3A4FAuL, // c5
            0x3FD0F83D8377C340uL, // c6
            0x3FCD97555AF455A0uL, // c7
        },
        // Breakpoint B = 31 * pi/256, function tan(B + x)
        {
            0x3FF30BC075DBAA2CuL, // B' = pi/2 - B (high double)
            0x3C995FE8F2B56DBDuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FD998188E816BF0uL, // c0 (high double)
            0x3C73DB2BD13E24E8uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FC478794806DE66uL, // c1 (low double)
            0x3FDDAFF33BF96274uL, // c2
            0x3FE24F039543C659uL, // c3
            0x3FD88A113E071D2DuL, // c4
            0x3FD7AB3E661E65FAuL, // c5
            0x3FD24E4E1B8DC6C3uL, // c6
            0x3FD01ED5FCAC1031uL, // c7
        },
        // Breakpoint B = 32 * pi/256, function tan(B + x)
        {
            0x3FF2D97C7F3321D2uL, // B' = pi/2 - B (high double)
            0x3C8A79394C9E8A0AuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FDA827999FCEF32uL, // c0 (high double)
            0x3C708A11019E58E9uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FC5F619980C4336uL, // c1 (low double)
            0x3FDF0ED99BED9B65uL, // c2
            0x3FE2EDD996C1BBEBuL, // c3
            0x3FDA08B785C6EA59uL, // c4
            0x3FD911BC7B12902DuL, // c5
            0x3FD3C0DA5B1130BAuL, // c6
            0x3FD1B590151E4269uL, // c7
        },
        // Breakpoint B = 33 * pi/256, function tan(B + x)
        {
            0x3FF2A738888A9978uL, // B' = pi/2 - B (high double)
            0x3C5195059E91C4D2uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FDB6F3FC8C61E5BuL, // c0 (high double)
            0xBC6A1CC48267E6BEuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FC7853E65CE5524uL, // c1 (low double)
            0x3FE03CE61B6A9E63uL, // c2
            0x3FE3965ABF5C8FABuL, // c3
            0x3FDB9E3938F261E9uL, // c4
            0x3FDA94A771ED6FE6uL, // c5
            0x3FD552B7D4636639uL, // c6
            0x3FD2F58F20A08A92uL, // c7
        },
        // Breakpoint B = 34 * pi/256, function tan(B + x)
        {
            0x3FF274F491E2111EuL, // B' = pi/2 - B (high double)
            0xBC8613F7E4FA18D6uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FDC5E87185E67B6uL, // c0 (high double)
            0x3C4FEBA1F149DF00uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FC92683A68EB5D0uL, // c1 (low double)
            0x3FE0F8C35C7247C7uL, // c2
            0x3FE4491C4DCA262EuL, // c3
            0x3FDD4C67683395D6uL, // c4
            0x3FDC3667C15EBF94uL, // c5
            0x3FD7070B7FE443C4uL, // c6
            0x3FD4805841782C0CuL, // c7
        },
        // Breakpoint B = 35 * pi/256, function tan(B + x)
        {
            0x3FF242B09B3988C4uL, // B' = pi/2 - B (high double)
            0xBC972D483EE33523uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FDD506C82A2C800uL, // c0 (high double)
            0x3C6E169413E029AEuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FCADA8ECCC42C48uL, // c1 (low double)
            0x3FE1BB681D0BE8A7uL, // c2
            0x3FE506BF662527D3uL, // c3
            0x3FDF153B07CB2306uL, // c4
            0x3FDDF9A636394DC9uL, // c5
            0x3FD8E153A4A78673uL, // c6
            0x3FD6937B3797DBB1uL, // c7
        },
        // Breakpoint B = 36 * pi/256, function tan(B + x)
        {
            0x3FF2106CA4910069uL, // B' = pi/2 - B (high double)
            0x3C9CAF6B74B6A225uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FDE450E0D273E7AuL, // c0 (high double)
            0x3C74138BBC424A0DuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FCCA20F5D4879EAuL, // c1 (low double)
            0x3FE2853E13FBB7C7uL, // c2
            0x3FE5CFF21905E477uL, // c3
            0x3FE07D6C7D0DAC66uL, // c4
            0x3FDFE1550F550216uL, // c5
            0x3FDAE570CE7F6BD7uL, // c6
            0x3FD89328A3ABE601uL, // c7
        },
        // Breakpoint B = 37 * pi/256, function tan(B + x)
        {
            0x3FF1DE28ADE8780FuL, // B' = pi/2 - B (high double)
            0x3C908C1F2850796DuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FDF3C8AD985D9EEuL, // c0 (high double)
            0x3C455C31F5355658uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FCE7DBF907D8618uL, // c1 (low double)
            0x3FE356B5A1D9CA5AuL, // c2
            0x3FE6A57085697F99uL, // c3
            0x3FE17FCB46BE280BuL, // c4
            0x3FE0F858275BAA48uL, // c5
            0x3FDD17B47CA8A83BuL, // c6
            0x3FDA9B369A707A32uL, // c7
        },
        // Breakpoint B = 38 * pi/256, function tan(B + x)
        {
            0x3FF1ABE4B73FEFB5uL, // B' = pi/2 - B (high double)
            0x3C71A34B6FA942D3uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE01B819B5A7CF7uL, // c0 (high double)
            0x3C80BED996FBB65FuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD03732803B2A4CuL, // c1 (low double)
            0x3FE430465B55C6C0uL, // c2
            0x3FE78806185ACF72uL, // c3
            0x3FE292FF3DF61A34uL, // c4
            0x3FE215A449967BC2uL, // c5
            0x3FDF7CEF2946D3A8uL, // c6
            0x3FDD0F4FF3CD442FuL, // c7
        },
        // Breakpoint B = 39 * pi/256, function tan(B + x)
        {
            0x3FF179A0C097675BuL, // B' = pi/2 - B (high double)
            0xBC7EE9E5C1EF600DuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE09A4C59BD0D4DuL, // c0 (high double)
            0xBC451D94A7B70320uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD13A68B2A5E855uL, // c1 (low double)
            0x3FE5126F9FD8D89FuL, // c2
            0x3FE8788EEDC53F19uL, // c3
            0x3FE3B86B3E883B76uL, // c4
            0x3FE34A881E7F09C9uL, // c5
            0x3FE10D3EF07D106AuL, // c6
            0x3FDFEBFE6AA0C4E7uL, // c7
        },
        // Breakpoint B = 40 * pi/256, function tan(B + x)
        {
            0x3FF1475CC9EEDF01uL, // B' = pi/2 - B (high double)
            0xBC93DDC5BCE200BBuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE11AB7190834ECuL, // c0 (high double)
            0xBC8E577A07DB7162uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD248F1B07217E3uL, // c1 (low double)
            0x3FE5FDB93DD01D06uL, // c2
            0x3FE977F956462152uL, // c3
            0x3FE4F192BE91E40AuL, // c4
            0x3FE49932EEB2A51BuL, // c5
            0x3FE27B305435DE1BuL, // c6
            0x3FE176C2BDD3B0A4uL, // c7
        },
        // Breakpoint B = 41 * pi/256, function tan(B + x)
        {
            0x3FF11518D34656A6uL, // B' = pi/2 - B (high double)
            0x3C9FFEEDF6B7D68DuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE19CD3FE8E405DuL, // c0 (high double)
            0xBC84E5C1D482FB99uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD36343ADCAF866uL, // c1 (low double)
            0x3FE6F2B4260FF6A8uL, // c2
            0x3FEA8747864942E1uL, // c3
            0x3FE6401D47362936uL, // c4
            0x3FE6040F50A52FBFuL, // c5
            0x3FE40BA6192CC669uL, // c6
            0x3FE302E7DAB9BDE8uL, // c7
        },
        // Breakpoint B = 42 * pi/256, function tan(B + x)
        {
            0x3FF0E2D4DC9DCE4CuL, // B' = pi/2 - B (high double)
            0x3C93DBA1AA51ADD5uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE220B5EF047825uL, // c0 (high double)
            0x3C5AE1F6C713C510uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD489DC399ADDECuL, // c1 (low double)
            0x3FE7F1FB2FEBE8C6uL, // c2
            0x3FEBA791734D6CCEuL, // c3
            0x3FE7A5DA56BAB9DAuL, // c4
            0x3FE78DC9A17C17D7uL, // c5
            0x3FE5C262725357A1uL, // c6
            0x3FE4D1FE58052A33uL, // c7
        },
        // Breakpoint B = 43 * pi/256, function tan(B + x)
        {
            0x3FF0B090E5F545F2uL, // B' = pi/2 - B (high double)
            0x3C7EE15577AE1472uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE2A6709A74F289uL, // c0 (high double)
            0x3C76E3B98C26B440uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD5BD40BF249496uL, // c1 (low double)
            0x3FE8FC33EFCF6F3AuL, // c2
            0x3FECDA06E4CFC822uL, // c3
            0x3FE924C5BA1C7515uL, // c4
            0x3FE9395A86CEDBCCuL, // c5
            0x3FE7A39B6931C7C5uL, // c6
            0x3FE6D875BED7333EuL, // c7
        },
        // Breakpoint B = 44 * pi/256, function tan(B + x)
        {
            0x3FF07E4CEF4CBD98uL, // B' = pi/2 - B (high double)
            0xBC71ABDBB9EA8E6EuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE32E1889047FFDuL, // c0 (high double)
            0xBC810D08F0134DCFuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD6FDFF12A4297BuL, // c1 (low double)
            0x3FEA120FA25B9D83uL, // c2
            0x3FEE1FF1BF27E6ABuL, // c3
            0x3FEABF0C6E045EA0uL, // c4
            0x3FEB0A0E2CDCAB88uL, // c5
            0x3FE9B40D41660E8EuL, // c6
            0x3FE92FCC1848D932uL, // c7
        },
        // Breakpoint B = 45 * pi/256, function tan(B + x)
        {
            0x3FF04C08F8A4353EuL, // B' = pi/2 - B (high double)
            0xBC908E433AE0CC54uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE3B7C3289ED6F3uL, // c0 (high double)
            0xBC8C4CF3D327FEECuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD84CAE0A06DE7CuL, // c1 (low double)
            0x3FEB344C2E4C0F47uL, // c2
            0x3FEF7AB88F5F820CuL, // c3
            0x3FEC771219A99E98uL, // c4
            0x3FED0390B144EACBuL, // c5
            0x3FEBF90A32A17503uL, // c6
            0x3FEBAC50A2912BE1uL, // c7
        },
        // Breakpoint B = 46 * pi/256, function tan(B + x)
        {
            0x3FF019C501FBACE4uL, // B' = pi/2 - B (high double)
            0xBC9CB18F8746F50CuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE44386DB9CE5DBuL, // c0 (high double)
            0xBC8A7F79AD342CDEuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD9A9EE22E82AF2uL, // c1 (low double)
            0x3FEC63B53FACFD1BuL, // c2
            0x3FF075F0B010F1C9uL, // c3
            0x3FEE4F7738A37AC1uL, // c4
            0x3FEF29F7068A6E02uL, // c5
            0x3FEE78912F3E4890uL, // c6
            0x3FEE7DE5AD19FDA8uL, // c7
        },
        // Breakpoint B = 47 * pi/256, function tan(B + x)
        {
            0x3FEFCF0216A64913uL, // B' = pi/2 - B (high double)
            0xBC81A9B7A75A3B87uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE4D17B087B265DuL, // c0 (high double)
            0xBC788AAC4AE1F0D0uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FDB166A3723839CuL, // c1 (low double)
            0x3FEDA1257F3D5E87uL, // c2
            0x3FF13A8A70704F6BuL, // c3
            0x3FF025900612C493uL, // c4
            0x3FF0C0E778680DE8uL, // c5
            0x3FF09CB2B02DFCCBuL, // c6
            0x3FF0D681147681D3uL, // c7
        },
        // Breakpoint B = 48 * pi/256, function tan(B + x)
        {
            0x3FEF6A7A2955385EuL, // B' = pi/2 - B (high double)
            0x3C860FAFBFD97309uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE561B82AB7F990uL, // c0 (high double)
            0x3C87A5FA3E37FD12uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FDC92D84170BCC6uL, // c1 (low double)
            0x3FEEED87E934644CuL, // c2
            0x3FF20C10F4D7D072uL, // c3
            0x3FF1369E37DA5741uL, // c4
            0x3FF20816F61AB3EEuL, // c5
            0x3FF221953D8712EFuL, // c6
            0x3FF29CA76CD561F8uL, // c7
        },
        // Breakpoint B = 49 * pi/256, function tan(B + x)
        {
            0x3FEF05F23C0427AAuL, // B' = pi/2 - B (high double)
            0xBC51B746C796F33CuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE5F457E4F4812EuL, // c0 (high double)
            0x3C8F6EE06A43F015uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FDE1FFA33ABCB2DuL, // c1 (low double)
            0x3FF024ECA3857FDFuL, // c2
            0x3FF2EB80B3E2C969uL, // c3
            0x3FF25CA858ECE33EuL, // c4
            0x3FF36D6159DB2BC3uL, // c5
            0x3FF3CF427530AFA1uL, // c6
            0x3FF4914D16BF30D2uL, // c7
        },
        // Breakpoint B = 50 * pi/256, function tan(B + x)
        {
            0x3FEEA16A4EB316F6uL, // B' = pi/2 - B (high double)
            0xBC8A7D8171BF2FD8uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE6897514751DB6uL, // c0 (high double)
            0x3C7432E08569C822uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FDFBE9EE099229DuL, // c1 (low double)
            0x3FF0DB94E83152C9uL, // c2
            0x3FF3D9ED602549D9uL, // c3
            0x3FF3999FCB35B63CuL, // c4
            0x3FF4F3F4BB18D34DuL, // c5
            0x3FF5AA9FE2712A89uL, // c6
            0x3FF6B61F156D157EuL, // c7
        },
        // Breakpoint B = 51 * pi/256, function tan(B + x)
        {
            0x3FEE3CE261620641uL, // B' = pi/2 - B (high double)
            0x3C7A77CBEAE8FD71uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE7212BE621BE6DuL, // c0 (high double)
            0x3C8AD14CAC5AE1D4uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFDE905D04D13F70uL, // c1 (low double)
            0x3FF19B4F7BD63206uL, // c2
            0x3FF4D8845C8EEC38uL, // c3
            0x3FF4EFAAD70924AAuL, // c4
            0x3FF69F610EDFA788uL, // c5
            0x3FF7B93C8525966AuL, // c6
            0x3FF928F49138F1A9uL, // c7
        },
        // Breakpoint B = 52 * pi/256, function tan(B + x)
        {
            0x3FEDD85A7410F58DuL, // B' = pi/2 - B (high double)
            0xBC76156546AFA570uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE7BB99ED2990CFuL, // c0 (high double)
            0xBC8AABB1289366EEuL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFDCCC0DD257FE9FuL, // c1 (low double)
            0x3FF264BAB3E8C3FCuL, // c2
            0x3FF5E88F7CC1F7BCuL, // c3
            0x3FF6612B1AE228E1uL, // c4
            0x3FF873A64164BF00uL, // c5
            0x3FFA016987C44E2AuL, // c6
            0x3FFBFCA608E41B3AuL, // c7
        },
        // Breakpoint B = 53 * pi/256, function tan(B + x)
        {
            0x3FED73D286BFE4D8uL, // B' = pi/2 - B (high double)
            0x3C8CAEB4C3DBDBD8uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE858DE3B716571uL, // c0 (high double)
            0x3C8C6340D74EFAEEuL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFDAF377B50F3DE8uL, // c1 (low double)
            0x3FF338819C6A585BuL, // c2
            0x3FF70B781BB3F9F8uL, // c3
            0x3FF7F0C4DC952BEEuL, // c4
            0x3FFA75444794B0BEuL, // c5
            0x3FFC8A58AEEA152DuL, // c6
            0x3FFF1509C3EC473FuL, // c7
        },
        // Breakpoint B = 54 * pi/256, function tan(B + x)
        {
            0x3FED0F4A996ED424uL, // B' = pi/2 - B (high double)
            0x3C61A070AC3E29A0uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE8F9197BF85EEBuL, // c0 (high double)
            0xBC77971D706288A0uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFD9058D62BBB98DuL, // c1 (low double)
            0x3FF4175D31199B0CuL, // c2
            0x3FF842CA948070CEuL, // c3
            0x3FF9A1675E6F2233uL, // c4
            0x3FFCA94A75D1DF36uL, // c5
            0x3FFF5C3EE98CB77CuL, // c6
            0x4001549BBD7761F7uL, // c7
        },
        // Breakpoint B = 55 * pi/256, function tan(B + x)
        {
            0x3FECAAC2AC1DC370uL, // B' = pi/2 - B (high double)
            0xBC83DE7C6DBCC708uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FE99C6E0F634394uL, // c0 (high double)
            0x3C81A125C4B0D75FuL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFD7012EAAC8A987uL, // c1 (low double)
            0x3FF50215B95FD940uL, // c2
            0x3FF9903A2B193F49uL, // c3
            0x3FFB7656607D37C1uL, // c4
            0x3FFF156F461C0EE0uL, // c5
            0x4001403E48F2741BuL, // c6
            0x40034ACD8D38409AuL, // c7
        },
        // Breakpoint B = 56 * pi/256, function tan(B + x)
        {
            0x3FEC463ABECCB2BBuL, // B' = pi/2 - B (high double)
            0x3C83DAEAF976E788uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FEA43002AE42850uL, // c0 (high double)
            0xBC7A7C0406D92972uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFD4E526E33345B8uL, // c1 (low double)
            0x3FF5F9844B626284uL, // c2
            0x3FFAF5A574C2F3BFuL, // c3
            0x3FFD7334FD198D13uL, // c4
            0x4000E01251879DC7uL, // c5
            0x400300E6F5213CC2uL, // c6
            0x40058BA67D3A054AuL, // c7
        },
        // Breakpoint B = 57 * pi/256, function tan(B + x)
        {
            0x3FEBE1B2D17BA207uL, // B' = pi/2 - B (high double)
            0xBC61AEB67D55A7A1uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FEAECF5F9BA35A6uL, // c0 (high double)
            0x3C4535083409FF20uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFD2B02B2D565D55uL, // c1 (low double)
            0x3FF6FE947B408BACuL, // c2
            0x3FFC751B52A83938uL, // c3
            0x3FFF9C12181CFBBAuL, // c4
            0x4002585B4421C075uL, // c5
            0x4004F6412119D9F1uL, // c6
            0x400814BACED69AF1uL, // c7
        },
        // Breakpoint B = 58 * pi/256, function tan(B + x)
        {
            0x3FEB7D2AE42A9153uL, // B' = pi/2 - B (high double)
            0xBC8CB2463821BB58uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FEB9A77C18C1AF2uL, // c0 (high double)
            0x3C76EA194A3B7D14uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFD060D88DF903DFuL, // c1 (low double)
            0x3FF812463C417D93uL, // c2
            0x3FFE10E093E6DE8DuL, // c3
            0x4000FABB50DD022EuL, // c4
            0x4003F7B45B053F76uL, // c5
            0x4007275FB67DBBCCuL, // c6
            0x400B02CB76CD8FDBuL, // c7
        },
        // Breakpoint B = 59 * pi/256, function tan(B + x)
        {
            0x3FEB18A2F6D9809EuL, // B' = pi/2 - B (high double)
            0x3C760E425E23E66FuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FEC4BB009E77983uL, // c0 (high double)
            0xBC5346B6895525B8uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFCBEB63A6D04570uL, // c1 (low double)
            0x3FF935AFFA8DCC1DuL, // c2
            0x3FFFCB7657D15F0BuL, // c3
            0x4002423AFCE75F54uL, // c4
            0x4005C2CF61AD4B89uL, // c5
            0x40099C675A298F48uL, // c6
            0x400E49F38B657F73uL, // c7
        },
        // Breakpoint B = 60 * pi/256, function tan(B + x)
        {
            0x3FEAB41B09886FEAuL, // B' = pi/2 - B (high double)
            0xBC7A7EEED374BC71uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FED00CBC7384D2EuL, // c0 (high double)
            0x3C78C4FC635D2596uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFC6DA3A87611EA6uL, // c1 (low double)
            0x3FFA6A00F50D3A6FuL, // c2
            0x4000D3D0A6ABBD46uL, // c3
            0x4003A760672280D0uL, // c4
            0x4007BF026C103550uL, // c5
            0x400C5EBD23F4E2D6uL, // c6
            0x4011024DB7867574uL, // c7
        },
        // Breakpoint B = 61 * pi/256, function tan(B + x)
        {
            0x3FEA4F931C375F35uL, // B' = pi/2 - B (high double)
            0x3C8A79EFFD795057uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FEDB9FA89953FCFuL, // c0 (high double)
            0x3C286C68F5592C40uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFC18AC418849A6EuL, // c1 (low double)
            0x3FFBB083E02786D7uL, // c2
            0x4001D438F8A20402uL, // c3
            0x40052D5D6FF7A4BDuL, // c4
            0x4009F262067BAA5AuL, // c5
            0x400F793DA8608BD3uL, // c6
            0x401328BFC5F60A12uL, // c7
        },
        // Breakpoint B = 62 * pi/256, function tan(B + x)
        {
            0x3FE9EB0B2EE64E81uL, // B' = pi/2 - B (high double)
            0x3C519ABB2567F739uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FEE776EAFC91706uL, // c0 (high double)
            0x3C890664EDC679F9uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFB7F295AD658764uL, // c1 (low double)
            0x3FFD0AA1DB7F5E58uL, // c2
            0x4002E8A6F9693556uL, // c3
            0x4006D7CA16B298C6uL, // c4
            0x400C63DFCAA3FB6CuL, // c5
            0x40117C3EE50D7733uL, // c6
            0x401599932C96B999uL, // c7
        },
        // Breakpoint B = 63 * pi/256, function tan(B + x)
        {
            0x3FE9868341953DCDuL, // B' = pi/2 - B (high double)
            0xBC861341341F5289uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0x3FEF395D9F0E3C92uL, // c0 (high double)
            0xBC550BEC65673F08uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFA8873C53BE0540uL, // c1 (low double)
            0x3FFE79E5C65202DEuL, // c2
            0x400412FD76797C7DuL, // c3
            0x4008AAB30DC82D37uL, // c4
            0x400F1B6C68B4AEA4uL, // c5
            0x4013758CE9BD9EE9uL, // c6
            0x40186872A06858F5uL, // c7
        },
        // Breakpoint B = 64 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE921FB54442D18uL, // B' = pi/2 - B (high double)
            0x3C81A62633145C07uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFD17CC1B727220BuL, // c0 (high double)
            0x3C7AC07996DF1E94uL, // c0 (low double)
            0x3FE0000000000000uL, // c1 (high single bit)
            0xBFBF02F6222C71FBuL, // c1 (low double)
            0xBFB068BD68BAEFC3uL, // c2
            0x3FA3C0213E30B2A4uL, // c3
            0xBF8A5494A0DD32B3uL, // c4
            0x3F7940CD1D8E923EuL, // c5
            0xBF63CC1BE5088783uL, // c6
            0x3F3FA4A6A27659EEuL, // c7
        },
        // Breakpoint B = 65 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE8BD7366F31C64uL, // B' = pi/2 - B (high double)
            0xBC6A81C996DFD5A5uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFD130BD396BB4A9uL, // c0 (high double)
            0x3C657D8592EA0982uL, // c0 (low double)
            0x3FE0000000000000uL, // c1 (high single bit)
            0xBFBF68ED595308D6uL, // c1 (low double)
            0xBFB00C6BF6A673E6uL, // c2
            0x3FA36E9DA9556AA6uL, // c3
            0xBF8991120C7D4BB9uL, // c4
            0x3F7889ABD600821AuL, // c5
            0xBF630EE40468887BuL, // c6
            0x3F52CAD75A194915uL, // c7
        },
        // Breakpoint B = 66 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE858EB79A20BB0uL, // B' = pi/2 - B (high double)
            0xBC8EE70AFE8446D9uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFD0E507EE8B93D7uL, // c0 (high double)
            0xBC679F09CDD5D366uL, // c0 (low double)
            0x3FE0000000000000uL, // c1 (high single bit)
            0xBFBFCCA52E2AE201uL, // c1 (low double)
            0xBFAF632A1681FD0BuL, // c2
            0x3FA31F7794BDF922uL, // c3
            0xBF88D31259E30525uL, // c4
            0x3F77D97B728544D6uL, // c5
            0xBF62591A3A2B69D8uL, // c6
            0x3F3605B3887E1AF3uL, // c7
        },
        // Breakpoint B = 67 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE7F4638C50FAFBuL, // B' = pi/2 - B (high double)
            0x3C71A4B8D15ECF6DuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFD099A0163BCD3FuL, // c0 (high double)
            0xBC7A7F11D963D761uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBFD1D937A1F094uL, // c1 (low double)
            0xBFAEB05B52EFB4C0uL, // c2
            0x3FA2D29E032A57DBuL, // c3
            0xBF881A60F7D38723uL, // c4
            0x3F772FBBF1BFA292uL, // c5
            0xBF61AA745B27BAF7uL, // c6
            0x3F51CC1913635CA5uL, // c7
        },
        // Breakpoint B = 68 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE78FDB9EFFEA47uL, // B' = pi/2 - B (high double)
            0xBC7EE8786039D373uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFD04E83F7474877uL, // c0 (high double)
            0xBC73B117AC20AD03uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBF7284F47C0E61uL, // c1 (low double)
            0xBFAE0056458FCED5uL, // c2
            0x3FA28800992AEEE6uL, // c3
            0xBF8766CB6439B11DuL, // c4
            0x3F768C54DBB3684BuL, // c5
            0xBF610297B13942A7uL, // c6
            0x3F5536BD1C2C92E8uL, // c7
        },
        // Breakpoint B = 69 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE72B53B1AED992uL, // B' = pi/2 - B (high double)
            0x3C88452B3716C4D6uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFD003B1DF5A0F86uL, // c0 (high double)
            0xBC716CEB71FB7130uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBF155566FA6884uL, // c1 (low double)
            0xBFAD530628A37C7AuL, // c2
            0x3FA23F8F96E80A99uL, // c3
            0xBF86B82113BDC04EuL, // c4
            0x3F75EF0B9C9DF1C1uL, // c5
            0xBF60613421CB6A8BuL, // c6
            0x3F48E48E9081596FuL, // c7
        },
        // Breakpoint B = 70 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE6C6CBC45DC8DEuL, // B' = pi/2 - B (high double)
            0xBBC6D61B58C99C43uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFCF7250459C1976uL, // c0 (high double)
            0xBC40B5B081F067D4uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBEBA422E0FE0E5uL, // c1 (low double)
            0xBFACA856C815D042uL, // c2
            0x3FA1F93BD218DCE9uL, // c3
            0xBF860E335AB42933uL, // c4
            0x3F75578EEA5D18CCuL, // c5
            0xBF5F8BFBAD564EB5uL, // c6
            0x3F434A80D203FDB7uL, // c7
        },
        // Breakpoint B = 71 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE66243D70CB82AuL, // B' = pi/2 - B (high double)
            0xBC884805FA81DE0AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFCEDDCA38F25FECuL, // c0 (high double)
            0x3C65AACB9E8B4E8CuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBE6143274A35BFuL, // c1 (low double)
            0xBFAC00347C15ABB2uL, // c2
            0x3FA1B4F6B03A62C8uL, // c3
            0xBF8568D557225239uL, // c4
            0x3F74C5A8346B014EuL, // c5
            0xBF5E61680C7E3887uL, // c6
            0x3F45A81E03A2004AuL, // c7
        },
        // Breakpoint B = 72 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE5FDBBE9BBA775uL, // B' = pi/2 - B (high double)
            0x3C7EE2C2D963A10CuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFCE49CE5AF9DC29uL, // c0 (high double)
            0x3C382336F1C6B420uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBE0A506D218F3CuL, // c1 (low double)
            0xBFAB5A8C23E49FEAuL, // c2
            0x3FA172B22117D625uL, // c3
            0xBF84C7DBDBF304B9uL, // c4
            0x3F74391DF553E941uL, // c5
            0xBF5D4211231EF20FuL, // c6
            0x3F52AB1CCD6F9451uL, // c7
        },
        // Breakpoint B = 73 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE59933FC6A96C1uL, // B' = pi/2 - B (high double)
            0xBC71AA6E583501D4uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFCDB6597A2E1B61uL, // c0 (high double)
            0x3C6CB71A947AAB6BuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBDB56255580CA4uL, // c1 (low double)
            0xBFAAB74B20D856E7uL, // c2
            0x3FA1326099896038uL, // c3
            0xBF842B1D5CF84F74uL, // c4
            0x3F73B1CC2C4E32E8uL, // c5
            0xBF5C2D7D75B7FBCEuL, // c6
            0x3F53DB94CE05D10AuL, // c7
        },
        // Breakpoint B = 74 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE534AC0F19860CuL, // B' = pi/2 - B (high double)
            0x3C8EE4303B192DA6uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFCD236870E769B5uL, // c0 (high double)
            0xBC523CE78218C3F0uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBD62716F68B5FEuL, // c1 (low double)
            0xBFAA165F518C38ABuL, // c2
            0x3FA0F3F50E789484uL, // c3
            0xBF839271DC18DB75uL, // c4
            0x3F732F8BAC4C5496uL, // c5
            0xBF5B23248F865BB8uL, // c6
            0x3F36E307E32E173BuL, // c7
        },
        // Breakpoint B = 75 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE4D02421C87558uL, // B' = pi/2 - B (high double)
            0x3C6A765E893370D7uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFCC90F8250157F7uL, // c0 (high double)
            0xBC657D8312BFEA16uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBD117683053E13uL, // c1 (low double)
            0xBFA977B70D411C29uL, // c2
            0x3FA0B762F022F359uL, // c3
            0xBF82FDB2D7324A8FuL, // c4
            0x3F72B2030A36AAE0uL, // c5
            0xBF5A22A22953FC15uL, // c6
            0x3F473A2F09CB1310uL, // c7
        },
        // Breakpoint B = 76 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE46B9C347764A4uL, // B' = pi/2 - B (high double)
            0xBC81A900F67F753AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFCBFF05878394D3uL, // c0 (high double)
            0xBC486912F813DEBCuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBCC26A8EA2077FuL, // c1 (low double)
            0xBFA8DB411F68EE69uL, // c2
            0x3FA07C9E25737218uL, // c3
            0xBF826CBB3713A292uL, // c4
            0x3F72392A32F26712uL, // c5
            0xBF592B76EEB6E333uL, // c6
            0x3F39BADB57E2A4EBuL, // c7
        },
        // Breakpoint B = 77 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE40714472653EFuL, // B' = pi/2 - B (high double)
            0x3C86106670B43955uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFCB6D8D944D0371uL, // c0 (high double)
            0x3C6CDD9D6EA4B883uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBC7546C60FE5C7uL, // c1 (low double)
            0xBFA840ECC35C53FDuL, // c2
            0x3FA0439B07C380F3uL, // c3
            0xBF81DF673F038AB5uL, // c4
            0x3F71C4B3C36B8A2BuL, // c5
            0xBF583D44FA2FB1ACuL, // c6
            0x3F52EE3E6107070CuL, // c7
        },
        // Breakpoint B = 78 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE3A28C59D5433BuL, // B' = pi/2 - B (high double)
            0xBC51B19140C0C0D5uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFCADC8D51C0FA5BuL, // c0 (high double)
            0xBC54B350B5D39454uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBC2A04912319A8uL, // c1 (low double)
            0xBFA7A8A9A038608EuL, // c2
            0x3FA00C4E5E972915uL, // c3
            0xBF8155947D4C5C83uL, // c4
            0x3F7154A7B149DA51uL, // c5
            0xBF5757960EDFC478uL, // c6
            0x3F3F895707140B54uL, // c7
        },
        // Breakpoint B = 79 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE33E046C843287uL, // B' = pi/2 - B (high double)
            0xBC8A7CCAC0E4698BuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFCA4C01D0769648uL, // c0 (high double)
            0xBC4DDC96CAE4A560uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBBE09D8A670D72uL, // c1 (low double)
            0xBFA71267C4E29753uL, // c2
            0x3F9FAD5AB78BB764uL, // c3
            0xBF80CF21BC414F68uL, // c4
            0x3F70E8AC1DCF4364uL, // c5
            0xBF567A14C0530821uL, // c6
            0x3F4F621D6DA2AD4CuL, // c7
        },
        // Breakpoint B = 80 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE2D97C7F3321D2uL, // B' = pi/2 - B (high double)
            0x3C7A79394C9E8A0AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC9BBE82AEA0D12uL, // c0 (high double)
            0x3C62D116CA770BB7uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBB990B7DDE5BFCuL, // c1 (low double)
            0xBFA67E17A4317C6CuL, // c2
            0x3F9F455B2F170454uL, // c3
            0xBF804BEEF437A2B3uL, // c4
            0x3F7080CEADF2974DuL, // c5
            0xBF55A44E3A71E98CuL, // c6
            0xBF039F9468E5AC7CuL, // c7
        },
        // Breakpoint B = 81 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE274F491E2111EuL, // B' = pi/2 - B (high double)
            0xBC7613F7E4FA18D6uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC92C3D852FEDF2uL, // c0 (high double)
            0xBC20F5E7403E63E0uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBB534867CEB1CBuL, // c1 (low double)
            0xBFA5EBAA113813D4uL, // c2
            0x3F9EE08A1A202CC5uL, // c3
            0xBF7F97BA7BA1301AuL, // c4
            0x3F701CAD7F191E00uL, // c5
            0xBF54D60D132212FEuL, // c6
            0x3F45F5FE99746255uL, // c7
        },
        // Breakpoint B = 82 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE2106CA4910069uL, // B' = pi/2 - B (high double)
            0x3C8CAF6B74B6A225uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC89CFF0CAA3CB4uL, // c0 (high double)
            0x3C6A8D50D06FC30EuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBB0F4E73981D4BuL, // c1 (low double)
            0xBFA55B103BB2BD6DuL, // c2
            0x3F9E7ED42D91EC8DuL, // c3
            0xBF7E9D9D8AAE7B14uL, // c4
            0x3F6F78B5039EFACAuL, // c5
            0xBF540EDBA3A7865EuL, // c6
            0x3F3AA2CB5E8CB7D1uL, // c7
        },
        // Breakpoint B = 83 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE1ABE4B73FEFB5uL, // B' = pi/2 - B (high double)
            0x3C61A34B6FA942D3uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC80E29F7BF5633uL, // c0 (high double)
            0xBC67F8F06ACC6DE1uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBACD17FA976786uL, // c1 (low double)
            0xBFA4CC3BAC93F075uL, // c2
            0x3F9E2026D6967836uL, // c3
            0xBF7DA94D7C694B78uL, // c4
            0x3F6EBF4F14B98A4DuL, // c5
            0xBF534E7211F8744BuL, // c6
            0xBF2290D134413454uL, // c7
        },
        // Breakpoint B = 84 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE1475CC9EEDF01uL, // B' = pi/2 - B (high double)
            0xBC83DDC5BCE200BBuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC77FBB85928D2DuL, // c0 (high double)
            0x3C6121F4E1AC8205uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBA8C9F83131391uL, // c1 (low double)
            0xBFA43F1E42AF6970uL, // c2
            0x3F9DC4703433CA75uL, // c3
            0xBF7CBA92AE7FE396uL, // c4
            0x3F6E0CB6D0475CBFuL, // c5
            0xBF529491FB8790F1uL, // c6
            0x3F37152AFD512A19uL, // c7
        },
        // Breakpoint B = 85 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE0E2D4DC9DCE4CuL, // B' = pi/2 - B (high double)
            0x3C83DBA1AA51ADD5uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC6F1B0FDBE6EDEuL, // c0 (high double)
            0xBC614F63025E350FuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBA4DDFBF32A727uL, // c1 (low double)
            0xBFA3B3AA2F826E47uL, // c2
            0x3F9D6B9F112A3CA9uL, // c3
            0xBF7BD1375F2B326EuL, // c4
            0x3F6D60EF002A5944uL, // c5
            0xBF51E0E0E3E64206uL, // c6
            0x3F32BFA3B8DB7E72uL, // c7
        },
        // Breakpoint B = 86 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE07E4CEF4CBD98uL, // B' = pi/2 - B (high double)
            0xBC61ABDBB9EA8E6EuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC66407B0109FA6uL, // c0 (high double)
            0xBC55C3EC26D04386uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBA10D38BFFE319uL, // c1 (low double)
            0xBFA329D1F417DBD6uL, // c2
            0x3F9D15A2DE544C7CuL, // c3
            0xBF7AED07977D3236uL, // c4
            0x3F6CBBAD3C53FDBFuL, // c5
            0xBF513318B342B4E5uL, // c6
            0x3F1AF93D1229A6F6uL, // c7
        },
        // Breakpoint B = 87 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FE019C501FBACE4uL, // B' = pi/2 - B (high double)
            0xBC8CB18F8746F50CuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC5D6BCF447403EuL, // c0 (high double)
            0x3C6C6C662876B222uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB9D575F071964AuL, // c1 (low double)
            0xBFA2A1885E06BEACuL, // c2
            0x3F9CC26BAD1B38B8uL, // c3
            0xBF7A0DD116DA5229uL, // c4
            0x3F6C1C921E809C9EuL, // c5
            0xBF508B03CEF8B07AuL, // c6
            0x3F4BCE57D03F879AuL, // c7
        },
        // Breakpoint B = 88 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FDF6A7A2955385EuL, // B' = pi/2 - B (high double)
            0x3C760FAFBFD97309uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC549CE29CFCCBEuL, // c0 (high double)
            0x3C69881846E0C1CCuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB99BC21C7FB9B2uL, // c1 (low double)
            0xBFA21AC0848A46C5uL, // c2
            0x3F9C71EA2A16819DuL, // c3
            0xBF7933633F8F0072uL, // c4
            0x3F6B83D1C55099C3uL, // c5
            0xBF4FD088727B159EuL, // c6
            0xBF2306C94816B8C4uL, // c7
        },
        // Breakpoint B = 89 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FDEA16A4EB316F6uL, // B' = pi/2 - B (high double)
            0xBC7A7D8171BF2FD8uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC4BD38B7876716uL, // c0 (high double)
            0xBC5504069A91A9E0uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB963B368408780uL, // c1 (low double)
            0xBFA1956DC5B1F35AuL, // c2
            0x3F9C240F984C07DCuL, // c3
            0xBF785D8F0388F11CuL, // c4
            0x3F6AF0B073DF1CCAuL, // c5
            0xBF4E9580D70D34E0uL, // c6
            0x3F4571315C44A425uL, // c7
        },
        // Breakpoint B = 90 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FDDD85A7410F58DuL, // B' = pi/2 - B (high double)
            0xBC66156546AFA570uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC430FA0B7C7F15uL, // c0 (high double)
            0xBC63115D3C69C624uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB92D45530E4319uL, // c1 (low double)
            0xBFA11183C3A8DFF6uL, // c2
            0x3F9BD8CDCC073185uL, // c3
            0xBF778C26D2F52E9AuL, // c4
            0x3F6A638650FB676DuL, // c5
            0xBF4D6420527BDB5BuL, // c6
            0xBF2DA5A645CA3434uL, // c7
        },
        // Breakpoint B = 91 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FDD0F4A996ED424uL, // B' = pi/2 - B (high double)
            0x3C51A070AC3E29A0uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC3A50F9AB1CA91uL, // c0 (high double)
            0xBC50CA3B48D907FAuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB8F87382B56ACBuL, // c1 (low double)
            0xBFA08EF662132882uL, // c2
            0x3F9B901726A3CD1CuL, // c3
            0xBF76BEFE8A4461A2uL, // c4
            0x3F69DBAE42D8A416uL, // c5
            0xBF4C3C52357BF011uL, // c6
            0x3F32456D9E7FA4E8uL, // c7
        },
        // Breakpoint B = 92 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FDC463ABECCB2BBuL, // B' = pi/2 - B (high double)
            0x3C73DAEAF976E788uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC31976E0E280B3uL, // c0 (high double)
            0x3C667516EE4F26F2uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB8C539C2AB0F89uL, // c1 (low double)
            0xBFA00DB9C3806462uL, // c2
            0x3F9B49DE91DB8E44uL, // c3
            0xBF75F5EB6258C838uL, // c4
            0x3F695952C6BEE603uL, // c5
            0xBF4B1D691B1E9F12uL, // c6
            0xBEFC715673917907uL, // c7
        },
        // Breakpoint B = 93 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FDB7D2AE42A9153uL, // B' = pi/2 - B (high double)
            0xBC7CB2463821BB58uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC28E2D6047CBC8uL, // c0 (high double)
            0x3C0CEAB2537EA300uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB89394034B11CFuL, // c1 (low double)
            0xBF9F1B848DE48876uL, // c2
            0x3F9B06177BDD3821uL, // c3
            0xBF7530C3E0150AA0uL, // c4
            0x3F68DBF62FBA70ECuL, // c5
            0xBF4A0736C860B400uL, // c6
            0x3F50A3F1D92A1F47uL, // c7
        },
        // Breakpoint B = 94 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FDAB41B09886FEAuL, // B' = pi/2 - B (high double)
            0xBC6A7EEED374BC71uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC20330A15F6568uL, // c0 (high double)
            0xBC61BB4F210EA632uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB8637E591E05FEuL, // c1 (low double)
            0xBF9E1E090AECCE81uL, // c2
            0x3F9AC4B5D3090946uL, // c3
            0xBF746F5FC5C316BAuL, // c4
            0x3F686400E51F5377uL, // c5
            0xBF48F8EADC3DBAD1uL, // c6
            0xBF320AB3DE6FDA08uL, // c7
        },
        // Breakpoint B = 95 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FD9EB0B2EE64E81uL, // B' = pi/2 - B (high double)
            0x3C419ABB2567F739uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC1787E32B35112uL, // c0 (high double)
            0x3C5D2D85EBE7CEA8uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB834F4FC268515uL, // c1 (low double)
            0xBF9D22EA9FA50E7DuL, // c2
            0x3F9A85AE02819BA3uL, // c3
            0xBF73B19803AB2B21uL, // c4
            0x3F67F0B8C64045E8uL, // c5
            0xBF47F293167B415AuL, // c6
            0xBF198BF45BBDE600uL, // c7
        },
        // Breakpoint B = 96 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FD921FB54442D18uL, // B' = pi/2 - B (high double)
            0x3C71A62633145C07uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC0EE13A8A2A9C6uL, // c0 (high double)
            0x3C5830A63DDC3262uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB807F44735B1A4uL, // c1 (low double)
            0xBF9C2A135884458AuL, // c2
            0x3F9A48F4EE376E9DuL, // c3
            0xBF72F746AAC73A47uL, // c4
            0x3F678215824701EAuL, // c5
            0xBF46F3C719FB6619uL, // c6
            0x3F4BA88D5E69B64EuL, // c7
        },
        // Breakpoint B = 97 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FD858EB79A20BB0uL, // B' = pi/2 - B (high double)
            0xBC7EE70AFE8446D9uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFC063EE9D2B777CuL, // c0 (high double)
            0xBC4E1787D8B78D9CuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB7DC78B746BB2FuL, // c1 (low double)
            0xBF9B336DA889AC7DuL, // c2
            0x3F9A0E7FEF8737D6uL, // c3
            0xBF724046DF328844uL, // c4
            0x3F67185639AB9961uL, // c5
            0xBF45FBD25ACB3AEDuL, // c6
            0x3F0E765A16190A6DuL, // c7
        },
        // Breakpoint B = 98 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FD78FDB9EFFEA47uL, // B' = pi/2 - B (high double)
            0xBC6EE8786039D373uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFBFB4195F6B0333uL, // c0 (high double)
            0xBC47DB89E27050CEuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB7B27EEAE12CDBuL, // c1 (low double)
            0xBF9A3EE464F938D9uL, // c2
            0x3F99D644D21E7AE8uL, // c3
            0xBF718C74CAAA7BA1uL, // c4
            0x3F66B2F778B94CFDuL, // c5
            0xBF450AB96EF7378BuL, // c6
            0x3F1412EF7D22C6CAuL, // c7
        },
        // Breakpoint B = 99 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FD6C6CBC45DC8DEuL, // B' = pi/2 - B (high double)
            0xBBB6D61B58C99C43uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFBEA0D709BC27CAuL, // c0 (high double)
            0xBC48928D103E7F70uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB78A03A181D6ACuL, // c1 (low double)
            0xBF994C62C1366C88uL, // c2
            0x3F99A039D09A41ADuL, // c3
            0xBF70DBAD90C6529AuL, // c4
            0x3F66520A2F7F4662uL, // c5
            0xBF441FF58F223C4FuL, // c6
            0xBF357CCF18E00FA5uL, // c7
        },
        // Breakpoint B = 100 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FD5FDBBE9BBA775uL, // B' = pi/2 - B (high double)
            0x3C6EE2C2D963A10CuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFBD8E118C89577FuL, // c0 (high double)
            0xBC3344F06C36BFACuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB76303BB0A22C5uL, // c1 (low double)
            0xBF985BD44ABC0416uL, // c2
            0x3F996C5591BA6DFAuL, // c3
            0xBF702DCF4265B5B3uL, // c4
            0x3F65F52766DBD146uL, // c5
            0xBF433B879649EE5CuL, // c6
            0x3F47C67A932EC95BuL, // c7
        },
        // Breakpoint B = 101 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FD534AC0F19860CuL, // B' = pi/2 - B (high double)
            0x3C7EE4303B192DA6uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFBC7BC444B10758uL, // c0 (high double)
            0x3C511540B2268E0EuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB73D7C3735AA80uL, // c1 (low double)
            0xBF976D24E52F1F79uL, // c2
            0x3F993A8F25560FF4uL, // c3
            0xBF6F0571A586F6F3uL, // c4
            0x3F659C8E8181F3F0uL, // c5
            0xBF425CCABE1033EAuL, // c6
            0x3F4D59CA0EE986DFuL, // c7
        },
        // Breakpoint B = 102 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FD46B9C347764A4uL, // B' = pi/2 - B (high double)
            0xBC71A900F67F753AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFBB69EA984DE6EFuL, // c0 (high double)
            0x3C573CB7437F1792uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB7196A3515E12EuL, // c1 (low double)
            0xBF968040C68C8FE8uL, // c2
            0x3F990ADE01D185E6uL, // c3
            0xBF6DB49417B3E65CuL, // c4
            0x3F6548393E856BDBuL, // c5
            0xBF418353F782833AuL, // c6
            0xBF36489E03176323uL, // c7
        },
        // Breakpoint B = 103 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FD3A28C59D5433BuL, // B' = pi/2 - B (high double)
            0xBC41B19140C0C0D5uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFBA587FF6553726uL, // c0 (high double)
            0x3C5559CFE8366BA8uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB6F6CAF293AC7FuL, // c1 (low double)
            0xBF959514736F007FuL, // c2
            0x3F98DD3A019E14EBuL, // c3
            0xBF6C68C706B3D0FBuL, // c4
            0x3F64F779DB6C50F9uL, // c5
            0xBF40AF7CD7497BC1uL, // c6
            0x3F4AD3B98213250EuL, // c7
        },
        // Breakpoint B = 104 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FD2D97C7F3321D2uL, // B' = pi/2 - B (high double)
            0x3C6A79394C9E8A0AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFB9477FD636A4D8uL, // c0 (high double)
            0x3C5063C8B22C4842uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB6D59BCBF6C48BuL, // c1 (low double)
            0xBF94AB8CBB6CBB16uL, // c2
            0x3F98B19B6070D56DuL, // c3
            0xBF6B21CD22DB24B2uL, // c4
            0x3F64AAE4575F07CBuL, // c5
            0xBF3FC08B3806F47FuL, // c6
            0xBF00E427650BA68AuL, // c7
        },
        // Breakpoint B = 105 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FD2106CA4910069uL, // B' = pi/2 - B (high double)
            0x3C7CAF6B74B6A225uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFB836E5B77D959AuL, // c0 (high double)
            0xBC51BDCEFF6925ECuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB6B5DA3B72B738uL, // c1 (low double)
            0xBF93C396B58BD74AuL, // c2
            0x3F9887FAB9649830uL, // c3
            0xBF69DF6A8A7C820CuL, // c4
            0x3F6461D2011761C1uL, // c5
            0xBF3E2C1220D97FBEuL, // c6
            0x3F49F3B20EDA3AFFuL, // c7
        },
        // Breakpoint B = 106 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FD1475CC9EEDF01uL, // B' = pi/2 - B (high double)
            0xBC73DDC5BCE200BBuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFB726AD2173D50EuL, // c0 (high double)
            0xBC1FCEC90C8BEF70uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB69783D8B96CDFuL, // c1 (low double)
            0xBF92DD1FBCCBA937uL, // c2
            0x3F986051045C3883uL, // c3
            0xBF68A164B9409BD2uL, // c4
            0x3F641CB77D330F8FuL, // c5
            0xBF3C9FE910D9CBC2uL, // c6
            0xBF302CCA98190297uL, // c7
        },
        // Breakpoint B = 107 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FD07E4CEF4CBD98uL, // B' = pi/2 - B (high double)
            0xBC51ABDBB9EA8E6EuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFB616D1A2C591A8uL, // c0 (high double)
            0x3C2EF832FD935ED0uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB67A9658930D84uL, // c1 (low double)
            0xBF91F8156CC25284uL, // c2
            0x3F983A9794563574uL, // c3
            0xBF67678272C8ECDDuL, // c4
            0x3F63DAE696E3A38CuL, // c5
            0xBF3B1CDBB342EE5DuL, // c6
            0x3F49E00A3B463CAFuL, // c7
        },
        // Breakpoint B = 108 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FCF6A7A2955385EuL, // B' = pi/2 - B (high double)
            0x3C660FAFBFD97309uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFB5074ED126983EuL, // c0 (high double)
            0x3C516E0EB4132D8CuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB65F0F8C7B2809uL, // c1 (low double)
            0xBF9114659E4D5E41uL, // c2
            0x3F9816C815084CDAuL, // c3
            0xBF66318BB353364DuL, // c4
            0x3F639CED9C49E6E4uL, // c5
            0xBF39A0F6E494228EuL, // c6
            0xBF2EBBDA5946D8F5uL, // c7
        },
        // Breakpoint B = 109 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FCDD85A7410F58DuL, // B' = pi/2 - B (high double)
            0xBC56156546AFA570uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFB3F82048F8BE00uL, // c0 (high double)
            0xBC4AFD1EBA1D1F08uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB644ED6242FE19uL, // c1 (low double)
            0xBF9031FE64545C32uL, // c2
            0x3F97F4DC89754A8CuL, // c3
            0xBF64FF499B056453uL, // c4
            0x3F636226CB0F9286uL, // c5
            0xBF382D35D5867708uL, // c6
            0x3F3E29EBBEAD3F41uL, // c7
        },
        // Breakpoint B = 110 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FCC463ABECCB2BBuL, // B' = pi/2 - B (high double)
            0x3C63DAEAF976E788uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFB2E941ACF368E6uL, // c0 (high double)
            0x3C28FF4CA630E160uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB62C2DE3B8D982uL, // c1 (low double)
            0xBF8EA19C1138E963uL, // c2
            0x3F97D4CF49C89DB5uL, // c3
            0xBF63D0865FBE1BACuL, // c4
            0x3F632AE80D59D97DuL, // c5
            0xBF36C01A5963B74DuL, // c6
            0x3F2F2207CBA79A42uL, // c7
        },
        // Breakpoint B = 111 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FCAB41B09886FEAuL, // B' = pi/2 - B (high double)
            0xBC5A7EEED374BC71uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFB1DAAEA5CC26DDuL, // c0 (high double)
            0x3C51577CBB7A4D94uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB614CF365450F4uL, // c1 (low double)
            0xBF8CE1861157F322uL, // c2
            0x3F97B69B01FB3B26uL, // c3
            0xBF62A50D3A3C528BuL, // c4
            0x3F62F6E9071D40DBuL, // c5
            0xBF3559C86E0B83FEuL, // c6
            0x3F41AFC8FEE8B00AuL, // c7
        },
        // Breakpoint B = 112 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FC921FB54442D18uL, // B' = pi/2 - B (high double)
            0x3C61A62633145C07uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFB0CC62E1E04467uL, // c0 (high double)
            0xBC47805D7AE2B26CuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB5FECF9AE763EAuL, // c1 (low double)
            0xBF8B23982579F8E5uL, // c2
            0x3F979A3AB01644F9uL, // c3
            0xBF617CAA57838224uL, // c4
            0x3F62C658DCA7F4ABuL, // c5
            0xBF33F9251280A953uL, // c6
            0xBF2D5C4E17766D5AuL, // c7
        },
        // Breakpoint B = 113 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FC78FDB9EFFEA47uL, // B' = pi/2 - B (high double)
            0xBC5EE8786039D373uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFAF7CB429BEA72CuL, // c0 (high double)
            0xBC3FFC35F2771324uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB5EA2D6D5456CAuL, // c1 (low double)
            0xBF8967B0057BF87EuL, // c2
            0x3F977FA9A2ED4D75uL, // c3
            0xBF60572AC7A74374uL, // c4
            0x3F6298E78AA42A2CuL, // c5
            0xBF329E8594822820uL, // c6
            0xBF37DBAE80B80EC8uL, // c7
        },
        // Breakpoint B = 114 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FC5FDBBE9BBA775uL, // B' = pi/2 - B (high double)
            0x3C5EE2C2D963A10CuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFAD611FEEED2925uL, // c0 (high double)
            0xBC2C094544F37630uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB5D6E7244839E4uL, // c1 (low double)
            0xBF87ADABBEBF7F7FuL, // c2
            0x3F9766E378A69DCAuL, // c3
            0xBF5E68B8DEC46BCDuL, // c4
            0x3F626E6BF47FF36DuL, // c5
            0xBF3149F4C32A02B4uL, // c6
            0x3F4B764BFF26FBAFuL, // c7
        },
        // Breakpoint B = 115 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FC46B9C347764A4uL, // B' = pi/2 - B (high double)
            0xBC61A900F67F753AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFAB46008DFA625DuL, // c0 (high double)
            0x3C1CFC71261A7FB0uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB5C4FB50FA0748uL, // c1 (low double)
            0xBF85F569AE5EF4C9uL, // c2
            0x3F974FE41D60092FuL, // c3
            0xBF5C281BF4A7A8E8uL, // c4
            0x3F62476FB6177831uL, // c5
            0xBF2FF1DE9AA130D5uL, // c6
            0xBF3D841C7D3057BCuL, // c7
        },
        // Breakpoint B = 116 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FC2D97C7F3321D2uL, // B' = pi/2 - B (high double)
            0x3C5A79394C9E8A0AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFA92B4D8A427A5EuL, // c0 (high double)
            0xBC48D0F742399A2EuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB5B4689EEE44C8uL, // c1 (low double)
            0xBF843EC87B78BC83uL, // c2
            0x3F973AA7CA62B764uL, // c3
            0xBF59EC1D907A7E74uL, // c4
            0x3F62232304CE0BCBuL, // c5
            0xBF2D5C019E5C9327uL, // c6
            0x3F4A5742B30D6EA4uL, // c7
        },
        // Breakpoint B = 117 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FC1475CC9EEDF01uL, // B' = pi/2 - B (high double)
            0xBC63DDC5BCE200BBuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFA710FE6F2ADA79uL, // c0 (high double)
            0x3C47FCD1AF1BC88CuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB5A52DD3BF1903uL, // c1 (low double)
            0xBF8289A7118F8015uL, // c2
            0x3F97272B048B0CB6uL, // c3
            0xBF57B45DC9402B07uL, // c4
            0x3F620239A251E2D3uL, // c5
            0xBF2ACB87C59ED9B1uL, // c6
            0xBF314F7BD600B48CuL, // c7
        },
        // Breakpoint B = 118 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FBF6A7A2955385EuL, // B' = pi/2 - B (high double)
            0x3C560FAFBFD97309uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFA4F70ACF7EED25uL, // c0 (high double)
            0xBC36D383618823A2uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB59749CEE8C351uL, // c1 (low double)
            0xBF80D5E49AFDF8A4uL, // c2
            0x3F97156A9BCB4ECEuL, // c3
            0xBF55807DE55737F7uL, // c4
            0x3F61E4232793575AuL, // c5
            0xBF28440F1D68D5AEuL, // c6
            0xBF29FA204333F00CuL, // c7
        },
        // Breakpoint B = 119 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FBC463ABECCB2BBuL, // B' = pi/2 - B (high double)
            0x3C53DAEAF976E788uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFA2DD6A44CE1394uL, // c0 (high double)
            0xBC4BFF776DACC8B5uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB58ABB899A6815uL, // c1 (low double)
            0xBF7E46C0F6F92ADFuL, // c2
            0x3F970563A9ED840FuL, // c3
            0xBF53502047076AB7uL, // c4
            0x3F61C8E9B000D23FuL, // c5
            0xBF25C4A8BD148199uL, // c6
            0x3F4131449FFC5A0CuL, // c7
        },
        // Breakpoint B = 120 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FB921FB54442D18uL, // B' = pi/2 - B (high double)
            0x3C51A62633145C07uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBFA0C4146ECAB658uL, // c0 (high double)
            0x3C0F927420EB8C00uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB57F82168B23ECuL, // c1 (low double)
            0xBF7AE3F4956ED507uL, // c2
            0x3F96F71391CF9DFEuL, // c3
            0xBF5122E851208A87uL, // c4
            0x3F61B0E12D5FF408uL, // c5
            0xBF23485E9450F437uL, // c6
            0xBF3CE0F3BAD9A7CFuL, // c7
        },
        // Breakpoint B = 121 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FB5FDBBE9BBA775uL, // B' = pi/2 - B (high double)
            0x3C4EE2C2D963A10CuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBF9D5601E554ACE4uL, // c0 (high double)
            0x3BF1725D940FAE40uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB5759CA1D35995uL, // c1 (low double)
            0xBF7783239DE5AD22uL, // c2
            0x3F96EA77FED69EC2uL, // c3
            0xBF4DF0F48A9F5DD6uL, // c4
            0x3F619B9AD4B4F692uL, // c5
            0xBF20D3375495B3AFuL, // c6
            0xBF3C5F6BDA9649C1uL, // c7
        },
        // Breakpoint B = 122 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FB2D97C7F3321D2uL, // B' = pi/2 - B (high double)
            0x3C4A79394C9E8A0AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBF99244EF50D08B4uL, // c0 (high double)
            0xBBF5D43FBFA6A650uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB56D0A70CA2FAEuL, // c1 (low double)
            0xBF74240DEF7AA427uL, // c2
            0x3F96DF8EE40AD931uL, // c3
            0xBF49A0F663CD9CEDuL, // c4
            0x3F61892DA49A01B1uL, // c5
            0xBF1CC6A5F5CD8613uL, // c6
            0xBF2979EF025F5A5DuL, // c7
        },
        // Breakpoint B = 123 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FAF6A7A2955385EuL, // B' = pi/2 - B (high double)
            0x3C460FAFBFD97309uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBF94F2FF697D458FuL, // c0 (high double)
            0xBC2D47EE0D8639CCuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB565CAE1E73473uL, // c1 (low double)
            0xBF70C673B159CE1DuL, // c2
            0x3F96D6567B9EC32EuL, // c3
            0xBF455521A900DE57uL, // c4
            0x3F6179A180C97DC6uL, // c5
            0xBF17EE9A1A2093ECuL, // c6
            0xBF0F289C23A5BE3DuL, // c7
        },
        // Breakpoint B = 124 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FA921FB54442D18uL, // B' = pi/2 - B (high double)
            0x3C41A62633145C07uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBF90C202A6B4B69CuL, // c0 (high double)
            0x3C232719F8027342uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB55FDD6CA81EFBuL, // c1 (low double)
            0xBF6AD42A909B3D64uL, // c2
            0x3F96CECD466E9161uL, // c3
            0xBF410CC300A993C8uL, // c4
            0x3F616CF28C1A20ADuL, // c5
            0xBF131CA14234B209uL, // c6
            0xBF0575725B2F6BD2uL, // c7
        },
        // Breakpoint B = 125 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3FA2D97C7F3321D2uL, // B' = pi/2 - B (high double)
            0x3C3A79394C9E8A0AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBF8922902DB2AC11uL, // c0 (high double)
            0xBC23466BF961968DuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB55B41A17AA68CuL, // c1 (low double)
            0xBF641D6698BEFAD5uL, // c2
            0x3F96C8F20B94D936uL, // c3
            0xBF398E50110336DCuL, // c4
            0x3F61630EF83C40BEuL, // c5
            0xBF0CA4CD1D39A327uL, // c6
            0x3F2D88AA4542B2A9uL, // c7
        },
        // Breakpoint B = 126 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3F9921FB54442D18uL, // B' = pi/2 - B (high double)
            0x3C31A62633145C07uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBF80C17E51DDB68FuL, // c0 (high double)
            0xBC2BCDF2CE757F58uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB557F729AA69CBuL, // c1 (low double)
            0xBF5AD039FA1B06A9uL, // c2
            0x3F96C4C3D80DF7DBuL, // c3
            0xBF31073E4843262FuL, // c4
            0x3F615BF0FBDD75F6uL, // c5
            0xBF032668F6EE96B2uL, // c6
            0x3F4AB2BDCBD5B586uL, // c7
        },
        // Breakpoint B = 127 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x3F8921FB54442D18uL, // B' = pi/2 - B (high double)
            0x3C21A62633145C07uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0xBF70C15D3E7A6D22uL, // c0 (high double)
            0xBC031AD41335B3D2uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB555FDC752E0B2uL, // c1 (low double)
            0xBF4ACF3DF3195D30uL, // c2
            0x3F96C241FE6B7D5FuL, // c3
            0xBF2105DD68706FD9uL, // c4
            0x3F6157CCAEE0E30BuL, // c5
            0xBEF320E95C86AAB5uL, // c6
            0x3F36CC59FF7E57F3uL, // c7
        },
        // Breakpoint B = 128 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0x0000000000000000uL, // B' = pi/2 - B (high double)
            0x0000000000000000uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x0000000000000000uL, // c0 (high double)
            0x0000000000000000uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB5555555555555uL, // c1 (low double)
            0x0000000000000000uL, // c2
            0x3F96C16C16C4A87EuL, // c3
            0x0000000000000000uL, // c4
            0x3F615658C88FB22BuL, // c5
            0x0000000000000000uL, // c6
            0x3F45FEB9DABEDDC8uL, // c7
        },
        // Breakpoint B = 129 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBF8921FB54442D18uL, // B' = pi/2 - B (high double)
            0xBC21A62633145C07uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3F70C15D3E7A6D22uL, // c0 (high double)
            0x3C031AD4127D9F3EuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB555FDC752E0B2uL, // c1 (low double)
            0x3F4ACF3DF3195D30uL, // c2
            0x3F96C241FE6B7D5FuL, // c3
            0x3F2105DD68707011uL, // c4
            0x3F6157CCAEE0E445uL, // c5
            0x3EF320E95BE3603DuL, // c6
            0x3F36CC59FB2A5B52uL, // c7
        },
        // Breakpoint B = 130 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBF9921FB54442D18uL, // B' = pi/2 - B (high double)
            0xBC31A62633145C07uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3F80C17E51DDB68FuL, // c0 (high double)
            0x3C2BCDF2CE751153uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB557F729AA69CBuL, // c1 (low double)
            0x3F5AD039FA1B06A9uL, // c2
            0x3F96C4C3D80DF7DBuL, // c3
            0x3F31073E4843262EuL, // c4
            0x3F615BF0FBDD7613uL, // c5
            0x3F032668F6F21096uL, // c6
            0x3F4AB2BDCBA34188uL, // c7
        },
        // Breakpoint B = 131 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFA2D97C7F3321D2uL, // B' = pi/2 - B (high double)
            0xBC3A79394C9E8A0AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3F8922902DB2AC11uL, // c0 (high double)
            0x3C23466BF9645874uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB55B41A17AA68CuL, // c1 (low double)
            0x3F641D6698BEFAD5uL, // c2
            0x3F96C8F20B94D936uL, // c3
            0x3F398E50110336D4uL, // c4
            0x3F61630EF83C4025uL, // c5
            0x3F0CA4CD1D548C70uL, // c6
            0x3F2D88AA497F5A74uL, // c7
        },
        // Breakpoint B = 132 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFA921FB54442D18uL, // B' = pi/2 - B (high double)
            0xBC41A62633145C07uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3F90C202A6B4B69CuL, // c0 (high double)
            0xBC232719F82E8E78uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB55FDD6CA81EFBuL, // c1 (low double)
            0x3F6AD42A909B3D64uL, // c2
            0x3F96CECD466E9161uL, // c3
            0x3F410CC300A993FBuL, // c4
            0x3F616CF28C1A27D9uL, // c5
            0x3F131CA1418B03D0uL, // c6
            0xBF0575732696DF46uL, // c7
        },
        // Breakpoint B = 133 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFAF6A7A2955385EuL, // B' = pi/2 - B (high double)
            0xBC460FAFBFD97309uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3F94F2FF697D458FuL, // c0 (high double)
            0x3C2D47EE0D83F132uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB565CAE1E73473uL, // c1 (low double)
            0x3F70C673B159CE1DuL, // c2
            0x3F96D6567B9EC32EuL, // c3
            0x3F455521A900DE58uL, // c4
            0x3F6179A180C97E50uL, // c5
            0x3F17EE9A1A1D88C1uL, // c6
            0xBF0F289C329281E6uL, // c7
        },
        // Breakpoint B = 134 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFB2D97C7F3321D2uL, // B' = pi/2 - B (high double)
            0xBC4A79394C9E8A0AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3F99244EF50D08B4uL, // c0 (high double)
            0x3BF5D43FBF890B40uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB56D0A70CA2FAEuL, // c1 (low double)
            0x3F74240DEF7AA427uL, // c2
            0x3F96DF8EE40AD931uL, // c3
            0x3F49A0F663CD9CEEuL, // c4
            0x3F61892DA49A0241uL, // c5
            0x3F1CC6A5F5CAD66BuL, // c6
            0xBF2979EF063C00F3uL, // c7
        },
        // Breakpoint B = 135 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFB5FDBBE9BBA775uL, // B' = pi/2 - B (high double)
            0xBC4EE2C2D963A10CuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3F9D5601E554ACE4uL, // c0 (high double)
            0xBBF1725D944B6FF0uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB5759CA1D35995uL, // c1 (low double)
            0x3F7783239DE5AD22uL, // c2
            0x3F96EA77FED69EC2uL, // c3
            0x3F4DF0F48A9F5DD8uL, // c4
            0x3F619B9AD4B4F72FuL, // c5
            0x3F20D33754930861uL, // c6
            0xBF3C5F6BDCAA4F14uL, // c7
        },
        // Breakpoint B = 136 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFB921FB54442D18uL, // B' = pi/2 - B (high double)
            0xBC51A62633145C07uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FA0C4146ECAB658uL, // c0 (high double)
            0xBC0F9274210C3E30uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB57F82168B23ECuL, // c1 (low double)
            0x3F7AE3F4956ED507uL, // c2
            0x3F96F71391CF9DFEuL, // c3
            0x3F5122E851208A88uL, // c4
            0x3F61B0E12D5FF46BuL, // c5
            0x3F23485E944E645FuL, // c6
            0xBF3CE0F3BC206D6AuL, // c7
        },
        // Breakpoint B = 137 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFBC463ABECCB2BBuL, // B' = pi/2 - B (high double)
            0xBC53DAEAF976E788uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FA2DD6A44CE1394uL, // c0 (high double)
            0x3C4BFF776DAE2ACCuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB58ABB899A6815uL, // c1 (low double)
            0x3F7E46C0F6F92ADFuL, // c2
            0x3F970563A9ED840FuL, // c3
            0x3F53502047076AB6uL, // c4
            0x3F61C8E9B000D225uL, // c5
            0x3F25C4A8BD17699AuL, // c6
            0x3F413144A024F67FuL, // c7
        },
        // Breakpoint B = 138 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFBF6A7A2955385EuL, // B' = pi/2 - B (high double)
            0xBC560FAFBFD97309uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FA4F70ACF7EED25uL, // c0 (high double)
            0x3C36D38361835224uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB59749CEE8C351uL, // c1 (low double)
            0x3F80D5E49AFDF8A4uL, // c2
            0x3F97156A9BCB4ECEuL, // c3
            0x3F55807DE55737F9uL, // c4
            0x3F61E42327935766uL, // c5
            0x3F28440F1D62D6E1uL, // c6
            0xBF29FA20436AFB73uL, // c7
        },
        // Breakpoint B = 139 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFC1475CC9EEDF01uL, // B' = pi/2 - B (high double)
            0x3C63DDC5BCE200BBuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FA710FE6F2ADA79uL, // c0 (high double)
            0xBC47FCD1AF1E5F81uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB5A52DD3BF1903uL, // c1 (low double)
            0x3F8289A7118F8015uL, // c2
            0x3F97272B048B0CB6uL, // c3
            0x3F57B45DC9402B09uL, // c4
            0x3F620239A251E2C5uL, // c5
            0x3F2ACB87C598F86BuL, // c6
            0xBF314F7BD5BDC85CuL, // c7
        },
        // Breakpoint B = 140 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFC2D97C7F3321D2uL, // B' = pi/2 - B (high double)
            0xBC5A79394C9E8A0AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FA92B4D8A427A5EuL, // c0 (high double)
            0x3C48D0F7423D245AuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB5B4689EEE44C8uL, // c1 (low double)
            0x3F843EC87B78BC83uL, // c2
            0x3F973AA7CA62B764uL, // c3
            0x3F59EC1D907A7E71uL, // c4
            0x3F62232304CE0BF5uL, // c5
            0x3F2D5C019E659382uL, // c6
            0x3F4A5742B2B8EFA5uL, // c7
        },
        // Breakpoint B = 141 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFC46B9C347764A4uL, // B' = pi/2 - B (high double)
            0x3C61A900F67F753AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FAB46008DFA625DuL, // c0 (high double)
            0xBC1CFC712619CB40uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB5C4FB50FA0748uL, // c1 (low double)
            0x3F85F569AE5EF4C9uL, // c2
            0x3F974FE41D60092FuL, // c3
            0x3F5C281BF4A7A8E8uL, // c4
            0x3F62476FB617789EuL, // c5
            0x3F2FF1DE9AA127B2uL, // c6
            0xBF3D841C7EAC9D16uL, // c7
        },
        // Breakpoint B = 142 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFC5FDBBE9BBA775uL, // B' = pi/2 - B (high double)
            0xBC5EE2C2D963A10CuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FAD611FEEED2925uL, // c0 (high double)
            0x3C2C094544F32A78uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB5D6E7244839E4uL, // c1 (low double)
            0x3F87ADABBEBF7F7FuL, // c2
            0x3F9766E378A69DCAuL, // c3
            0x3F5E68B8DEC46BCDuL, // c4
            0x3F626E6BF47FF315uL, // c5
            0x3F3149F4C32A0685uL, // c6
            0x3F4B764BFFC07D27uL, // c7
        },
        // Breakpoint B = 143 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFC78FDB9EFFEA47uL, // B' = pi/2 - B (high double)
            0x3C5EE8786039D373uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FAF7CB429BEA72CuL, // c0 (high double)
            0x3C3FFC35F277313AuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB5EA2D6D5456CAuL, // c1 (low double)
            0x3F8967B0057BF87EuL, // c2
            0x3F977FA9A2ED4D75uL, // c3
            0x3F60572AC7A74374uL, // c4
            0x3F6298E78AA42A95uL, // c5
            0x3F329E8594822508uL, // c6
            0xBF37DBAE822657B7uL, // c7
        },
        // Breakpoint B = 144 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFC921FB54442D18uL, // B' = pi/2 - B (high double)
            0xBC61A62633145C07uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FB0CC62E1E04467uL, // c0 (high double)
            0x3C47805D7AE2BD94uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB5FECF9AE763EAuL, // c1 (low double)
            0x3F8B23982579F8E5uL, // c2
            0x3F979A3AB01644F9uL, // c3
            0x3F617CAA57838224uL, // c4
            0x3F62C658DCA7F4F9uL, // c5
            0x3F33F9251280A70AuL, // c6
            0xBF2D5C4E19969DC6uL, // c7
        },
        // Breakpoint B = 145 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFCAB41B09886FEAuL, // B' = pi/2 - B (high double)
            0x3C5A7EEED374BC71uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FB1DAAEA5CC26DDuL, // c0 (high double)
            0xBC51577CBB7A50E2uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB614CF365450F4uL, // c1 (low double)
            0x3F8CE1861157F322uL, // c2
            0x3F97B69B01FB3B26uL, // c3
            0x3F62A50D3A3C528BuL, // c4
            0x3F62F6E9071D40ACuL, // c5
            0x3F3559C86E0B8558uL, // c6
            0x3F41AFC8FF3AAA17uL, // c7
        },
        // Breakpoint B = 146 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFCC463ABECCB2BBuL, // B' = pi/2 - B (high double)
            0xBC63DAEAF976E788uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FB2E941ACF368E6uL, // c0 (high double)
            0xBC28FF4CA630E010uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB62C2DE3B8D982uL, // c1 (low double)
            0x3F8EA19C1138E963uL, // c2
            0x3F97D4CF49C89DB5uL, // c3
            0x3F63D0865FBE1BACuL, // c4
            0x3F632AE80D59D97FuL, // c5
            0x3F36C01A5963B73CuL, // c6
            0x3F2F2207CB99A5D3uL, // c7
        },
        // Breakpoint B = 147 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFCDD85A7410F58DuL, // B' = pi/2 - B (high double)
            0x3C56156546AFA570uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FB3F82048F8BE00uL, // c0 (high double)
            0x3C4AFD1EBA1D1ABAuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB644ED6242FE19uL, // c1 (low double)
            0x3F9031FE64545C32uL, // c2
            0x3F97F4DC89754A8CuL, // c3
            0x3F64FF499B056453uL, // c4
            0x3F636226CB0F9266uL, // c5
            0x3F382D35D58677E8uL, // c6
            0x3F3E29EBBF1CE01EuL, // c7
        },
        // Breakpoint B = 148 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFCF6A7A2955385EuL, // B' = pi/2 - B (high double)
            0xBC660FAFBFD97309uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FB5074ED126983EuL, // c0 (high double)
            0xBC516E0EB4132866uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB65F0F8C7B2809uL, // c1 (low double)
            0x3F9114659E4D5E41uL, // c2
            0x3F9816C815084CDAuL, // c3
            0x3F66318BB353364DuL, // c4
            0x3F639CED9C49E72FuL, // c5
            0x3F39A0F6E4942078uL, // c6
            0xBF2EBBDA5B521AEEuL, // c7
        },
        // Breakpoint B = 149 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFD07E4CEF4CBD98uL, // B' = pi/2 - B (high double)
            0x3C51ABDBB9EA8E6EuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FB616D1A2C591A8uL, // c0 (high double)
            0xBC2EF832FD938898uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB67A9658930D84uL, // c1 (low double)
            0x3F91F8156CC25284uL, // c2
            0x3F983A9794563574uL, // c3
            0x3F67678272C8ECDDuL, // c4
            0x3F63DAE696E3A33FuL, // c5
            0x3F3B1CDBB342F07AuL, // c6
            0x3F49E00A3BCC8A23uL, // c7
        },
        // Breakpoint B = 150 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFD1475CC9EEDF01uL, // B' = pi/2 - B (high double)
            0x3C73DDC5BCE200BBuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FB726AD2173D50EuL, // c0 (high double)
            0x3C1FCEC90C8C4150uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB69783D8B96CDFuL, // c1 (low double)
            0x3F92DD1FBCCBA937uL, // c2
            0x3F986051045C3883uL, // c3
            0x3F68A164B9409BD2uL, // c4
            0x3F641CB77D330FDAuL, // c5
            0x3F3C9FE910D9C9B2uL, // c6
            0xBF302CCA991EA2A4uL, // c7
        },
        // Breakpoint B = 151 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFD2106CA4910069uL, // B' = pi/2 - B (high double)
            0xBC7CAF6B74B6A225uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FB836E5B77D959AuL, // c0 (high double)
            0x3C51BDCEFF692111uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB6B5DA3B72B738uL, // c1 (low double)
            0x3F93C396B58BD74AuL, // c2
            0x3F9887FAB9649830uL, // c3
            0x3F69DF6A8A7C820CuL, // c4
            0x3F6461D201176179uL, // c5
            0x3F3E2C1220D981B3uL, // c6
            0x3F49F3B20F57CFA8uL, // c7
        },
        // Breakpoint B = 152 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFD2D97C7F3321D2uL, // B' = pi/2 - B (high double)
            0xBC6A79394C9E8A0AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FB9477FD636A4D8uL, // c0 (high double)
            0xBC5063C8B22C4544uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB6D59BCBF6C48BuL, // c1 (low double)
            0x3F94AB8CBB6CBB16uL, // c2
            0x3F98B19B6070D56DuL, // c3
            0x3F6B21CD22DB24B2uL, // c4
            0x3F64AAE4575F07F6uL, // c5
            0x3F3FC08B3806F34CuL, // c6
            0xBF00E42769BBA67AuL, // c7
        },
        // Breakpoint B = 153 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFD3A28C59D5433BuL, // B' = pi/2 - B (high double)
            0x3C41B19140C0C0D5uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FBA587FF6553726uL, // c0 (high double)
            0xBC5559CFE8367078uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB6F6CAF293AC7FuL, // c1 (low double)
            0x3F959514736F007FuL, // c2
            0x3F98DD3A019E14EBuL, // c3
            0x3F6C68C706B3D0FBuL, // c4
            0x3F64F779DB6C50B2uL, // c5
            0x3F40AF7CD7497CB8uL, // c6
            0x3F4AD3B9828EFAF2uL, // c7
        },
        // Breakpoint B = 154 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFD46B9C347764A4uL, // B' = pi/2 - B (high double)
            0x3C71A900F67F753AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FBB69EA984DE6EFuL, // c0 (high double)
            0xBC573CB7437F11CAuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB7196A3515E12EuL, // c1 (low double)
            0x3F968040C68C8FE8uL, // c2
            0x3F990ADE01D185E6uL, // c3
            0x3F6DB49417B3E65CuL, // c4
            0x3F6548393E856C2FuL, // c5
            0x3F418353F7828212uL, // c6
            0xBF36489E043C6801uL, // c7
        },
        // Breakpoint B = 155 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFD534AC0F19860CuL, // B' = pi/2 - B (high double)
            0xBC7EE4303B192DA6uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FBC7BC444B10758uL, // c0 (high double)
            0xBC511540B2269343uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB73D7C3735AA80uL, // c1 (low double)
            0x3F976D24E52F1F79uL, // c2
            0x3F993A8F25560FF4uL, // c3
            0x3F6F0571A586F6F3uL, // c4
            0x3F659C8E8181F3A3uL, // c5
            0x3F425CCABE1034F4uL, // c6
            0x3F4D59CA0F6FD35DuL, // c7
        },
        // Breakpoint B = 156 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFD5FDBBE9BBA775uL, // B' = pi/2 - B (high double)
            0xBC6EE2C2D963A10CuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FBD8E118C89577FuL, // c0 (high double)
            0x3C3344F06C36B16CuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB76303BB0A22C5uL, // c1 (low double)
            0x3F985BD44ABC0416uL, // c2
            0x3F996C5591BA6DFAuL, // c3
            0x3F702DCF4265B5B3uL, // c4
            0x3F65F52766DBD111uL, // c5
            0x3F433B879649EF12uL, // c6
            0x3F47C67A938B39B9uL, // c7
        },
        // Breakpoint B = 157 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFD6C6CBC45DC8DEuL, // B' = pi/2 - B (high double)
            0x3BB6D61B58C99C43uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FBEA0D709BC27CAuL, // c0 (high double)
            0x3C48928D103E8AACuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB78A03A181D6ACuL, // c1 (low double)
            0x3F994C62C1366C88uL, // c2
            0x3F99A039D09A41ADuL, // c3
            0x3F70DBAD90C6529AuL, // c4
            0x3F66520A2F7F46B3uL, // c5
            0x3F441FF58F223B31uL, // c6
            0xBF357CCF19FA9D15uL, // c7
        },
        // Breakpoint B = 158 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFD78FDB9EFFEA47uL, // B' = pi/2 - B (high double)
            0x3C6EE8786039D373uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FBFB4195F6B0333uL, // c0 (high double)
            0x3C47DB89E2705528uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB7B27EEAE12CDBuL, // c1 (low double)
            0x3F9A3EE464F938D9uL, // c2
            0x3F99D644D21E7AE8uL, // c3
            0x3F718C74CAAA7BA1uL, // c4
            0x3F66B2F778B94D1CuL, // c5
            0x3F450AB96EF7371CuL, // c6
            0x3F1412EF7B7239D9uL, // c7
        },
        // Breakpoint B = 159 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFD858EB79A20BB0uL, // B' = pi/2 - B (high double)
            0x3C7EE70AFE8446D9uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC063EE9D2B777CuL, // c0 (high double)
            0x3C4E1787D8B79264uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB7DC78B746BB2FuL, // c1 (low double)
            0x3F9B336DA889AC7DuL, // c2
            0x3F9A0E7FEF8737D6uL, // c3
            0x3F724046DF328844uL, // c4
            0x3F67185639AB9983uL, // c5
            0x3F45FBD25ACB3A74uL, // c6
            0x3F0E765A126439F7uL, // c7
        },
        // Breakpoint B = 160 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFD921FB54442D18uL, // B' = pi/2 - B (high double)
            0xBC71A62633145C07uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC0EE13A8A2A9C6uL, // c0 (high double)
            0xBC5830A63DDC366CuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB807F44735B1A4uL, // c1 (low double)
            0x3F9C2A135884458AuL, // c2
            0x3F9A48F4EE376E9DuL, // c3
            0x3F72F746AAC73A47uL, // c4
            0x3F678215824701AFuL, // c5
            0x3F46F3C719FB66E6uL, // c6
            0x3F4BA88D5ED09DD1uL, // c7
        },
        // Breakpoint B = 161 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFD9EB0B2EE64E81uL, // B' = pi/2 - B (high double)
            0xBC419ABB2567F739uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC1787E32B35112uL, // c0 (high double)
            0xBC5D2D85EBE7CAE6uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB834F4FC268515uL, // c1 (low double)
            0x3F9D22EA9FA50E7DuL, // c2
            0x3F9A85AE02819BA3uL, // c3
            0x3F73B19803AB2B21uL, // c4
            0x3F67F0B8C640461DuL, // c5
            0x3F47F293167B409BuL, // c6
            0xBF198BF45EA16C3AuL, // c7
        },
        // Breakpoint B = 162 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFDAB41B09886FEAuL, // B' = pi/2 - B (high double)
            0x3C6A7EEED374BC71uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC20330A15F6568uL, // c0 (high double)
            0x3C61BB4F210EA8D1uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB8637E591E05FEuL, // c1 (low double)
            0x3F9E1E090AECCE81uL, // c2
            0x3F9AC4B5D3090946uL, // c3
            0x3F746F5FC5C316BAuL, // c4
            0x3F686400E51F53C2uL, // c5
            0x3F48F8EADC3DB9C7uL, // c6
            0xBF320AB3DF7578B7uL, // c7
        },
        // Breakpoint B = 163 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFDB7D2AE42A9153uL, // B' = pi/2 - B (high double)
            0x3C7CB2463821BB58uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC28E2D6047CBC8uL, // c0 (high double)
            0xBC0CEAB2537F4380uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB89394034B11CFuL, // c1 (low double)
            0x3F9F1B848DE48876uL, // c2
            0x3F9B06177BDD3821uL, // c3
            0x3F7530C3E0150AA0uL, // c4
            0x3F68DBF62FBA70A3uL, // c5
            0x3F4A0736C860B4FEuL, // c6
            0x3F50A3F1D969C867uL, // c7
        },
        // Breakpoint B = 164 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFDC463ABECCB2BBuL, // B' = pi/2 - B (high double)
            0xBC73DAEAF976E788uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC31976E0E280B3uL, // c0 (high double)
            0xBC667516EE4F2544uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB8C539C2AB0F89uL, // c1 (low double)
            0x3FA00DB9C3806462uL, // c2
            0x3F9B49DE91DB8E44uL, // c3
            0x3F75F5EB6258C838uL, // c4
            0x3F695952C6BEE632uL, // c5
            0x3F4B1D691B1E9E68uL, // c6
            0xBEFC71567DD0AC40uL, // c7
        },
        // Breakpoint B = 165 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFDD0F4A996ED424uL, // B' = pi/2 - B (high double)
            0xBC51A070AC3E29A0uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC3A50F9AB1CA91uL, // c0 (high double)
            0x3C50CA3B48D90906uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB8F87382B56ACBuL, // c1 (low double)
            0x3FA08EF662132882uL, // c2
            0x3F9B901726A3CD1CuL, // c3
            0x3F76BEFE8A4461A2uL, // c4
            0x3F69DBAE42D8A424uL, // c5
            0x3F4C3C52357BEFDCuL, // c6
            0x3F32456D9E4ECEE8uL, // c7
        },
        // Breakpoint B = 166 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFDDD85A7410F58DuL, // B' = pi/2 - B (high double)
            0x3C66156546AFA570uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC430FA0B7C7F15uL, // c0 (high double)
            0x3C63115D3C69C8ABuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB92D45530E4319uL, // c1 (low double)
            0x3FA11183C3A8DFF6uL, // c2
            0x3F9BD8CDCC073185uL, // c3
            0x3F778C26D2F52E9AuL, // c4
            0x3F6A638650FB67B4uL, // c5
            0x3F4D6420527BDA5CuL, // c6
            0xBF2DA5A647B98994uL, // c7
        },
        // Breakpoint B = 167 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFDEA16A4EB316F6uL, // B' = pi/2 - B (high double)
            0x3C7A7D8171BF2FD8uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC4BD38B7876716uL, // c0 (high double)
            0x3C5504069A91A838uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB963B368408780uL, // c1 (low double)
            0x3FA1956DC5B1F35AuL, // c2
            0x3F9C240F984C07DCuL, // c3
            0x3F785D8F0388F11CuL, // c4
            0x3F6AF0B073DF1CB2uL, // c5
            0x3F4E9580D70D3533uL, // c6
            0x3F4571315C6E8041uL, // c7
        },
        // Breakpoint B = 168 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFDF6A7A2955385EuL, // B' = pi/2 - B (high double)
            0xBC760FAFBFD97309uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC549CE29CFCCBEuL, // c0 (high double)
            0xBC69881846E0BF86uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB99BC21C7FB9B2uL, // c1 (low double)
            0x3FA21AC0848A46C5uL, // c2
            0x3F9C71EA2A16819DuL, // c3
            0x3F7933633F8F0072uL, // c4
            0x3F6B83D1C5509A03uL, // c5
            0x3F4FD088727B14B9uL, // c6
            0xBF2306C949D5376CuL, // c7
        },
        // Breakpoint B = 169 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE019C501FBACE4uL, // B' = pi/2 - B (high double)
            0x3C8CB18F8746F50CuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC5D6BCF447403EuL, // c0 (high double)
            0xBC6C6C662876B393uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FB9D575F071964AuL, // c1 (low double)
            0x3FA2A1885E06BEACuL, // c2
            0x3F9CC26BAD1B38B8uL, // c3
            0x3F7A0DD116DA5229uL, // c4
            0x3F6C1C921E809C74uL, // c5
            0x3F508B03CEF8B0C3uL, // c6
            0x3F4BCE57D088C848uL, // c7
        },
        // Breakpoint B = 170 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE07E4CEF4CBD98uL, // B' = pi/2 - B (high double)
            0x3C61ABDBB9EA8E6EuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC66407B0109FA6uL, // c0 (high double)
            0x3C55C3EC26D04666uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBA10D38BFFE319uL, // c1 (low double)
            0x3FA329D1F417DBD6uL, // c2
            0x3F9D15A2DE544C7CuL, // c3
            0x3F7AED07977D3236uL, // c4
            0x3F6CBBAD3C53FDE7uL, // c5
            0x3F513318B342B49DuL, // c6
            0x3F1AF93D0FFB8A50uL, // c7
        },
        // Breakpoint B = 171 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE0E2D4DC9DCE4CuL, // B' = pi/2 - B (high double)
            0xBC83DBA1AA51ADD5uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC6F1B0FDBE6EDEuL, // c0 (high double)
            0x3C614F63025E35DCuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBA4DDFBF32A727uL, // c1 (low double)
            0x3FA3B3AA2F826E47uL, // c2
            0x3F9D6B9F112A3CA9uL, // c3
            0x3F7BD1375F2B326EuL, // c4
            0x3F6D60EF002A5959uL, // c5
            0x3F51E0E0E3E641DEuL, // c6
            0x3F32BFA3B8923D31uL, // c7
        },
        // Breakpoint B = 172 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE1475CC9EEDF01uL, // B' = pi/2 - B (high double)
            0x3C83DDC5BCE200BBuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC77FBB85928D2DuL, // c0 (high double)
            0xBC6121F4E1AC8166uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBA8C9F83131391uL, // c1 (low double)
            0x3FA43F1E42AF6970uL, // c2
            0x3F9DC4703433CA75uL, // c3
            0x3F7CBA92AE7FE396uL, // c4
            0x3F6E0CB6D0475CCFuL, // c5
            0x3F529491FB8790D2uL, // c6
            0x3F37152AFD195A10uL, // c7
        },
        // Breakpoint B = 173 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE1ABE4B73FEFB5uL, // B' = pi/2 - B (high double)
            0xBC61A34B6FA942D3uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC80E29F7BF5633uL, // c0 (high double)
            0x3C67F8F06ACC7061uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBACD17FA976786uL, // c1 (low double)
            0x3FA4CC3BAC93F075uL, // c2
            0x3F9E2026D6967836uL, // c3
            0x3F7DA94D7C694B78uL, // c4
            0x3F6EBF4F14B98A92uL, // c5
            0x3F534E7211F873CEuL, // c6
            0xBF2290D13622948AuL, // c7
        },
        // Breakpoint B = 174 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE2106CA4910069uL, // B' = pi/2 - B (high double)
            0xBC8CAF6B74B6A225uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC89CFF0CAA3CB4uL, // c0 (high double)
            0xBC6A8D50D06FC283uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBB0F4E73981D4BuL, // c1 (low double)
            0x3FA55B103BB2BD6DuL, // c2
            0x3F9E7ED42D91EC8DuL, // c3
            0x3F7E9D9D8AAE7B14uL, // c4
            0x3F6F78B5039EFAD8uL, // c5
            0x3F540EDBA3A78643uL, // c6
            0x3F3AA2CB5E5BE1EDuL, // c7
        },
        // Breakpoint B = 175 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE274F491E2111EuL, // B' = pi/2 - B (high double)
            0x3C7613F7E4FA18D6uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC92C3D852FEDF2uL, // c0 (high double)
            0x3C20F5E7403E5E20uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBB534867CEB1CBuL, // c1 (low double)
            0x3FA5EBAA113813D4uL, // c2
            0x3F9EE08A1A202CC5uL, // c3
            0x3F7F97BA7BA1301AuL, // c4
            0x3F701CAD7F191DFAuL, // c5
            0x3F54D60D13221310uL, // c6
            0x3F45F5FE99894FC6uL, // c7
        },
        // Breakpoint B = 176 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE2D97C7F3321D2uL, // B' = pi/2 - B (high double)
            0xBC7A79394C9E8A0AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FC9BBE82AEA0D12uL, // c0 (high double)
            0xBC62D116CA77096BuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBB990B7DDE5BFCuL, // c1 (low double)
            0x3FA67E17A4317C6CuL, // c2
            0x3F9F455B2F170454uL, // c3
            0x3F804BEEF437A2B3uL, // c4
            0x3F7080CEADF2976DuL, // c5
            0x3F55A44E3A71E919uL, // c6
            0xBF039F946FDF9BBBuL, // c7
        },
        // Breakpoint B = 177 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE33E046C843287uL, // B' = pi/2 - B (high double)
            0x3C8A7CCAC0E4698BuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FCA4C01D0769648uL, // c0 (high double)
            0x3C4DDC96CAE4A06CuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBBE09D8A670D72uL, // c1 (low double)
            0x3FA71267C4E29753uL, // c2
            0x3F9FAD5AB78BB764uL, // c3
            0x3F80CF21BC414F68uL, // c4
            0x3F70E8AC1DCF4352uL, // c5
            0x3F567A14C053085FuL, // c6
            0x3F4F621D6DE176C4uL, // c7
        },
        // Breakpoint B = 178 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE3A28C59D5433BuL, // B' = pi/2 - B (high double)
            0x3C51B19140C0C0D5uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FCADC8D51C0FA5BuL, // c0 (high double)
            0x3C54B350B5D39570uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBC2A04912319A8uL, // c1 (low double)
            0x3FA7A8A9A038608EuL, // c2
            0x3FA00C4E5E972915uL, // c3
            0x3F8155947D4C5C83uL, // c4
            0x3F7154A7B149DA58uL, // c5
            0x3F5757960EDFC45CuL, // c6
            0x3F3F895706E3352BuL, // c7
        },
        // Breakpoint B = 179 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE40714472653EFuL, // B' = pi/2 - B (high double)
            0xBC86106670B43955uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FCB6D8D944D0371uL, // c0 (high double)
            0xBC6CDD9D6EA4BA41uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBC7546C60FE5C7uL, // c1 (low double)
            0x3FA840ECC35C53FDuL, // c2
            0x3FA0439B07C380F3uL, // c3
            0x3F81DF673F038AB5uL, // c4
            0x3F71C4B3C36B8A12uL, // c5
            0x3F583D44FA2FB203uL, // c6
            0x3F52EE3E6132A151uL, // c7
        },
        // Breakpoint B = 180 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE46B9C347764A4uL, // B' = pi/2 - B (high double)
            0x3C81A900F67F753AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FCBFF05878394D3uL, // c0 (high double)
            0x3C486912F813E2ECuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBCC26A8EA2077FuL, // c1 (low double)
            0x3FA8DB411F68EE69uL, // c2
            0x3FA07C9E25737218uL, // c3
            0x3F826CBB3713A292uL, // c4
            0x3F72392A32F26720uL, // c5
            0x3F592B76EEB6E2FFuL, // c6
            0x3F39BADB5780F9BEuL, // c7
        },
        // Breakpoint B = 181 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE4D02421C87558uL, // B' = pi/2 - B (high double)
            0xBC6A765E893370D7uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FCC90F8250157F7uL, // c0 (high double)
            0x3C657D8312BFEA17uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBD117683053E13uL, // c1 (low double)
            0x3FA977B70D411C29uL, // c2
            0x3FA0B762F022F359uL, // c3
            0x3F82FDB2D7324A8FuL, // c4
            0x3F72B2030A36AADFuL, // c5
            0x3F5A22A22953FC15uL, // c6
            0x3F473A2F09CE8F95uL, // c7
        },
        // Breakpoint B = 182 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE534AC0F19860CuL, // B' = pi/2 - B (high double)
            0xBC8EE4303B192DA6uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FCD236870E769B5uL, // c0 (high double)
            0x3C523CE78218C6AAuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBD62716F68B5FEuL, // c1 (low double)
            0x3FAA165F518C38ABuL, // c2
            0x3FA0F3F50E789484uL, // c3
            0x3F839271DC18DB75uL, // c4
            0x3F732F8BAC4C54A8uL, // c5
            0x3F5B23248F865B74uL, // c6
            0x3F36E307E2B0837CuL, // c7
        },
        // Breakpoint B = 183 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE59933FC6A96C1uL, // B' = pi/2 - B (high double)
            0x3C71AA6E583501D4uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FCDB6597A2E1B61uL, // c0 (high double)
            0xBC6CB71A947AACF6uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBDB56255580CA4uL, // c1 (low double)
            0x3FAAB74B20D856E7uL, // c2
            0x3FA1326099896038uL, // c3
            0x3F842B1D5CF84F74uL, // c4
            0x3F73B1CC2C4E32D2uL, // c5
            0x3F5C2D7D75B7FC1BuL, // c6
            0x3F53DB94CE2C2FD8uL, // c7
        },
        // Breakpoint B = 184 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE5FDBBE9BBA775uL, // B' = pi/2 - B (high double)
            0xBC7EE2C2D963A10CuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FCE49CE5AF9DC29uL, // c0 (high double)
            0xBC382336F1C6BD90uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBE0A506D218F3CuL, // c1 (low double)
            0x3FAB5A8C23E49FEAuL, // c2
            0x3FA172B22117D625uL, // c3
            0x3F84C7DBDBF304B9uL, // c4
            0x3F74391DF553E930uL, // c5
            0x3F5D4211231EF24AuL, // c6
            0x3F52AB1CCD8D3AA9uL, // c7
        },
        // Breakpoint B = 185 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE66243D70CB82AuL, // B' = pi/2 - B (high double)
            0x3C884805FA81DE0AuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FCEDDCA38F25FECuL, // c0 (high double)
            0xBC65AACB9E8B4DF8uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBE6143274A35BFuL, // c1 (low double)
            0x3FAC00347C15ABB2uL, // c2
            0x3FA1B4F6B03A62C8uL, // c3
            0x3F8568D557225239uL, // c4
            0x3F74C5A8346B0155uL, // c5
            0x3F5E61680C7E386AuL, // c6
            0x3F45A81E03899535uL, // c7
        },
        // Breakpoint B = 186 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE6C6CBC45DC8DEuL, // B' = pi/2 - B (high double)
            0x3BC6D61B58C99C43uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FCF7250459C1976uL, // c0 (high double)
            0x3C40B5B081F06BA0uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBEBA422E0FE0E5uL, // c1 (low double)
            0x3FACA856C815D042uL, // c2
            0x3FA1F93BD218DCE9uL, // c3
            0x3F860E335AB42933uL, // c4
            0x3F75578EEA5D18D8uL, // c5
            0x3F5F8BFBAD564E86uL, // c6
            0x3F434A80D1DA21D1uL, // c7
        },
        // Breakpoint B = 187 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE72B53B1AED992uL, // B' = pi/2 - B (high double)
            0xBC88452B3716C4D6uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FD003B1DF5A0F86uL, // c0 (high double)
            0x3C716CEB71FB716BuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBF155566FA6884uL, // c1 (low double)
            0x3FAD530628A37C7AuL, // c2
            0x3FA23F8F96E80A99uL, // c3
            0x3F86B82113BDC04EuL, // c4
            0x3F75EF0B9C9DF1C6uL, // c5
            0x3F60613421CB6A80uL, // c6
            0x3F48E48E906FE806uL, // c7
        },
        // Breakpoint B = 188 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE78FDB9EFFEA47uL, // B' = pi/2 - B (high double)
            0x3C7EE8786039D373uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FD04E83F7474877uL, // c0 (high double)
            0x3C73B117AC20AC65uL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBF7284F47C0E61uL, // c1 (low double)
            0x3FAE0056458FCED5uL, // c2
            0x3FA28800992AEEE6uL, // c3
            0x3F8766CB6439B11DuL, // c4
            0x3F768C54DBB36839uL, // c5
            0x3F610297B13942C6uL, // c6
            0x3F5536BD1C4BF79EuL, // c7
        },
        // Breakpoint B = 189 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE7F4638C50FAFBuL, // B' = pi/2 - B (high double)
            0xBC71A4B8D15ECF6DuL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FD099A0163BCD3FuL, // c0 (high double)
            0x3C7A7F11D963D72EuL, // c0 (low double)
            0x3FD0000000000000uL, // c1 (high single bit)
            0x3FBFD1D937A1F094uL, // c1 (low double)
            0x3FAEB05B52EFB4C0uL, // c2
            0x3FA2D29E032A57DBuL, // c3
            0x3F881A60F7D38723uL, // c4
            0x3F772FBBF1BFA28BuL, // c5
            0x3F61AA745B27BB01uL, // c6
            0x3F51CC19136F91C1uL, // c7
        },
        // Breakpoint B = 190 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE858EB79A20BB0uL, // B' = pi/2 - B (high double)
            0x3C8EE70AFE8446D9uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FD0E507EE8B93D7uL, // c0 (high double)
            0x3C679F09CDD5D5B2uL, // c0 (low double)
            0x3FE0000000000000uL, // c1 (high single bit)
            0xBFBFCCA52E2AE201uL, // c1 (low double)
            0x3FAF632A1681FD0BuL, // c2
            0x3FA31F7794BDF922uL, // c3
            0x3F88D31259E30525uL, // c4
            0x3F77D97B728544F5uL, // c5
            0x3F62591A3A2B699FuL, // c6
            0x3F3605B387A5D7A0uL, // c7
        },
        // Breakpoint B = 191 * pi/256, function tan(B + x) - 1/(pi/2 - (B + x))
        {
            0xBFE8BD7366F31C64uL, // B' = pi/2 - B (high double)
            0x3C6A81C996DFD5A5uL, // B' = pi/2 - B (low double)
            0x3FF0000000000000uL, // tau (1 for cot path)
            0x3FD130BD396BB4A9uL, // c0 (high double)
            0xBC657D8592EA09D4uL, // c0 (low double)
            0x3FE0000000000000uL, // c1 (high single bit)
            0xBFBF68ED595308D6uL, // c1 (low double)
            0x3FB00C6BF6A673E6uL, // c2
            0x3FA36E9DA9556AA6uL, // c3
            0x3F8991120C7D4BB9uL, // c4
            0x3F7889ABD6008215uL, // c5
            0x3F630EE404688883uL, // c6
            0x3F52CAD75A2201AFuL, // c7
        },
        // Breakpoint B = 192 * pi/256, function tan(B + x)
        {
            0xBFE921FB54442D18uL, // B' = pi/2 - B (high double)
            0xBC81A62633145C07uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFF0000000000000uL, // c0 (high double)
            0x3C0492B176DF0500uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0x0000000000000000uL, // c1 (low double)
            0xC00000000000010AuL, // c2
            0x4005555555555614uL, // c3
            0xC00AAAAAA890EF4DuL, // c4
            0x4011111110018C73uL, // c5
            0xC015B108FB920A0BuL, // c6
            0x401BA285D6669571uL, // c7
        },
        // Breakpoint B = 193 * pi/256, function tan(B + x)
        {
            0xBFE9868341953DCDuL, // B' = pi/2 - B (high double)
            0x3C861341341F5289uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFEF395D9F0E3C92uL, // c0 (high double)
            0x3C550BEC65726E28uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFA8873C53BE0540uL, // c1 (low double)
            0xBFFE79E5C65202DEuL, // c2
            0x400412FD76797C7DuL, // c3
            0xC008AAB30DC82D37uL, // c4
            0x400F1B6C68B4AEA4uL, // c5
            0xC013758CE9BD9EFAuL, // c6
            0x40186872A0685915uL, // c7
        },
        // Breakpoint B = 194 * pi/256, function tan(B + x)
        {
            0xBFE9EB0B2EE64E81uL, // B' = pi/2 - B (high double)
            0xBC519ABB2567F739uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFEE776EAFC91706uL, // c0 (high double)
            0xBC890664EDC53F33uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFB7F295AD658764uL, // c1 (low double)
            0xBFFD0AA1DB7F5E58uL, // c2
            0x4002E8A6F9693556uL, // c3
            0xC006D7CA16B298C6uL, // c4
            0x400C63DFCAA3FB6CuL, // c5
            0xC0117C3EE50D7742uL, // c6
            0x401599932C96B9B6uL, // c7
        },
        // Breakpoint B = 195 * pi/256, function tan(B + x)
        {
            0xBFEA4F931C375F35uL, // B' = pi/2 - B (high double)
            0xBC8A79EFFD795057uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFEDB9FA89953FCFuL, // c0 (high double)
            0xBC286C68F514A640uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFC18AC418849A6EuL, // c1 (low double)
            0xBFFBB083E02786D7uL, // c2
            0x4001D438F8A20402uL, // c3
            0xC0052D5D6FF7A4BDuL, // c4
            0x4009F262067BAA5AuL, // c5
            0xC00F793DA8608BEDuL, // c6
            0x401328BFC5F60A29uL, // c7
        },
        // Breakpoint B = 196 * pi/256, function tan(B + x)
        {
            0xBFEAB41B09886FEAuL, // B' = pi/2 - B (high double)
            0x3C7A7EEED374BC71uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFED00CBC7384D2EuL, // c0 (high double)
            0xBC78C4FC635B4060uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFC6DA3A87611EA6uL, // c1 (low double)
            0xBFFA6A00F50D3A6FuL, // c2
            0x4000D3D0A6ABBD46uL, // c3
            0xC003A760672280D0uL, // c4
            0x4007BF026C103550uL, // c5
            0xC00C5EBD23F4E2EDuL, // c6
            0x4011024DB786758BuL, // c7
        },
        // Breakpoint B = 197 * pi/256, function tan(B + x)
        {
            0xBFEB18A2F6D9809EuL, // B' = pi/2 - B (high double)
            0xBC760E425E23E66FuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFEC4BB009E77983uL, // c0 (high double)
            0x3C5346B6895BC358uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFCBEB63A6D04570uL, // c1 (low double)
            0xBFF935AFFA8DCC1DuL, // c2
            0x3FFFCB7657D15F0BuL, // c3
            0xC002423AFCE75F54uL, // c4
            0x4005C2CF61AD4B89uL, // c5
            0xC0099C675A298F5CuL, // c6
            0x400E49F38B657F96uL, // c7
        },
        // Breakpoint B = 198 * pi/256, function tan(B + x)
        {
            0xBFEB7D2AE42A9153uL, // B' = pi/2 - B (high double)
            0x3C8CB2463821BB58uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFEB9A77C18C1AF2uL, // c0 (high double)
            0xBC76EA194A3A10DEuL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFD060D88DF903DFuL, // c1 (low double)
            0xBFF812463C417D93uL, // c2
            0x3FFE10E093E6DE8DuL, // c3
            0xC000FABB50DD022EuL, // c4
            0x4003F7B45B053F77uL, // c5
            0xC007275FB67DBBDDuL, // c6
            0x400B02CB76CD2071uL, // c7
        },
        // Breakpoint B = 199 * pi/256, function tan(B + x)
        {
            0xBFEBE1B2D17BA207uL, // B' = pi/2 - B (high double)
            0x3C61AEB67D55A7A1uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFEAECF5F9BA35A6uL, // c0 (high double)
            0xBC45350833FF1F00uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFD2B02B2D565D55uL, // c1 (low double)
            0xBFF6FE947B408BACuL, // c2
            0x3FFC751B52A83938uL, // c3
            0xBFFF9C12181CFBBAuL, // c4
            0x4002585B4421C074uL, // c5
            0xC004F6412119DA02uL, // c6
            0x400814BACED70A9AuL, // c7
        },
        // Breakpoint B = 200 * pi/256, function tan(B + x)
        {
            0xBFEC463ABECCB2BBuL, // B' = pi/2 - B (high double)
            0xBC83DAEAF976E788uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFEA43002AE42850uL, // c0 (high double)
            0x3C7A7C0406DA50C8uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFD4E526E33345B8uL, // c1 (low double)
            0xBFF5F9844B626284uL, // c2
            0x3FFAF5A574C2F3BFuL, // c3
            0xBFFD7334FD198D13uL, // c4
            0x4000E01251879DC7uL, // c5
            0xC00300E6F5213CD0uL, // c6
            0x40058BA67D3A0565uL, // c7
        },
        // Breakpoint B = 201 * pi/256, function tan(B + x)
        {
            0xBFECAAC2AC1DC370uL, // B' = pi/2 - B (high double)
            0x3C83DE7C6DBCC708uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE99C6E0F634394uL, // c0 (high double)
            0xBC81A125C4B050B6uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFD7012EAAC8A987uL, // c1 (low double)
            0xBFF50215B95FD940uL, // c2
            0x3FF9903A2B193F49uL, // c3
            0xBFFB7656607D37C1uL, // c4
            0x3FFF156F461C0EDFuL, // c5
            0xC001403E48F27428uL, // c6
            0x40034ACD8D387876uL, // c7
        },
        // Breakpoint B = 202 * pi/256, function tan(B + x)
        {
            0xBFED0F4A996ED424uL, // B' = pi/2 - B (high double)
            0xBC61A070AC3E29A0uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE8F9197BF85EEBuL, // c0 (high double)
            0x3C77971D70636FACuL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFD9058D62BBB98DuL, // c1 (low double)
            0xBFF4175D31199B0CuL, // c2
            0x3FF842CA948070CEuL, // c3
            0xBFF9A1675E6F2233uL, // c4
            0x3FFCA94A75D1DF36uL, // c5
            0xBFFF5C3EE98CB792uL, // c6
            0x4001549BBD776205uL, // c7
        },
        // Breakpoint B = 203 * pi/256, function tan(B + x)
        {
            0xBFED73D286BFE4D8uL, // B' = pi/2 - B (high double)
            0xBC8CAEB4C3DBDBD8uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE858DE3B716571uL, // c0 (high double)
            0xBC8C6340D74E9430uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFDAF377B50F3DE8uL, // c1 (low double)
            0xBFF338819C6A585BuL, // c2
            0x3FF70B781BB3F9F8uL, // c3
            0xBFF7F0C4DC952BEEuL, // c4
            0x3FFA75444794B0BEuL, // c5
            0xBFFC8A58AEEA1540uL, // c6
            0x3FFF1509C3EC4766uL, // c7
        },
        // Breakpoint B = 204 * pi/256, function tan(B + x)
        {
            0xBFEDD85A7410F58DuL, // B' = pi/2 - B (high double)
            0x3C76156546AFA570uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE7BB99ED2990CFuL, // c0 (high double)
            0x3C8AABB12893C19FuL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFDCCC0DD257FE9FuL, // c1 (low double)
            0xBFF264BAB3E8C3FCuL, // c2
            0x3FF5E88F7CC1F7BCuL, // c3
            0xBFF6612B1AE228E1uL, // c4
            0x3FF873A64164BF01uL, // c5
            0xBFFA016987C44E3BuL, // c6
            0x3FFBFCA608E3ABD2uL, // c7
        },
        // Breakpoint B = 205 * pi/256, function tan(B + x)
        {
            0xBFEE3CE261620641uL, // B' = pi/2 - B (high double)
            0xBC7A77CBEAE8FD71uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE7212BE621BE6DuL, // c0 (high double)
            0xBC8AD14CAC5A8A36uL, // c0 (low double)
            0x4000000000000000uL, // c1 (high single bit)
            0xBFDE905D04D13F70uL, // c1 (low double)
            0xBFF19B4F7BD63206uL, // c2
            0x3FF4D8845C8EEC38uL, // c3
            0xBFF4EFAAD70924AAuL, // c4
            0x3FF69F610EDFA787uL, // c5
            0xBFF7B93C8525967BuL, // c6
            0x3FF928F491396159uL, // c7
        },
        // Breakpoint B = 206 * pi/256, function tan(B + x)
        {
            0xBFEEA16A4EB316F6uL, // B' = pi/2 - B (high double)
            0x3C8A7D8171BF2FD8uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE6897514751DB6uL, // c0 (high double)
            0xBC7432E085692DD0uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FDFBE9EE099229DuL, // c1 (low double)
            0xBFF0DB94E83152C9uL, // c2
            0x3FF3D9ED602549D9uL, // c3
            0xBFF3999FCB35B63CuL, // c4
            0x3FF4F3F4BB18D34DuL, // c5
            0xBFF5AA9FE2712A98uL, // c6
            0x3FF6B61F156D15A2uL, // c7
        },
        // Breakpoint B = 207 * pi/256, function tan(B + x)
        {
            0xBFEF05F23C0427AAuL, // B' = pi/2 - B (high double)
            0x3C51B746C796F33CuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE5F457E4F4812EuL, // c0 (high double)
            0xBC8F6EE06A43AD15uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FDE1FFA33ABCB2DuL, // c1 (low double)
            0xBFF024ECA3857FDFuL, // c2
            0x3FF2EB80B3E2C969uL, // c3
            0xBFF25CA858ECE33EuL, // c4
            0x3FF36D6159DB2BC3uL, // c5
            0xBFF3CF427530AFAEuL, // c6
            0x3FF4914D16BF30E1uL, // c7
        },
        // Breakpoint B = 208 * pi/256, function tan(B + x)
        {
            0xBFEF6A7A2955385EuL, // B' = pi/2 - B (high double)
            0xBC860FAFBFD97309uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE561B82AB7F990uL, // c0 (high double)
            0xBC87A5FA3E37C2A5uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FDC92D84170BCC6uL, // c1 (low double)
            0xBFEEED87E934644CuL, // c2
            0x3FF20C10F4D7D072uL, // c3
            0xBFF1369E37DA5741uL, // c4
            0x3FF20816F61AB3EEuL, // c5
            0xBFF221953D8712FAuL, // c6
            0x3FF29CA76CD56201uL, // c7
        },
        // Breakpoint B = 209 * pi/256, function tan(B + x)
        {
            0xBFEFCF0216A64913uL, // B' = pi/2 - B (high double)
            0x3C81A9B7A75A3B87uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE4D17B087B265DuL, // c0 (high double)
            0x3C788AAC4AE25B54uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FDB166A3723839CuL, // c1 (low double)
            0xBFEDA1257F3D5E87uL, // c2
            0x3FF13A8A70704F6BuL, // c3
            0xBFF025900612C493uL, // c4
            0x3FF0C0E778680DE8uL, // c5
            0xBFF09CB2B02DFCD5uL, // c6
            0x3FF0D681147681E0uL, // c7
        },
        // Breakpoint B = 210 * pi/256, function tan(B + x)
        {
            0xBFF019C501FBACE4uL, // B' = pi/2 - B (high double)
            0x3C9CB18F8746F50CuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE44386DB9CE5DBuL, // c0 (high double)
            0x3C8A7F79AD345EDEuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD9A9EE22E82AF2uL, // c1 (low double)
            0xBFEC63B53FACFD1BuL, // c2
            0x3FF075F0B010F1C9uL, // c3
            0xBFEE4F7738A37AC1uL, // c4
            0x3FEF29F7068A6E01uL, // c5
            0xBFEE78912F3E48A3uL, // c6
            0x3FEE7DE5AD1A6D54uL, // c7
        },
        // Breakpoint B = 211 * pi/256, function tan(B + x)
        {
            0xBFF04C08F8A4353EuL, // B' = pi/2 - B (high double)
            0x3C908E433AE0CC54uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE3B7C3289ED6F3uL, // c0 (high double)
            0x3C8C4CF3D3282C13uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD84CAE0A06DE7CuL, // c1 (low double)
            0xBFEB344C2E4C0F47uL, // c2
            0x3FEF7AB88F5F820CuL, // c3
            0xBFEC771219A99E98uL, // c4
            0x3FED0390B144EACAuL, // c5
            0xBFEBF90A32A17514uL, // c6
            0x3FEBAC50A2919B8CuL, // c7
        },
        // Breakpoint B = 212 * pi/256, function tan(B + x)
        {
            0xBFF07E4CEF4CBD98uL, // B' = pi/2 - B (high double)
            0x3C71ABDBB9EA8E6EuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE32E1889047FFDuL, // c0 (high double)
            0x3C810D08F013759CuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD6FDFF12A4297BuL, // c1 (low double)
            0xBFEA120FA25B9D83uL, // c2
            0x3FEE1FF1BF27E6ABuL, // c3
            0xBFEABF0C6E045EA0uL, // c4
            0x3FEB0A0E2CDCAB88uL, // c5
            0xBFE9B40D41660E9DuL, // c6
            0x3FE92FCC1848D94BuL, // c7
        },
        // Breakpoint B = 213 * pi/256, function tan(B + x)
        {
            0xBFF0B090E5F545F2uL, // B' = pi/2 - B (high double)
            0xBC7EE15577AE1472uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE2A6709A74F289uL, // c0 (high double)
            0xBC76E3B98C266662uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD5BD40BF249496uL, // c1 (low double)
            0xBFE8FC33EFCF6F3AuL, // c2
            0x3FECDA06E4CFC822uL, // c3
            0xBFE924C5BA1C7515uL, // c4
            0x3FE9395A86CEDBCBuL, // c5
            0xBFE7A39B6931C7D4uL, // c6
            0x3FE6D875BED7A2F5uL, // c7
        },
        // Breakpoint B = 214 * pi/256, function tan(B + x)
        {
            0xBFF0E2D4DC9DCE4CuL, // B' = pi/2 - B (high double)
            0xBC93DBA1AA51ADD5uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE220B5EF047825uL, // c0 (high double)
            0xBC5AE1F6C712B338uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD489DC399ADDECuL, // c1 (low double)
            0xBFE7F1FB2FEBE8C6uL, // c2
            0x3FEBA791734D6CCEuL, // c3
            0xBFE7A5DA56BAB9DAuL, // c4
            0x3FE78DC9A17C17D7uL, // c5
            0xBFE5C262725357AEuL, // c6
            0x3FE4D1FE58052A5FuL, // c7
        },
        // Breakpoint B = 215 * pi/256, function tan(B + x)
        {
            0xBFF11518D34656A6uL, // B' = pi/2 - B (high double)
            0xBC9FFEEDF6B7D68DuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE19CD3FE8E405DuL, // c0 (high double)
            0x3C84E5C1D4831B21uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD36343ADCAF866uL, // c1 (low double)
            0xBFE6F2B4260FF6A8uL, // c2
            0x3FEA8747864942E1uL, // c3
            0xBFE6401D47362936uL, // c4
            0x3FE6040F50A52FBEuL, // c5
            0xBFE40BA6192CC675uL, // c6
            0x3FE302E7DABA2D8EuL, // c7
        },
        // Breakpoint B = 216 * pi/256, function tan(B + x)
        {
            0xBFF1475CC9EEDF01uL, // B' = pi/2 - B (high double)
            0x3C93DDC5BCE200BBuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE11AB7190834ECuL, // c0 (high double)
            0x3C8E577A07DB8B3EuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD248F1B07217E3uL, // c1 (low double)
            0xBFE5FDB93DD01D06uL, // c2
            0x3FE977F956462152uL, // c3
            0xBFE4F192BE91E40AuL, // c4
            0x3FE49932EEB2A51CuL, // c5
            0xBFE27B305435DE25uL, // c6
            0x3FE176C2BDD3411EuL, // c7
        },
        // Breakpoint B = 217 * pi/256, function tan(B + x)
        {
            0xBFF179A0C097675BuL, // B' = pi/2 - B (high double)
            0x3C7EE9E5C1EF600DuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE09A4C59BD0D4DuL, // c0 (high double)
            0x3C451D94A7B87B10uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD13A68B2A5E855uL, // c1 (low double)
            0xBFE5126F9FD8D89FuL, // c2
            0x3FE8788EEDC53F19uL, // c3
            0xBFE3B86B3E883B76uL, // c4
            0x3FE34A881E7F09CAuL, // c5
            0xBFE10D3EF07D1073uL, // c6
            0x3FDFEBFE6A9FE5DDuL, // c7
        },
        // Breakpoint B = 218 * pi/256, function tan(B + x)
        {
            0xBFF1ABE4B73FEFB5uL, // B' = pi/2 - B (high double)
            0xBC71A34B6FA942D3uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFE01B819B5A7CF7uL, // c0 (high double)
            0xBC80BED996FB9ED6uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FD03732803B2A4CuL, // c1 (low double)
            0xBFE430465B55C6C0uL, // c2
            0x3FE78806185ACF72uL, // c3
            0xBFE292FF3DF61A34uL, // c4
            0x3FE215A449967BC2uL, // c5
            0xBFDF7CEF2946D3BAuL, // c6
            0x3FDD0F4FF3CD4465uL, // c7
        },
        // Breakpoint B = 219 * pi/256, function tan(B + x)
        {
            0xBFF1DE28ADE8780FuL, // B' = pi/2 - B (high double)
            0xBC908C1F2850796DuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFDF3C8AD985D9EEuL, // c0 (high double)
            0xBC455C31F533FF68uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FCE7DBF907D8618uL, // c1 (low double)
            0xBFE356B5A1D9CA5AuL, // c2
            0x3FE6A57085697F99uL, // c3
            0xBFE17FCB46BE280BuL, // c4
            0x3FE0F858275BAA48uL, // c5
            0xBFDD17B47CA8A84BuL, // c6
            0x3FDA9B369A707A6BuL, // c7
        },
        // Breakpoint B = 220 * pi/256, function tan(B + x)
        {
            0xBFF2106CA4910069uL, // B' = pi/2 - B (high double)
            0xBC9CAF6B74B6A225uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFDE450E0D273E7AuL, // c0 (high double)
            0xBC74138BBC4225C5uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FCCA20F5D4879EAuL, // c1 (low double)
            0xBFE2853E13FBB7C7uL, // c2
            0x3FE5CFF21905E477uL, // c3
            0xBFE07D6C7D0DAC66uL, // c4
            0x3FDFE1550F550217uL, // c5
            0xBFDAE570CE7F6BE5uL, // c6
            0x3FD89328A3AB7677uL, // c7
        },
        // Breakpoint B = 221 * pi/256, function tan(B + x)
        {
            0xBFF242B09B3988C4uL, // B' = pi/2 - B (high double)
            0x3C972D483EE33523uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFDD506C82A2C800uL, // c0 (high double)
            0xBC6E169413DFE956uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FCADA8ECCC42C48uL, // c1 (low double)
            0xBFE1BB681D0BE8A7uL, // c2
            0x3FE506BF662527D3uL, // c3
            0xBFDF153B07CB2306uL, // c4
            0x3FDDF9A636394DCAuL, // c5
            0xBFD8E153A4A7867FuL, // c6
            0x3FD6937B37976C23uL, // c7
        },
        // Breakpoint B = 222 * pi/256, function tan(B + x)
        {
            0xBFF274F491E2111EuL, // B' = pi/2 - B (high double)
            0x3C8613F7E4FA18D6uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFDC5E87185E67B6uL, // c0 (high double)
            0xBC4FEBA1F148ADA8uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FC92683A68EB5D0uL, // c1 (low double)
            0xBFE0F8C35C7247C7uL, // c2
            0x3FE4491C4DCA262EuL, // c3
            0xBFDD4C67683395D6uL, // c4
            0x3FDC3667C15EBF92uL, // c5
            0xBFD7070B7FE443D3uL, // c6
            0x3FD4805841790B6EuL, // c7
        },
        // Breakpoint B = 223 * pi/256, function tan(B + x)
        {
            0xBFF2A738888A9978uL, // B' = pi/2 - B (high double)
            0xBC5195059E91C4D2uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFDB6F3FC8C61E5BuL, // c0 (high double)
            0x3C6A1CC48268287AuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FC7853E65CE5524uL, // c1 (low double)
            0xBFE03CE61B6A9E63uL, // c2
            0x3FE3965ABF5C8FABuL, // c3
            0xBFDB9E3938F261E9uL, // c4
            0x3FDA94A771ED6FE5uL, // c5
            0xBFD552B7D4636646uL, // c6
            0x3FD2F58F20A0FA49uL, // c7
        },
        // Breakpoint B = 224 * pi/256, function tan(B + x)
        {
            0xBFF2D97C7F3321D2uL, // B' = pi/2 - B (high double)
            0xBC8A79394C9E8A0AuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFDA827999FCEF32uL, // c0 (high double)
            0xBC708A11019E41CEuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FC5F619980C4336uL, // c1 (low double)
            0xBFDF0ED99BED9B65uL, // c2
            0x3FE2EDD996C1BBEBuL, // c3
            0xBFDA08B785C6EA59uL, // c4
            0x3FD911BC7B12902FuL, // c5
            0xBFD3C0DA5B1130C3uL, // c6
            0x3FD1B590151D6334uL, // c7
        },
        // Breakpoint B = 225 * pi/256, function tan(B + x)
        {
            0xBFF30BC075DBAA2CuL, // B' = pi/2 - B (high double)
            0xBC995FE8F2B56DBDuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFD998188E816BF0uL, // c0 (high double)
            0xBC73DB2BD13E0A0CuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FC478794806DE66uL, // c1 (low double)
            0xBFDDAFF33BF96274uL, // c2
            0x3FE24F039543C659uL, // c3
            0xBFD88A113E071D2DuL, // c4
            0x3FD7AB3E661E65F9uL, // c5
            0xBFD24E4E1B8DC6CDuL, // c6
            0x3FD01ED5FCAC7FDDuL, // c7
        },
        // Breakpoint B = 226 * pi/256, function tan(B + x)
        {
            0xBFF33E046C843287uL, // B' = pi/2 - B (high double)
            0x3C9A7CCAC0E4698BuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFD8B00196B3D022uL, // c0 (high double)
            0x3C67341EE30455A6uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FC30BCA73878656uL, // c1 (low double)
            0xBFDC5C67FF403B73uL, // c2
            0x3FE1B94E72981059uL, // c3
            0xBFD7209990BAE1F6uL, // c4
            0x3FD65EF72AC3A4F8uL, // c5
            0xBFD0F83D8377C34BuL, // c6
            0x3FCD97555AF6144CuL, // c7
        },
        // Breakpoint B = 227 * pi/256, function tan(B + x)
        {
            0xBFF37048632CBAE1uL, // B' = pi/2 - B (high double)
            0x3C8CB2FCE8FC81A5uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFD7CA1A832D0F84uL, // c0 (high double)
            0x3C7EE5A712091434uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FC1AF828B5ED674uL, // c1 (low double)
            0xBFDB13909C0269A2uL, // c2
            0x3FE12C3A0394B7FAuL, // c3
            0xBFD5CAC4A421216CuL, // c4
            0x3FD52AE625505252uL, // c5
            0xBFCF7831018A616AuL, // c6
            0x3FCB98AB42F62940uL, // c7
        },
        // Breakpoint B = 228 * pi/256, function tan(B + x)
        {
            0xBFF3A28C59D5433BuL, // B' = pi/2 - B (high double)
            0x3C61B19140C0C0D5uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFD6E649F7D78649uL, // c0 (high double)
            0xBC678D2AC0322984uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FC0631EFCBE0D16uL, // c1 (low double)
            0xBFD9D4CEDF38DFCDuL, // c2
            0x3FE0A74F7488C3A6uL, // c3
            0xBFD487249EA3BD6CuL, // c4
            0x3FD40D3A9BDEFA27uL, // c5
            0xBFCD2F1CBD1B58CDuL, // c6
            0x3FC98CBDCEBDAD19uL, // c7
        },
        // Breakpoint B = 229 * pi/256, function tan(B + x)
        {
            0xBFF3D4D0507DCB95uL, // B' = pi/2 - B (high double)
            0xBC83DA34489C213BuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFD604775FBB27DFuL, // c0 (high double)
            0xBC6AC647C9CF0A4CuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FBE4C49866D2D78uL, // c1 (low double)
            0xBFD89F8D033946D5uL, // c2
            0x3FE02A20954EE23AuL, // c3
            0xBFD35466F88C5ABCuL, // c4
            0x3FD3044BAC5DC8C6uL, // c5
            0xBFCB110C3D6FEB70uL, // c6
            0x3FC7761A8DC12688uL, // c7
        },
        // Breakpoint B = 230 * pi/256, function tan(B + x)
        {
            0xBFF40714472653EFuL, // B' = pi/2 - B (high double)
            0xBC96106670B43955uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFD5248AE1701B17uL, // c0 (high double)
            0xBC7BE8CC944DAFEAuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FBBF04005723CCCuL, // c1 (low double)
            0xBFD7733D1300258AuL, // c2
            0x3FDF688E6ACC7DF5uL, // c3
            0xBFD231520EA793C8uL, // c4
            0x3FD20E93C5183A34uL, // c5
            0xBFC91A331D55B3F7uL, // c6
            0x3FC651721E1371DDuL, // c7
        },
        // Breakpoint B = 231 * pi/256, function tan(B + x)
        {
            0xBFF439583DCEDC4AuL, // B' = pi/2 - B (high double)
            0x3C9DCC4D42E59DF2uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFD4466D542BAC92uL, // c0 (high double)
            0x3C705073A04E444AuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FB9B14751F8BA20uL, // c1 (low double)
            0xBFD64F58590170A2uL, // c2
            0x3FDE8AC91D15CFCFuL, // c3
            0xBFD11CC2EF09C664uL, // c4
            0x3FD12AB5AB65B965uL, // c5
            0xBFC7471C9AC81567uL, // c6
            0x3FC4826BA97C3E69uL, // c7
        },
        // Breakpoint B = 232 * pi/256, function tan(B + x)
        {
            0xBFF46B9C347764A4uL, // B' = pi/2 - B (high double)
            0x3C91A900F67F753AuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFD36A08355C63DCuL, // c0 (high double)
            0x3C6D7145B6A10D86uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FB78E922B82AB64uL, // c1 (low double)
            0xBFD5335ED87AB0D5uL, // c2
            0x3FDDBA417913D89CuL, // c3
            0xBFD015AB59BD876BuL, // c4
            0x3FD0576D1EF5DB85uL, // c5
            0xBFC594A6DD2CA17DuL, // c6
            0x3FC2F51FBC1DE653uL, // c7
        },
        // Breakpoint B = 233 * pi/256, function tan(B + x)
        {
            0xBFF49DE02B1FECFEuL, // B' = pi/2 - B (high double)
            0x3C7616D2A8653209uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFD28F459ECAD74DuL, // c0 (high double)
            0xBC302C86D50E46C0uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FB5875F92EDBDF4uL, // c1 (low double)
            0xBFD41ED6D05A0010uL, // c2
            0x3FDCF65481FC5BEBuL, // c3
            0xBFCE361FE028368DuL, // c4
            0x3FCF2728B941710BuL, // c5
            0xBFC3FFF674960878uL, // c6
            0x3FC1A12109F12125uL, // c7
        },
        // Breakpoint B = 234 * pi/256, function tan(B + x)
        {
            0xBFF4D02421C87558uL, // B' = pi/2 - B (high double)
            0xBC7A765E893370D7uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFD1B6103D3597E9uL, // c0 (high double)
            0xBC6CBCB5FED7AC2CuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FB39AFA3392E164uL, // c1 (low double)
            0xBFD3114C46E2AEF3uL, // c2
            0x3FDC3E6B23EF73B5uL, // c3
            0xBFCC580D1B42EE7FuL, // c4
            0x3FCDBC3D6E14D202uL, // c5
            0xBFC28671B66B7EC4uL, // c6
            0x3FC0BFBCFE62E275uL, // c7
        },
        // Breakpoint B = 235 * pi/256, function tan(B + x)
        {
            0xBFF502681870FDB2uL, // B' = pi/2 - B (high double)
            0xBC92C0E3EEB304EEuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFD0DE53475F3B3CuL, // c0 (high double)
            0xBC67B2CC0A839654uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FB1C8B7D76C7EE0uL, // c1 (low double)
            0xBFD20A509D4BB6C1uL, // c2
            0x3FDB91F96510DF38uL, // c3
            0xBFCA8F698D62832EuL, // c4
            0x3FCC6C3408291325uL, // c5
            0xBFC125B6D8F69817uL, // c6
            0x3FBE80E8F3B35090uL, // c7
        },
        // Breakpoint B = 236 * pi/256, function tan(B + x)
        {
            0xBFF534AC0F19860CuL, // B' = pi/2 - B (high double)
            0xBC9EE4303B192DA6uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFD007FA758626AEuL, // c0 (high double)
            0xBC65A5EB9CA4F97AuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FB00FF8E583BEBCuL, // c1 (low double)
            0xBFD1097A2AA570F5uL, // c2
            0x3FDAF07DA8CAEBA7uL, // c3
            0xBFC8DA9D107525ADuL, // c4
            0x3FCB3545F2ADC1DBuL, // c5
            0xBFBFB739DB46589AuL, // c6
            0x3FBDA2C2374E7733uL, // c7
        },
        // Breakpoint B = 237 * pi/256, function tan(B + x)
        {
            0xBFF566F005C20E67uL, // B' = pi/2 - B (high double)
            0x3C94F8837880A9A2uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFCE65E3F27C9F2AuL, // c0 (high double)
            0xBC6F62A8B8B3F2A2uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FACE04FD3EC1C90uL, // c1 (low double)
            0xBFD00E63DD639A45uL, // c2
            0x3FDA598003751442uL, // c3
            0xBFC73829AC409F93uL, // c4
            0x3FCA15E3A26F9D69uL, // c5
            0xBFBD4C4574E0CB3EuL, // c6
            0x3FBB9C588182E205uL, // c7
        },
        // Breakpoint B = 238 * pi/256, function tan(B + x)
        {
            0xBFF59933FC6A96C1uL, // B' = pi/2 - B (high double)
            0x3C81AA6E583501D4uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFCCBE4CEB4B4CF2uL, // c0 (high double)
            0xBC64A514F790D588uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FA9D1724DE6C940uL, // c1 (low double)
            0xBFCE3159C5ED3A4EuL, // c2
            0x3FD9CC919D4271B2uL, // c3
            0xBFC5A6A9489AA7F4uL, // c4
            0x3FC90C961E2B49F9uL, // c5
            0xBFBB06EE7CFA369EuL, // c6
            0x3FB9E9447AAF378DuL, // c7
        },
        // Breakpoint B = 239 * pi/256, function tan(B + x)
        {
            0xBFF5CB77F3131F1BuL, // B' = pi/2 - B (high double)
            0xBC6A70A9025D3E70uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFCB1909EFD8B762uL, // c0 (high double)
            0xBC5843049E1318BCuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FA6F2545C478E00uL, // c1 (low double)
            0xBFCC4FF0A9D74F70uL, // c2
            0x3FD9494C22EF2A54uL, // c3
            0xBFC424CB909A6722uL, // c4
            0x3FC8180A481D593BuL, // c5
            0xBFB8E3D90C36FA86uL, // c6
            0x3FB8D27EAFEAC4D3uL, // c7
        },
        // Breakpoint B = 240 * pi/256, function tan(B + x)
        {
            0xBFF5FDBBE9BBA775uL, // B' = pi/2 - B (high double)
            0xBC8EE2C2D963A10CuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFC975F5E0553158uL, // c0 (high double)
            0xBC2EEEA31AB9D050uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FA4420303DEC550uL, // c1 (low double)
            0xBFCA77D9D62CA610uL, // c2
            0x3FD8CF5143323608uL, // c3
            0xBFC2B15401455507uL, // c4
            0x3FC7370FF096556FuL, // c5
            0xBFB6DFEACA56EADAuL, // c6
            0x3FB6B75DF89EC13CuL, // c7
        },
        // Breakpoint B = 241 * pi/256, function tan(B + x)
        {
            0xBFF62FFFE0642FCFuL, // B' = pi/2 - B (high double)
            0xBC9B94ADB917F93EuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFC7D4EC54FB5968uL, // c0 (high double)
            0xBC4FB65648F8A4A0uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3FA1BF9C7C60B940uL, // c1 (low double)
            0xBFC8A86964A1796BuL, // c2
            0x3FD85E4A37F23352uL, // c3
            0xBFC14B181F5BF6AEuL, // c4
            0x3FC66888E3A5D3BAuL, // c5
            0xBFB4F84D7AC0F5DBuL, // c6
            0x3FB62BA6F07C5587uL, // c7
        },
        // Breakpoint B = 242 * pi/256, function tan(B + x)
        {
            0xBFF66243D70CB82AuL, // B' = pi/2 - B (high double)
            0x3C984805FA81DE0AuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFC635C990CE0D36uL, // c0 (high double)
            0x3C59DFC0C48C2B20uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F9ED49F1F8144C0uL, // c1 (low double)
            0xBFC6E0F98ED00C32uL, // c2
            0x3FD7F5E75A2DD66EuL, // c3
            0xBFBFE1FBA10F6B0EuL, // c4
            0x3FC5AB7CD1905EBBuL, // c5
            0xBFB32A57A4FAE3FFuL, // c6
            0x3FB4BA60BA62CA68uL, // c7
        },
        // Breakpoint B = 243 * pi/256, function tan(B + x)
        {
            0xBFF69487CDB54084uL, // B' = pi/2 - B (high double)
            0x3C8849735C376AA4uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFC4986A74CF4E57uL, // c0 (high double)
            0xBC69AB0A243914D0uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F9A82B60D75F5C0uL, // c1 (low double)
            0xBFC520EA2BA6A0AEuL, // c2
            0x3FD795DFBFFEE39EuL, // c3
            0xBFBD43F3A7862CC7uL, // c4
            0x3FC4FF04247D7006uL, // c5
            0xBFB1739726786F64uL, // c6
            0x3FB2FAC67536C2B7uL, // c7
        },
        // Breakpoint B = 244 * pi/256, function tan(B + x)
        {
            0xBFF6C6CBC45DC8DEuL, // B' = pi/2 - B (high double)
            0x3BD6D61B58C99C43uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFC2FCAC73A60640uL, // c0 (high double)
            0xBC52CF95F70D438CuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F96881A43B33590uL, // c1 (low double)
            0xBFC367A033B52F5EuL, // c2
            0x3FD73DF0E3CC557AuL, // c3
            0xBFBABA1CAC4F18CDuL, // c4
            0x3FC4624C36BE941DuL, // c5
            0xBFAFA38B2F88747DuL, // c6
            0x3FB215B1F402C9E1uL, // c7
        },
        // Breakpoint B = 245 * pi/256, function tan(B + x)
        {
            0xBFF6F90FBB065138uL, // B' = pi/2 - B (high double)
            0xBC8843BDD561383DuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFC1626D85A91E70uL, // c0 (high double)
            0x3C50084844291A00uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F92E3863C9A69E0uL, // c1 (low double)
            0xBFC1B4854BB2B147uL, // c2
            0x3FD6EDDE5416DFFBuL, // c3
            0xBFB8429341CCA17AuL, // c4
            0x3FC3D49AC2C9F46EuL, // c5
            0xBFAC85843914CF63uL, // c6
            0x3FB1B1DC3FA41043uL, // c7
        },
        // Breakpoint B = 246 * pi/256, function tan(B + x)
        {
            0xBFF72B53B1AED992uL, // B' = pi/2 - B (high double)
            0xBC98452B3716C4D6uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFBF93183A8DB9E9uL, // c0 (high double)
            0x3C4124AC452DA622uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F8F27A3181EE680uL, // c1 (low double)
            0xBFC00707549D7ECCuL, // c2
            0x3FD6A5716B500730uL, // c3
            0xBFB5DB89BB6B245FuL, // c4
            0x3FC3554867F59647uL, // c5
            0xBFA98919BF6D1AC0uL, // c6
            0x3FB1AF9A18244AC2uL, // c7
        },
        // Breakpoint B = 247 * pi/256, function tan(B + x)
        {
            0xBFF75D97A85761EDuL, // B' = pi/2 - B (high double)
            0x3C9B97887C831272uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFBC63CE377FC802uL, // c0 (high double)
            0x3C543F7D42D9FFC4uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F892FE02A47EFA0uL, // c1 (low double)
            0xBFBCBD3001C956C5uL, // c2
            0x3FD664790F293C47uL, // c3
            0xBFB38345DE7C4B0BuL, // c4
            0x3FC2E3C4C02FFDF9uL, // c5
            0xBFA6AA7F0C135018uL, // c6
            0x3FAFE3C754ACF687uL, // c7
        },
        // Breakpoint B = 248 * pi/256, function tan(B + x)
        {
            0xBFF78FDB9EFFEA47uL, // B' = pi/2 - B (high double)
            0x3C8EE8786039D373uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFB936BB8C5B2DA2uL, // c0 (high double)
            0xBC587D23392D360CuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F83DDE2A8935DE0uL, // c1 (low double)
            0xBFB97558DC230BA8uL, // c2
            0x3FD62AC976FBC1B8uL, // c3
            0xBFB1381EB60159F6uL, // c4
            0x3FC27F84438D5027uL, // c5
            0xBFA3E63B3A462D11uL, // c6
            0x3FAF0AE614B60705uL, // c7
        },
        // Breakpoint B = 249 * pi/256, function tan(B + x)
        {
            0xBFF7C21F95A872A1uL, // B' = pi/2 - B (high double)
            0x3C6A877F1DB6080CuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFB60B9F7597FDECuL, // c0 (high double)
            0xBC3D1E3FBF349CC0uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F7E5FFEF4D8DFC0uL, // c1 (low double)
            0xBFB6357984E124B6uL, // c2
            0x3FD5F83BF8B56900uL, // c3
            0xBFADF0F50CAA06C2uL, // c4
            0x3FC228147690425DuL, // c5
            0xBFA138F0F640FF12uL, // c6
            0x3FAF8D1BCDD067BDuL, // c7
        },
        // Breakpoint B = 250 * pi/256, function tan(B + x)
        {
            0xBFF7F4638C50FAFBuL, // B' = pi/2 - B (high double)
            0xBC81A4B8D15ECF6DuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFB2E239CCFF3831uL, // c0 (high double)
            0xBC5879ED3FF4D06CuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F764980AEEE3740uL, // c1 (low double)
            0xBFB2FC879C81123CuL, // c2
            0x3FD5CCAEDC1DC96EuL, // c3
            0xBFA98599B71D86B1uL, // c4
            0x3FC1DD150D55E5CEuL, // c5
            0xBF9D3EB9450DBD20uL, // c6
            0x3FADD6CFBE9A9DDEuL, // c7
        },
        // Breakpoint B = 251 * pi/256, function tan(B + x)
        {
            0xBFF826A782F98355uL, // B' = pi/2 - B (high double)
            0xBC94F5A8B515906FuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFAF7495EA3F3783uL, // c0 (high double)
            0x3C407F10D50F8AC9uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F6EEB8B377B1A00uL, // c1 (low double)
            0xBFAF92FABFD0B7B3uL, // c2
            0x3FD5A805341437FFuL, // c3
            0xBFA52B295E2DCF92uL, // c4
            0x3FC19E2C59B8C033uL, // c5
            0xBF982CF88466F013uL, // c6
            0x3FAD27509D6C3D7CuL, // c7
        },
        // Breakpoint B = 252 * pi/256, function tan(B + x)
        {
            0xBFF858EB79A20BB0uL, // B' = pi/2 - B (high double)
            0x3C9EE70AFE8446D9uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFA927278A3B1162uL, // c0 (high double)
            0xBBFB01BD7F1D4620uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F63C55DB08E1680uL, // c1 (low double)
            0xBFA936B1EC652161uL, // c2
            0x3FD58A26BD6CCCCDuL, // c3
            0xBFA0DEB5445B4095uL, // c4
            0x3FC16B146127E0E0uL, // c5
            0xBF9336A9836851D5uL, // c6
            0x3FAC19302F7F3ACCuL, // c7
        },
        // Breakpoint B = 253 * pi/256, function tan(B + x)
        {
            0xBFF88B2F704A940AuL, // B' = pi/2 - B (high double)
            0x3C92C3BEB21E1E21uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBFA2DBAAE9A05DB0uL, // c0 (high double)
            0x3C4371F88E195029uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F563A086B860B00uL, // c1 (low double)
            0xBFA2E23785EEB488uL, // c2
            0x3FD572FFC353B741uL, // c3
            0xBF993ABD2276BF58uL, // c4
            0x3FC143926088643DuL, // c5
            0xBF8CAC359DDE7BCEuL, // c6
            0x3FAB56F9A2A6CD28uL, // c7
        },
        // Breakpoint B = 254 * pi/256, function tan(B + x)
        {
            0xBFF8BD7366F31C64uL, // B' = pi/2 - B (high double)
            0x3C7A81C996DFD5A5uL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBF992346247A91F0uL, // c0 (high double)
            0x3BF126D920C63160uL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F43BF447B091000uL, // c1 (low double)
            0xBF992726F0F98D7FuL, // c2
            0x3FD5628108EFA55CuL, // c3
            0xBF90C8A60639C114uL, // c4
            0x3FC127789EC0BD38uL, // c5
            0xBF830BA6F46C58D9uL, // c6
            0x3FABBEE2C1E62B03uL, // c7
        },
        // Breakpoint B = 255 * pi/256, function tan(B + x)
        {
            0xBFF8EFB75D9BA4BEuL, // B' = pi/2 - B (high double)
            0xBC760B679AB8CD3BuL, // B' = pi/2 - B (low double)
            0x0000000000000000uL, // tau (1 for cot path)
            0xBF89224E047E368EuL, // c0 (high double)
            0x3C1A268EB3BB9B7AuL, // c0 (low double)
            0x3FF0000000000000uL, // c1 (high single bit)
            0x3F23BDBEAD8CF800uL, // c1 (low double)
            0xBF8923461AE96D62uL, // c2
            0x3FD5589FB822D5EEuL, // c3
            0xBF80C326DCDCFCD2uL, // c4
            0x3FC116A999125B8AuL, // c5
            0xBF7300CC26580304uL, // c6
            0x3FAB7B6F1DCF22D1uL, // c7
        },
    },
    0x40F000000E4DB24CuL, /* dReductionRangeVal */
    /* Distincted */
    /*
    //
    //      This value guarantees that N*PIoHi fits in double precision
    mantissa,
    //      here N = [x*PIu].
    //
    //      N * mantissa(PioHi) < 2^(53) + 1 ==> N*(0x1921FB544>>2) < 2^53 + 1
    //      ==> N < (2^53+1) / 0x6487ED51 ==> N < 5340353.71 ==> N <= 5340353
    //      Then, obviously 5340354 is also OK
    //      (e.g. if rounding x*PIu actually rounded up)
    //      since it has even fewer mantissa bits.
    //      But we never want to cross the 5340354:
    //      [x * PIu] <= 5340353 ==> x*PIu < 5340354 ==> x <= 40F000000E4DB24C.
    //      Self check:
    //      convert to double and multiply without rounding:
    //          40F000000E4DB24C * 40545F306DC9C883 < 5340354,
    //      but 40F000000E4DB24D * 40545F306DC9C883 > 5340354.
    //
    */
}; /*dTan_Table*/
static __constant _iml_v2_dp_union_t __dtan_ha_Tab[2] = {
    0x00000000, 0x00000000, /* ZERO = 0.0 */
    0x00000000, 0x7FF00000, /* INF = 0x7ff00000 00000000 */
};
__attribute__((always_inline)) inline int
__ocl_svml_internal_dtan_ha(double *a, double *r) {
  int nRet = 0;
  double x, absx;
  /* Get absolute value of argument */
  absx = (*a);
  (((_iml_v2_dp_union_t *)&absx)->dwords.hi_dword =
       (((_iml_v2_dp_union_t *)&absx)->dwords.hi_dword & 0x7FFFFFFF) |
       ((_iml_uint32_t)(0) << 31));
  if (!((((((_iml_v2_dp_union_t *)&(*a))->dwords.hi_dword >> 20) & 0x7FF) !=
         0x7FF))) /********* |(*a)| = Inf,NaN *************/
  {
    if ((((_iml_v2_dp_union_t *)&(absx))->hex[0] ==
         ((__constant _iml_v2_dp_union_t *)&(
              ((__constant double *)__dtan_ha_Tab)[1]))
             ->hex[0]) &&
        (((_iml_v2_dp_union_t *)&(absx))->hex[1] ==
         ((__constant _iml_v2_dp_union_t *)&(
              ((__constant double *)__dtan_ha_Tab)[1]))
             ->hex[1])) /****** |(*a)|=Inf (path 1) ***********/
    {
      /* Return NaN with invalid exception */
      (*r) = (*a) * ((__constant double *)__dtan_ha_Tab)[0];
      /* Set ERRDOM error code */
      nRet = 1;
    }    /* |(*a)| = Inf */
    else /**************************** |(*a)|=NaN (path 2) ***********/
    {
      /* Just return NaN and set necessary flags */
      (*r) = (*a) * (*a);
    } /* |(*a)| = NaN */
  }
  return nRet;
}
double __ocl_svml_tan_ha(double x) {
  double r;
  unsigned int vm;
  double va1;
  double vr1;
  va1 = x;
  {
    double dAbsMask;
    double dAbsX;
    double dReductionRangeVal;
    double dReductionRangeMask;
    unsigned int mReductionRangeMask;
    double dX;
    double dZ;
    double dRangeVal;
    double dRangeMask;
    unsigned long lRangeMask;
    unsigned int iRound;
    unsigned long lRound;
    unsigned int iSign;
    unsigned long lSign;
    double dInvPI;
    double dRShifter;
    double dY;
    double dYLarge;
    double dN;
    double dPI1;
    double dPI2;
    double dPI3;
    double dR;
    double dRLarge;
    double dRh;
    double dRm;
    double dRl;
    double dRlLarge;
    double dRl3;
    double dNP2;
    double dNP3;
    unsigned int iIndex;
    unsigned long lIndex;
    double dTLh;
    double dTLl;
    double dTh;
    double dTl;
    double dPC[5];
    double dR2;
    double dPoly;
    double dP;
    double dPlow;
    double dOne;
    double dNumerator;
    double dNlow;
    double dDenominator;
    double dDlow;
    double dRcp;
    double dE;
    double dQuotient;
    double dQlow;
    double dAE;
    vm = 0;
    dAbsMask = as_double(__ocl_svml_internal_dtan_ha_data._dAbsMask);
    dAbsX = as_double((as_ulong(va1) & as_ulong(dAbsMask)));
    /* Large values check */
    dReductionRangeVal =
        as_double(__ocl_svml_internal_dtan_ha_data._dReductionRangeVal);
    {
      unsigned long lReductionRangeMask;
      dReductionRangeMask = as_double((unsigned long)((
          (!(dAbsX <= dReductionRangeVal)) ? 0xffffffffffffffff : 0x0)));
      lReductionRangeMask = as_ulong(dReductionRangeMask);
      mReductionRangeMask = 0;
      mReductionRangeMask = lReductionRangeMask;
    }
    {
      /* Main path */
      // start arg. reduction
      dInvPI = as_double(__ocl_svml_internal_dtan_ha_data._dInvPI_uisa);
      dRShifter = as_double(__ocl_svml_internal_dtan_ha_data._dRShift);
      // Y = x + BIG_VALUE
      dY = __spirv_ocl_fma(va1, dInvPI, dRShifter);
      // N = y - BIG_VALUE
      dN = (dY - dRShifter);
      dPI1 = as_double(__ocl_svml_internal_dtan_ha_data._dPI1_uisa);
      dPI2 = as_double(__ocl_svml_internal_dtan_ha_data._dPI2_ha_uisa);
      dPI3 = as_double(__ocl_svml_internal_dtan_ha_data._dPI3_ha_uisa);
      // Rh = x - N*Pi1
      dRh = __spirv_ocl_fma(-(dN), dPI1, va1);
      // Rm = Rh - N*Pi2
      dRm = __spirv_ocl_fma(-(dN), dPI2, dRh);
      // R = Rm - N*Pi3
      dR = __spirv_ocl_fma(-(dN), dPI3, dRm);
      // Calculate the lower part dRl
      dNP2 = (dRh - dRm);
      dNP3 = (dR - dRm);
      dRl = __spirv_ocl_fma(-(dN), dPI2, dNP2);
      dRl3 = __spirv_ocl_fma(dN, dPI3, dNP3);
      dRl = (dRl - dRl3);
    }
    if (__builtin_expect((mReductionRangeMask) != 0, 0)) {
      /* Large arguments path */
      // Hence set rangemask bit if corresponding exponent is above EMax
      // Check for Inf and NaNs
      dRangeVal = as_double(__ocl_svml_internal_dtan_ha_data._dRangeVal);
      dX = as_double((as_ulong(dRangeVal) & as_ulong(dAbsX)));
      dRangeMask = as_double(
          (unsigned long)((dX == dRangeVal) ? 0xffffffffffffffff : 0x0));
      lRangeMask = as_ulong(dRangeMask);
      vm = 0;
      vm = lRangeMask;
      /* file: _vdreduction_core.i */
      //
      // Vectorized huge argument reduction for trig functions
      // The scale factor is Pi/2^8
      // The input is argument for reduction _VARG_A(of type D)
      // The output is high _VRES_Z (of type D) + low _VRES_E (D), and the
      // integer part is _VRES_IND (L)
      //
      {
        double dExponent;
        unsigned long lInput;
        unsigned long lExponent;
        unsigned long lSignificand;
        unsigned long lIntegerBit;
        double dP_hi;
        double dP_med;
        double dP_lo;
        unsigned long lP_hi;
        unsigned long lP_med;
        unsigned long lP_lo;
        unsigned long lLowMask;
        unsigned long lP5;
        unsigned long lP4;
        unsigned long lP3;
        unsigned long lP2;
        unsigned long lP1;
        unsigned long lP0;
        unsigned long lM1;
        unsigned long lM0;
        unsigned long lM15;
        unsigned long lM14;
        unsigned long lM13;
        unsigned long lM12;
        unsigned long lM11;
        unsigned long lM10;
        unsigned long lM05;
        unsigned long lM04;
        unsigned long lM03;
        unsigned long lM02;
        unsigned long lM01;
        unsigned long lM00;
        unsigned long lN14;
        unsigned long lN13;
        unsigned long lN12;
        unsigned long lN11;
        unsigned long lP15;
        unsigned long lP14;
        unsigned long lP13;
        unsigned long lP12;
        unsigned long lP11;
        unsigned long lQ14;
        unsigned long lQ13;
        unsigned long lQ12;
        unsigned long lQ11;
        unsigned long lReducedHi;
        unsigned long lReducedMed;
        unsigned long lReducedLo;
        unsigned long lRoundBump;
        unsigned long lShiftedN;
        unsigned long lNMask;
        double dReducedHi;
        double dReducedMed;
        double dReducedLo;
        unsigned long lExponentPart;
        unsigned long lShiftedSig;
        double dShifter;
        double dIntegerPart;
        double dRHi;
        double dRLo;
        unsigned long lSignBit;
        double d2pi_full;
        double d2pi_lead;
        double d2pi_trail;
        double dLeadmask;
        double dRHi_lead;
        double dRHi_trail;
        double dPir1;
        double dPir2;
        double dPir3;
        double dPir4;
        double dPir12;
        double dPir34;
        double dRedPreHi;
        double dRedHi;
        double dRedLo;
        double dMinInput;
        double dAbs;
        double dMultiplex;
        double dNotMultiplex;
        double dMultiplexedInput;
        double dMultiplexedOutput;
        // Cast the input to an integer
        lInput = as_ulong(va1);
        // Grab exponent and convert it to a table offset
        lExponent = 0x7ff0000000000000uL;
        ;
        lExponent = (lExponent & lInput);
        lExponent = ((unsigned long)(lExponent) >> (52));
        // Get the (2^a / 2pi) mod 1 values from the table.
        // Because VLANG doesn't have L-type gather, we need a trivial cast
        dP_hi = as_double(
            ((unsigned long
                  *)(__ocl_svml_internal_dtan_ha_reduction_data
                         ._dPtable))[(((0 + lExponent) * (3 * 8)) >> (3)) + 0]);
        dP_med = as_double(
            ((unsigned long
                  *)(__ocl_svml_internal_dtan_ha_reduction_data
                         ._dPtable))[(((0 + lExponent) * (3 * 8)) >> (3)) + 1]);
        dP_lo = as_double(
            ((unsigned long
                  *)(__ocl_svml_internal_dtan_ha_reduction_data
                         ._dPtable))[(((0 + lExponent) * (3 * 8)) >> (3)) + 2]);
        lP_hi = as_ulong(dP_hi);
        lP_med = as_ulong(dP_med);
        lP_lo = as_ulong(dP_lo);
        // Also get the significand as an integer
        // NB: adding in the integer bit is wrong for denorms!
        // To make this work for denorms we should do something slightly
        // different
        lSignificand = 0x000fffffffffffffuL;
        ;
        lIntegerBit = 0x0010000000000000uL;
        lSignificand = (lSignificand & lInput);
        lSignificand = (lSignificand + lIntegerBit);
        // Break the P_xxx and m into 32-bit chunks ready for
        // the long multiplication via 32x32->64 multiplications
        lLowMask = 0x00000000FFFFFFFFuL;
        lP5 = ((unsigned long)(lP_hi) >> (32));
        lP4 = (lP_hi & lLowMask);
        lP3 = ((unsigned long)(lP_med) >> (32));
        lP2 = (lP_med & lLowMask);
        lP1 = ((unsigned long)(lP_lo) >> (32));
        lP0 = (lP_lo & lLowMask);
        lM1 = ((unsigned long)(lSignificand) >> (32));
        lM0 = (lSignificand & lLowMask);
        // Now do the big multiplication and carry propagation
        lM15 = (lM1 * lP5);
        lM14 = (lM1 * lP4);
        lM13 = (lM1 * lP3);
        lM12 = (lM1 * lP2);
        lM11 = (lM1 * lP1);
        lM10 = (lM1 * lP0);
        lM05 = (lM0 * lP5);
        lM04 = (lM0 * lP4);
        lM03 = (lM0 * lP3);
        lM02 = (lM0 * lP2);
        lM01 = (lM0 * lP1);
        lM00 = (lM0 * lP0);
        lN11 = ((unsigned long)(lM01) >> (32));
        lN12 = ((unsigned long)(lM02) >> (32));
        lN13 = ((unsigned long)(lM03) >> (32));
        lN14 = ((unsigned long)(lM04) >> (32));
        lN11 = (lM11 + lN11);
        lN12 = (lM12 + lN12);
        lN13 = (lM13 + lN13);
        lN14 = (lM14 + lN14);
        lP11 = (lM02 & lLowMask);
        lP12 = (lM03 & lLowMask);
        lP13 = (lM04 & lLowMask);
        lP14 = (lM05 & lLowMask);
        lP15 = ((unsigned long)(lM05) >> (32));
        lP11 = (lP11 + lN11);
        lP12 = (lP12 + lN12);
        lP13 = (lP13 + lN13);
        lP14 = (lP14 + lN14);
        lP15 = (lP15 + lM15);
        lQ11 = ((unsigned long)(lM10) >> (32));
        lQ11 = (lQ11 + lP11);
        lQ12 = ((unsigned long)(lQ11) >> (32));
        lQ12 = (lQ12 + lP12);
        lQ13 = ((unsigned long)(lQ12) >> (32));
        lQ13 = (lQ13 + lP13);
        lQ14 = ((unsigned long)(lQ13) >> (32));
        lQ14 = (lQ14 + lP14);
        // Assemble reduced argument from the pieces
        lQ11 = (lQ11 & lLowMask);
        lQ13 = (lQ13 & lLowMask);
        lReducedHi = ((unsigned long)(lQ14) << (32));
        lReducedLo = ((unsigned long)(lQ12) << (32));
        lReducedHi = (lReducedHi + lQ13);
        lReducedLo = (lReducedLo + lQ11);
        // We want to incorporate the original sign now too.
        // Do it here for convenience in getting the right N value,
        // though we could wait right to the end if we were prepared
        // to modify the sign of N later too.
        // So get the appropriate sign mask now (or sooner).
        lSignBit = 0x8000000000000000uL;
        ;
        lSignBit = (lSignBit & lInput);
        // Create floating-point high part, implicitly adding integer bit 1
        // Incorporate overall sign at this stage too.
        lExponentPart = 0x3FF0000000000000uL;
        lExponentPart = (lSignBit ^ lExponentPart);
        lShiftedSig = ((unsigned long)(lReducedHi) >> (12));
        lShiftedSig = (lShiftedSig | lExponentPart);
        dReducedHi = as_double(lShiftedSig);
        // Now round at the 2^-9 bit position for reduction mod pi/2^8
        // instead of the original 2pi (but still with the same 2pi scaling).
        // Use a shifter of 2^43 + 2^42.
        // The N we get is our final version; it has an offset of
        // 2^9 because of the implicit integer bit, and anyway for negative
        // starting value it's a 2s complement thing. But we need to mask
        // off the exponent part anyway so it's fine.
        dShifter = as_double(0x42A8000000000000uL);
        dIntegerPart = (dShifter + dReducedHi);
        dN = (dIntegerPart - dShifter);
        dReducedHi = (dReducedHi - dN);
        // Grab our final N value as an integer, appropriately masked mod 2^9
        lIndex = as_ulong(dIntegerPart);
        lNMask = 0x00000000000001FFuL;
        lIndex = (lIndex & lNMask);
        // Create floating-point low and medium parts, respectively
        //
        // lo_23, ... lo_0, 0, ..., 0
        // hi_11, ... hi_0, lo_63, ..., lo_24
        //
        // then subtract off the implicitly added integer bits,
        // 2^-104 and 2^-52, respectively.
        // Put the original sign into all of them at this stage.
        lExponentPart = 0x3970000000000000uL;
        lExponentPart = (lSignBit ^ lExponentPart);
        lShiftedSig = 0x0000000000FFFFFFuL;
        lShiftedSig = (lShiftedSig & lReducedLo);
        lShiftedSig = ((unsigned long)(lShiftedSig) << (28));
        lShiftedSig = (lShiftedSig | lExponentPart);
        dReducedLo = as_double(lShiftedSig);
        dShifter = as_double(lExponentPart);
        dReducedLo = (dReducedLo - dShifter);
        lExponentPart = 0x3CB0000000000000uL;
        lExponentPart = (lSignBit ^ lExponentPart);
        lShiftedSig = 0x0000000000000FFFuL;
        lShiftedSig = (lShiftedSig & lReducedHi);
        lShiftedSig = ((unsigned long)(lShiftedSig) << (40));
        lReducedLo = ((unsigned long)(lReducedLo) >> (24));
        lShiftedSig = (lShiftedSig | lReducedLo);
        lShiftedSig = (lShiftedSig | lExponentPart);
        dReducedMed = as_double(lShiftedSig);
        dShifter = as_double(lExponentPart);
        dReducedMed = (dReducedMed - dShifter);
        // Now add them up into 2 reasonably aligned pieces
        dRHi = (dReducedHi + dReducedMed);
        dReducedHi = (dReducedHi - dRHi);
        dReducedMed = (dReducedMed + dReducedHi);
        dRLo = (dReducedMed + dReducedLo);
        // Now multiply those numbers all by 2 pi, reasonably accurately.
        // (RHi + RLo) * (pi_lead + pi_trail) ~=
        // RHi * pi_lead + (RHi * pi_trail + RLo * pi_lead)
        d2pi_lead = as_double(0x401921fb54442d18uL);
        d2pi_trail = as_double(0x3cb1a62633145c07uL);
        dRedHi = (dRHi * d2pi_lead);
        dRedLo =
            __spirv_ocl_fma(dRHi, d2pi_lead, -(dRedHi));
        dRedLo =
            __spirv_ocl_fma(dRHi, d2pi_trail, dRedLo);
        dRedLo =
            __spirv_ocl_fma(dRLo, d2pi_lead, dRedLo);
        // If the magnitude of the input is <= 2^-20, then
        // just pass through the input, since no reduction will be needed and
        // the main path will only work accurately if the reduced argument is
        // about >= 2^-70 (which it is for all large pi multiples)
        dAbsMask = as_double(0x7FFFFFFFFFFFFFFFuL);
        dMinInput = as_double(0x3EB0000000000000uL);
        dAbs = as_double((as_ulong(va1) & as_ulong(dAbsMask)));
        dMultiplex = as_double(
            (unsigned long)((dAbs > dMinInput) ? 0xffffffffffffffff : 0x0));
        dNotMultiplex = as_double(
            (unsigned long)((dAbs <= dMinInput) ? 0xffffffffffffffff : 0x0));
        dMultiplexedInput =
            as_double((as_ulong(dNotMultiplex) & as_ulong(va1)));
        dMultiplexedOutput =
            as_double((as_ulong(dMultiplex) & as_ulong(dRedHi)));
        dZ = as_double(
            (as_ulong(dMultiplexedInput) | as_ulong(dMultiplexedOutput)));
        dE = as_double((as_ulong(dMultiplex) & as_ulong(dRedLo)));
        // The output is _VRES_Z (high) + _VRES_E (low), and the integer part is
        // _VRES_IND
      }

      iIndex = (((unsigned int)lIndex & (unsigned int)-1));
      iRound = 0x8u;
      lSign = as_ulong(dZ);
      iSign = ((unsigned int)((unsigned long)lSign >> 32));
      iSign = ((unsigned int)(iSign) >> (31));
      iRound = (iRound - iSign);
      iRound = (iIndex + iRound);
      iRound = ((unsigned int)(iRound) >> (4));
      lRound = ((unsigned long)(unsigned int)iRound);
      dYLarge = as_double(lRound);
      iRound = ((unsigned int)(iRound) << (4));
      iIndex = (iIndex - iRound);
      dN = ((double)((int)(iIndex)));
      dPI1 = as_double(0x3f8921fb54442d18uL);
      dPI2 = as_double(0x3c21a62633145c07uL);
      dRLarge = __spirv_ocl_fma(dN, dPI1, dZ);
      dRlLarge = __spirv_ocl_fma(-(dN), dPI1, dRLarge);
      dRlLarge = (dZ - dRlLarge);
      dRlLarge = __spirv_ocl_fma(dN, dPI2, dRlLarge);
      dRlLarge = (dRlLarge + dE);
      /* Merge results from main and large paths: */
      dY = as_double((((~as_ulong(dReductionRangeMask)) & as_ulong(dY)) |
                      (as_ulong(dReductionRangeMask) & as_ulong(dYLarge))));
      dR = as_double((((~as_ulong(dReductionRangeMask)) & as_ulong(dR)) |
                      (as_ulong(dReductionRangeMask) & as_ulong(dRLarge))));
      dRl = as_double((((~as_ulong(dReductionRangeMask)) & as_ulong(dRl)) |
                       (as_ulong(dReductionRangeMask) & as_ulong(dRlLarge))));
    };
    {
      unsigned long _t_emask;
      unsigned long _t_index;
      lIndex = as_ulong(dY);
      _t_emask = 0x0000000000000078uL;
      _t_index = ((unsigned long)(lIndex) << (3));
      lIndex = (_t_index & _t_emask);
    };
    dTh = as_double(
        ((unsigned long *)((double *)(&__ocl_svml_internal_dtan_ha_data
                                           .Th_tbl_uisa[0])))[lIndex >> 3]);
    dTl = as_double(
        ((unsigned long *)((double *)(&__ocl_svml_internal_dtan_ha_data
                                           .Tl_tbl_uisa[0])))[lIndex >> 3]);
    // R2 = R*R
    dR2 = (dR * dR);
    dPC[4] = as_double(__ocl_svml_internal_dtan_ha_data._dPC11_uisa);
    dPC[3] = as_double(__ocl_svml_internal_dtan_ha_data._dPC9_uisa);
    // Start polynomial computation
    dPoly = __spirv_ocl_fma(dPC[4], dR2, dPC[3]);
    dPC[2] = as_double(__ocl_svml_internal_dtan_ha_data._dPC7_uisa);
    dPoly = __spirv_ocl_fma(dPoly, dR2, dPC[2]);
    dPC[1] = as_double(__ocl_svml_internal_dtan_ha_data._dPC5_uisa);
    dPoly = __spirv_ocl_fma(dPoly, dR2, dPC[1]);
    dPC[0] = as_double(__ocl_svml_internal_dtan_ha_data._dPC3_uisa);
    dPoly = __spirv_ocl_fma(dPoly, dR2, dPC[0]);
    dPoly = (dPoly * dR);
    dPoly = __spirv_ocl_fma(-(dPoly), dR2, -(dRl));
    dP = (dR - dPoly);
    dPlow = (dR - dP);
    dPlow = (dPlow - dPoly);
    // Compute Numerator:
    // dNumerator + dNlow ~= dTh+dTl+dP+dPlow
    dNumerator = (dP + dTh);
    dNlow = (dNumerator - dTh);
    dNlow = (dP - dNlow);
    dNlow = (dNlow + dTl);
    dNlow = (dNlow + dPlow);
    // Computer Denominator:
    // dDenominator - dDlow ~= 1-(dTh+dTl)*(dP+dPlow)
    dOne = as_double(__ocl_svml_internal_dtan_ha_data._dOne_uisa);
    dDenominator = __spirv_ocl_fma(-(dTh), dP, dOne);
    dDlow = (dDenominator - dOne);
    dDlow = __spirv_ocl_fma(dTh, dP, dDlow);
    dDlow = __spirv_ocl_fma(dTh, dPlow, dDlow);
    dDlow = __spirv_ocl_fma(dTl, dP, dDlow);
    // Now computes (dNumerator + dNlow)/(dDenominator - dDlow)
    // Choose NR iteration instead of hardware division
    double dMant;
    double dExp;
    float sMant;
    float sRcp_Mant;
    double dRcp_Mant;
    double dShifter;
    unsigned long lExp;
    unsigned long lBias_d;
    double dRcp_Exp;
    // Gen denominator's mantissa
    dMant = as_double((as_ulong(dDenominator) & 0x800fffffffffffffuL) |
                      0x3ff0000000000000uL);
    // Gen denominator's exponent
    dExp = (double)((long)((as_ulong(dDenominator) >> 52) & 0x7ffu) - 0x3ff);
    // Compute mamtissa's reciprocal
    dRcp_Mant = ((double)(1.0f / ((float)(dMant))));
    dShifter = as_double(0x4138000000000000uL);
    // Shift exponent field right
    dExp = (dExp + dShifter);
    // Represent exponent as 64-bit integer
    lExp = as_ulong(dExp);
    lBias_d = 0x3ff0000000000000uL;
    // Shift exponent field left
    lExp = ((unsigned long)(lExp) << (20));
    // Subtract bias
    lExp = (lBias_d - lExp);
    // Represent exponent as double
    dRcp_Exp = as_double(lExp);
    // Multiply exponent by mantissa
    dRcp = (dRcp_Exp * dRcp_Mant);
    // One NR iteration to refine dRcp
    dE = __spirv_ocl_fma(-(dRcp), dDenominator, dOne);
    dE = __spirv_ocl_fma(dRcp, dDlow, dE);
    dRcp = __spirv_ocl_fma(dRcp, dE, dRcp);
    dQuotient = (dRcp * dNumerator);
    // One NR iteration to refine dQuotient
    dAE = __spirv_ocl_fma(dDenominator, dQuotient,
                                                 -(dNumerator));
    dAE = __spirv_ocl_fma(-(dDlow), dQuotient, dAE);
    dAE = (dAE - dNlow);
    vr1 = __spirv_ocl_fma(-(dAE), dRcp, dQuotient);
  }
  if (__builtin_expect((vm) != 0, 0)) {
    double __cout_a1;
    double __cout_r1;
    ((double *)&__cout_a1)[0] = va1;
    ((double *)&__cout_r1)[0] = vr1;
    __ocl_svml_internal_dtan_ha(&__cout_a1, &__cout_r1);
    vr1 = ((double *)&__cout_r1)[0];
  }
  r = vr1;
  return r;
}
