// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package slhdsa

import (
	"slices"
	"testing"
)

func TestChainKat(t *testing.T) {
	par := SLH_DSA_SHA2_128s
	x := []byte{0xe, 0x67, 0x56, 0xac, 0x2, 0x60, 0x33, 0x62, 0x35, 0x1, 0xc9, 0x18, 0x8d, 0x17, 0x38, 0x8f}
	i := uint32(9)
	s := uint32(6)
	pkSeed := []byte{0x65, 0x54, 0x50, 0x6b, 0x6a, 0x55, 0x89, 0xdd, 0x2, 0xea, 0x35, 0x56, 0x92, 0xa9, 0xe, 0xf2}
	adrs := &address{0x0, 0x0, 0x0, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x81, 0x0, 0x0, 0x0, 0x1d, 0x0, 0x0, 0x0, 0xe}
	want := []byte{0x1a, 0xdf, 0x4a, 0x24, 0x50, 0x32, 0xaf, 0x6a, 0xb3, 0x1f, 0xf8, 0x3e, 0xd, 0xdb, 0xd9, 0xcb}
	if got := par.chain(x, i, s, pkSeed, adrs); !slices.Equal(got, want) {
		t.Errorf("par.chain() = %v, want %v", got, want)
	}
}

func TestWotsPkGenKat(t *testing.T) {
	par := SLH_DSA_SHA2_128s
	skSeed := []byte{0x9b, 0xea, 0xfd, 0x4b, 0xd1, 0x6c, 0xd9, 0x12, 0x4b, 0xc2, 0xa9, 0x3d, 0x3e, 0x69, 0xa7, 0x44}
	pkSeed := []byte{0x54, 0x94, 0x44, 0x6e, 0xdc, 0xe5, 0x70, 0x4, 0xf2, 0x49, 0x5a, 0xbc, 0x93, 0xf0, 0x4, 0x70}
	adrs := &address{0x0, 0x0, 0x0, 0x2, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x2, 0x56, 0x7a, 0x35, 0xf1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0xb4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0}
	want := []byte{0xa6, 0xd3, 0x1e, 0xc6, 0x54, 0x58, 0x48, 0xe1, 0xab, 0x75, 0xa1, 0x8a, 0x11, 0x84, 0x85, 0xb6}
	if got := par.wotsPkGen(skSeed, pkSeed, adrs); !slices.Equal(got, want) {
		t.Errorf("par.wotsPkGen() = %v, want %v", got, want)
	}
}

func TestWotsSignKat(t *testing.T) {
	par := SLH_DSA_SHA2_128s
	msg := []byte{0x3, 0xea, 0x96, 0x83, 0x50, 0x24, 0x18, 0x83, 0xc4, 0xd1, 0x5f, 0xbb, 0x2e, 0x96, 0xfc, 0x7e}
	skSeed := []byte{0x1d, 0xff, 0x3f, 0xe7, 0x80, 0x72, 0x96, 0x70, 0x56, 0x57, 0xd2, 0xb6, 0x8a, 0xd7, 0x7d, 0xd5}
	pkSeed := []byte{0xe8, 0x8e, 0xc9, 0x8e, 0x19, 0x9a, 0x85, 0xf, 0xf8, 0xbf, 0x4e, 0x7e, 0x6b, 0x20, 0xe9, 0xa7}
	adrs := &address{0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xe, 0xdf, 0x7e, 0x3d, 0xd7, 0x8d, 0x7e, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xef, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0}
	want := []byte{0x5d, 0x9e, 0xaf, 0x9c, 0xdc, 0x14, 0x59, 0xf5, 0x9e, 0xcd, 0x22, 0x2e, 0x82, 0x1e, 0xf0, 0x6e, 0x42, 0x48, 0xa1, 0x97, 0xed, 0x58, 0x64, 0xe8, 0xb8, 0x41, 0x6f, 0xda, 0x6, 0xea, 0xe1, 0x26, 0x7d, 0xf8, 0xd0, 0x35, 0x40, 0xc, 0x92, 0x7, 0x73, 0x28, 0xe9, 0x6, 0x6, 0x3d, 0x3a, 0xa, 0xaa, 0xdd, 0x2e, 0x56, 0x80, 0xb0, 0x4d, 0x58, 0xa1, 0x98, 0x0, 0x6c, 0x38, 0x64, 0xb0, 0x95, 0xe8, 0xa4, 0x93, 0x3, 0xf, 0xa3, 0xa, 0x62, 0x89, 0xb1, 0x17, 0x42, 0x8d, 0xa3, 0x0, 0x75, 0x77, 0x52, 0x5f, 0x13, 0x71, 0xf3, 0x74, 0x0, 0xb3, 0x17, 0x73, 0x37, 0x17, 0xb0, 0x85, 0x92, 0x7b, 0x4b, 0x93, 0x64, 0x6a, 0x16, 0x6c, 0xb0, 0x5, 0xe9, 0xb9, 0x9, 0x50, 0x44, 0xc6, 0x4, 0xb2, 0x54, 0xa6, 0x59, 0x1d, 0xba, 0x42, 0x73, 0xb6, 0x48, 0xaf, 0xf0, 0x14, 0x68, 0x7a, 0x0, 0x43, 0xd7, 0xe5, 0xbe, 0x92, 0xb2, 0xc1, 0x1a, 0xeb, 0x87, 0xd9, 0x33, 0xb, 0x7b, 0x12, 0x60, 0xf6, 0xe7, 0x65, 0x8d, 0xf0, 0x89, 0x4, 0x1a, 0xf0, 0x74, 0x15, 0xe, 0x41, 0x1b, 0x8, 0x99, 0x4e, 0x1d, 0x8b, 0xd1, 0x43, 0x30, 0xa8, 0xe9, 0x7f, 0x1a, 0xd0, 0xd7, 0xf5, 0x80, 0x12, 0xb8, 0x65, 0x33, 0x7e, 0xf4, 0x5e, 0x11, 0x71, 0x84, 0x6b, 0x92, 0x1, 0xeb, 0x4d, 0x9f, 0x9e, 0x7a, 0xdf, 0x3b, 0x4c, 0xba, 0x5b, 0xb0, 0x44, 0xae, 0x7a, 0x24, 0x3, 0x2c, 0xac, 0x51, 0xfe, 0x1c, 0x74, 0x0, 0x7a, 0xc7, 0xc0, 0xdb, 0x22, 0x72, 0xa5, 0x8a, 0x63, 0x80, 0x9e, 0xe2, 0xe6, 0x23, 0xc8, 0xa7, 0xb9, 0x9b, 0x7a, 0x21, 0xc7, 0xd, 0x98, 0x66, 0x67, 0x88, 0xac, 0x5, 0x37, 0x4d, 0x2e, 0x3b, 0x2c, 0x12, 0xc9, 0x9c, 0x0, 0x24, 0x1f, 0x6f, 0xa2, 0xcd, 0xf3, 0x37, 0x73, 0x7d, 0x5d, 0x67, 0x74, 0x99, 0xe6, 0x18, 0x23, 0xa8, 0x1c, 0x86, 0x53, 0x71, 0x34, 0x8b, 0x88, 0x6b, 0xd3, 0x8a, 0xdb, 0xc1, 0x7a, 0xa8, 0x41, 0x6a, 0xcc, 0x29, 0x1d, 0xa1, 0xda, 0x3d, 0xd1, 0x33, 0x2f, 0x4e, 0x59, 0x51, 0x8d, 0xc, 0xc0, 0x2e, 0xe2, 0x8f, 0x6, 0x96, 0x6c, 0x9b, 0x57, 0x5f, 0xcb, 0xb7, 0xd4, 0x8c, 0xbd, 0x16, 0xc5, 0x2a, 0xf3, 0x89, 0x72, 0xa2, 0xd5, 0xde, 0x3e, 0xe6, 0xee, 0xfa, 0xca, 0x5, 0xe7, 0xfe, 0xe9, 0x36, 0xa4, 0x2c, 0xe9, 0xff, 0xeb, 0x69, 0x63, 0x6a, 0xe5, 0xa1, 0x68, 0xb4, 0x2, 0x6, 0x73, 0xf, 0x99, 0x7a, 0xb3, 0x56, 0x2d, 0x64, 0xf4, 0x6b, 0xd7, 0x3b, 0x95, 0x1f, 0x7c, 0x1e, 0x81, 0x8c, 0x3e, 0xec, 0xea, 0xe1, 0xa4, 0x31, 0x76, 0xe8, 0xa0, 0x9d, 0xaf, 0x55, 0x88, 0xc5, 0x4a, 0x53, 0x8b, 0xbd, 0xf7, 0xc5, 0x55, 0xf1, 0x47, 0x1b, 0x5, 0x86, 0x7a, 0x51, 0xc7, 0x5b, 0x44, 0x34, 0x83, 0x1f, 0x58, 0x94, 0x4e, 0x4b, 0xce, 0x7, 0x2b, 0xa, 0xd0, 0x63, 0x4e, 0xc9, 0xe9, 0x76, 0x14, 0xb1, 0xb6, 0x50, 0x40, 0x43, 0x31, 0xa9, 0x50, 0xb7, 0x71, 0x26, 0x52, 0xef, 0x2a, 0xa1, 0x5a, 0xaa, 0x29, 0x90, 0xbc, 0x23, 0x92, 0xa8, 0x82, 0x45, 0xe7, 0xd0, 0x7e, 0x43, 0x81, 0x74, 0xa, 0xf8, 0xbd, 0x1c, 0x5, 0x84, 0xca, 0x8b, 0xfe, 0x92, 0xd2, 0xf5, 0x3d, 0x79, 0x4, 0x6f, 0x70, 0x45, 0x9c, 0x38, 0x96, 0x3d, 0x63, 0xe8, 0xd7, 0xbb, 0x4a, 0x9a, 0xfc, 0xd6, 0x9, 0x7d, 0xa4, 0xb, 0x62, 0xb4, 0x3b, 0x76, 0x2, 0xc, 0x18, 0x45, 0x2, 0x62, 0xb6, 0x4b, 0x5f, 0x31, 0xa6, 0xcf, 0x38, 0x89, 0x1f, 0xe4, 0xfa, 0xab, 0x5a, 0x4a, 0x51, 0xe0, 0x5c, 0x92, 0xbd, 0x33, 0xf8, 0x49, 0x5b, 0x1, 0x4f, 0xaa, 0xa0, 0x4c, 0x51, 0x6f, 0xde, 0xc8, 0xbf, 0xf8, 0x9d, 0xd5, 0x82, 0xe3, 0x24, 0x94, 0x90, 0xf2, 0xa1, 0xb6, 0xbe, 0x6d, 0x20, 0x70, 0xc1, 0x93, 0x9d, 0x44, 0x62, 0x79, 0xb, 0x70, 0x6f, 0xff, 0x96, 0x78, 0x21, 0x2c, 0xed, 0xf7, 0x31, 0xbc, 0xa5, 0x5, 0x2e, 0x13, 0xdf, 0x31, 0xa2, 0x1e, 0x5a, 0xaf}
	if got := par.wotsSign(msg, skSeed, pkSeed, adrs); !slices.Equal(got, want) {
		t.Errorf("par.wotsSign() = %v, want %v", got, want)
	}
}

func TestWotsPkFromSigKat(t *testing.T) {
	par := SLH_DSA_SHA2_128s
	sig := []byte{0x93, 0x3e, 0x6f, 0x62, 0xc4, 0x4b, 0x7f, 0x4a, 0x4d, 0x63, 0x9f, 0xc6, 0x1e, 0x93, 0x5e, 0x4a, 0xfa, 0xf6, 0x2e, 0x9a, 0xc, 0xa0, 0xcb, 0x38, 0xac, 0xec, 0xd5, 0x87, 0xcf, 0x7c, 0x5e, 0xc, 0xfa, 0xc2, 0x89, 0xe3, 0xab, 0x9e, 0xba, 0xb1, 0x8c, 0x2a, 0xc6, 0x88, 0xc1, 0xac, 0x76, 0x78, 0xd1, 0xb9, 0x57, 0xa, 0x7b, 0x95, 0x6e, 0x92, 0x11, 0x3f, 0x54, 0xce, 0x49, 0xd8, 0x82, 0x68, 0x8d, 0xf7, 0x95, 0xb, 0xa6, 0xfc, 0x4a, 0x94, 0x5e, 0xb5, 0x22, 0x7b, 0xcd, 0x8e, 0xe1, 0x2c, 0x11, 0x7e, 0xbd, 0xf, 0x73, 0xa3, 0x14, 0xe1, 0xf9, 0x87, 0x91, 0x3a, 0x5e, 0xd7, 0x53, 0xf8, 0x47, 0x6c, 0xe5, 0xaf, 0x36, 0x76, 0x7d, 0x43, 0xe, 0x76, 0x3a, 0x5c, 0x33, 0x1f, 0x3d, 0x27, 0x80, 0x70, 0x6e, 0x83, 0x2f, 0x3b, 0x4f, 0xe, 0xbd, 0xc6, 0x55, 0x67, 0x2e, 0x5a, 0x80, 0x9b, 0xd9, 0xcf, 0x18, 0x62, 0xac, 0x25, 0x35, 0x21, 0x98, 0x39, 0x91, 0x92, 0x8f, 0x11, 0x4b, 0x30, 0x2f, 0x79, 0xae, 0xa0, 0x70, 0xdd, 0xaa, 0xb2, 0x24, 0x31, 0x1, 0xda, 0xff, 0xbe, 0xe5, 0x88, 0xd2, 0x6b, 0x7b, 0x1c, 0x8f, 0xdb, 0x77, 0x8c, 0x51, 0x2b, 0x2d, 0xfe, 0x5b, 0x13, 0x2a, 0xd9, 0xfc, 0xd3, 0xe2, 0x41, 0xaa, 0x48, 0xaa, 0x7b, 0xd7, 0xc3, 0x2b, 0x7f, 0x20, 0x3d, 0xf8, 0x27, 0x15, 0x93, 0x7f, 0xa5, 0xa4, 0x7, 0x2e, 0xd, 0x4e, 0x70, 0x9a, 0xaf, 0x9e, 0x47, 0x37, 0x93, 0x9a, 0x7f, 0x50, 0x68, 0xfb, 0x47, 0xe9, 0x2f, 0xd6, 0x28, 0xfa, 0xd1, 0x9, 0xac, 0x58, 0x30, 0x3b, 0x19, 0x3e, 0xfc, 0x8f, 0x83, 0x67, 0x31, 0x83, 0xd4, 0x29, 0x5e, 0x6f, 0x59, 0xc7, 0xc, 0x3c, 0xfc, 0xf4, 0xf3, 0x4b, 0xc6, 0x8f, 0xad, 0xf9, 0x55, 0x24, 0x94, 0xeb, 0x4f, 0xf7, 0xd9, 0xc0, 0xfd, 0x26, 0xf4, 0x47, 0x98, 0xb3, 0xf, 0x3e, 0xfa, 0x38, 0x80, 0xb4, 0x59, 0x13, 0x2c, 0xd0, 0x45, 0xdc, 0xbd, 0xfc, 0x86, 0x72, 0xa5, 0xfa, 0xb2, 0xb3, 0xe, 0x6a, 0x9, 0xb2, 0xa8, 0x6a, 0xbf, 0x12, 0x95, 0x53, 0x52, 0xf7, 0x23, 0x7f, 0xdb, 0x6c, 0x5e, 0xcf, 0x29, 0x90, 0xfb, 0xcb, 0x5c, 0x31, 0xe2, 0x3f, 0x90, 0xa4, 0x3, 0xdb, 0xde, 0xa5, 0x36, 0xf4, 0xda, 0xc0, 0xfa, 0x9, 0x3b, 0xa6, 0x91, 0x7f, 0x1c, 0x88, 0x2e, 0x93, 0x7f, 0xdf, 0xe2, 0x8e, 0x49, 0xe3, 0xdd, 0x39, 0x7b, 0x30, 0xb7, 0xb4, 0x8, 0xf0, 0x85, 0x55, 0x53, 0xc1, 0x50, 0x3f, 0x3b, 0x3a, 0xa8, 0xd1, 0x20, 0xd, 0xc4, 0x28, 0x29, 0xb6, 0xec, 0x44, 0x46, 0x75, 0x39, 0x2d, 0xd1, 0xf, 0xa3, 0xf2, 0x85, 0xf3, 0x70, 0x45, 0x48, 0x55, 0xae, 0xc8, 0x37, 0xdd, 0x0, 0x3c, 0xa6, 0x86, 0xa6, 0xe5, 0xd7, 0x3e, 0x6f, 0x23, 0xe, 0xba, 0x45, 0x9a, 0x87, 0x40, 0x40, 0x1, 0x12, 0xc2, 0xd4, 0x2a, 0x99, 0xe6, 0x63, 0x60, 0xec, 0xfb, 0xef, 0xf4, 0x34, 0x4, 0xe2, 0x16, 0x21, 0xb6, 0x9b, 0xb8, 0xb9, 0x63, 0x4a, 0x6b, 0xb1, 0x75, 0xf4, 0x84, 0x39, 0x5, 0x66, 0x65, 0x34, 0xa6, 0xb, 0x2e, 0xcd, 0x8e, 0x9e, 0xc5, 0x72, 0xeb, 0xfe, 0xab, 0xdf, 0x80, 0x86, 0x98, 0xfa, 0x9f, 0xf9, 0x5, 0x8b, 0x85, 0x36, 0xd8, 0x13, 0xb1, 0xfc, 0x72, 0x53, 0xbb, 0xe4, 0x76, 0x38, 0xf8, 0x56, 0xdb, 0x38, 0x2c, 0xed, 0x4, 0x8f, 0xb2, 0xf0, 0xec, 0xf0, 0xc3, 0xa0, 0x64, 0xac, 0x58, 0x11, 0xa, 0x1a, 0xe2, 0x40, 0x2f, 0x1, 0x6e, 0xbf, 0x4a, 0xa1, 0xde, 0x64, 0x28, 0x53, 0x76, 0xff, 0xa9, 0x74, 0xc1, 0xc3, 0xc6, 0x9e, 0xc0, 0x51, 0x3, 0xe8, 0xb9, 0x3c, 0xa3, 0x9d, 0xfb, 0xa2, 0xde, 0x5d, 0x2, 0x22, 0xd7, 0xfe, 0x7a, 0x41, 0x91, 0x13, 0x64, 0xdb, 0x1a, 0xdb, 0x49, 0x19, 0x8c, 0x87, 0xa0, 0xe, 0x53, 0x52, 0x9e, 0x72, 0x7, 0x31, 0xf4, 0x34, 0xc9, 0x87, 0x1c, 0x63, 0xba, 0x52, 0x6e, 0xf7, 0x72, 0xcf, 0xae, 0x88, 0x85, 0xc2, 0x6f, 0x6e, 0x66, 0x8e, 0x3, 0xf2}
	msg := []byte{0xcf, 0xb7, 0x4b, 0x7d, 0x8a, 0xa4, 0x90, 0xed, 0x1e, 0xcc, 0xf0, 0xd8, 0x8d, 0xc5, 0x2c, 0xda}
	pkSeed := []byte{0xce, 0x6a, 0x3e, 0xcf, 0x8b, 0x69, 0x93, 0x61, 0xb6, 0xea, 0x8d, 0x7c, 0xee, 0x96, 0x29, 0xa7}
	adrs := &address{0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x2c, 0x32, 0x46, 0xe6, 0x5b, 0x90, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x40, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0}
	want := []byte{0x6, 0xf9, 0x49, 0xdf, 0x70, 0x78, 0xfb, 0x5b, 0x6c, 0xa3, 0x8c, 0x2a, 0xd1, 0xd6, 0xb7, 0x67}
	if got := par.wotsPkFromSig(sig, msg, pkSeed, adrs); !slices.Equal(got, want) {
		t.Errorf("par.wotsPkFromSig() = %v, want %v", got, want)
	}
}
